### Common behaviour

A PCA Subscriber Agent manages the service offerings for one subscriber organisation (also referred to as a 'business partner' in the PCA portal).      

The PCA service instantiates one PCA Subscriber Agent for each participating subscriber organisation. That is, each subscriber organisation interacts with a PCA Subscriber Agent located at a *subscriber agent base url* that is unique to that organisation.    

The *subscriber agent base URL* for a particular organisation is available via the PCA portal on the 'Organisation Details' screen of the subscriber organisation.    

A service offering is made available to a subscriber organisation when a publisher authorises publication of a resource to one or more of the subscriber organisation's partner services.    
    
 **Publication process**    
    
 A PCA Publisher Agent creates the ```Location```, ```HealthcareService``` and ```PractitionerRole``` resources (that are associated with existing ```Organization``` and ```Practitioner``` resources) that represent their service offerings.  These resources are made available to a subscriber organisation when a publisher authorises publication of a ```HealthcareService``` or ```PractitionerRole``` resource to one or more of the subscriber organisation's partner services.    
    
 The authority to publish is represented using ```Contract``` resources which are also implemented by the PCA Publisher Agent (see the [Contract resource endpoint](CapabilityStatement-pca-publisher-agent.html#contract)).    
    
 **Identifier disclosure rules**    
    
 The PCA service enforces certain business rules regarding the disclosure of *Identifiers* (including subscriber-issued identifiers) in the resources that it makes available to subscribers.     
    
 The result of these rules may be that a copy of  a resource received by one subscriber may contain *Identifiers* that are not included in the copy of the same resource that is received by a different subscriber.    
    
 In the case of HPI-Is these *identifier disclosure rules* ensure that relevant legal requirements are complied with.    
    
 Other *identifier disclosure rules* are defined by the PCA operator.     
    
 The *identifier disclosure rules* are applied automatically at the time that the resource is retrieved by the target PCA Subscriber Agent.    
    
 **Authorisation rules**    
    
Authorised client systems are only permitted to interact with a PCA Subscriber Agent API where the participating organisation associated with the PCA Subscriber Agent has a subscriber approval status of 'Approved'.    
    
 **Resource profiles and extensions**    
    
This API specification defines the required resource profiles for each operation.  The PCA Subscriber Agent will reject any request to create or update a resource that contains any extensions (including modifier extensions) not explicitly defined in that resource profile.    
    

<hr/>

### Whole system interactions



<hr/>

#### search-system

<b>Supported search parameters:</b>

<table border="1">
   <tr>
      <td>
         <b>Parameter</b>
      </td>
      <td>
         <b>Type</b>
      </td>
      <td>
         <b>Definition</b>
      </td>
   </tr>
   <tr>
      <td>_lastUpdated</td>
      <td>date</td>
      <td>http://hl7.org/fhir/SearchParameter/Resource-lastUpdated</td>
   </tr>
   <tr>
      <td>_type</td>
      <td>token</td>
      <td/>
   </tr>
   <tr>
      <td>_include</td>
      <td>special</td>
      <td/>
   </tr>
   <tr>
      <td>_include:iterate</td>
      <td>special</td>
      <td/>
   </tr>
   <tr>
      <td>_filter</td>
      <td>special</td>
      <td/>
   </tr>
   <tr>
      <td>_page</td>
      <td>special</td>
      <td/>
   </tr>
   <tr>
      <td>_count</td>
      <td>special</td>
      <td/>
   </tr>
   <tr>
      <td>_resultId</td>
      <td>special</td>
      <td/>
   </tr>
</table>**Definition**:     
 This operation is a RESTful *search* interaction (whole system interaction), across multiple resource types (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#search)).      
     
 **Behaviour**:     
 This operation is used by a client system operated by a subscriber organisation to get resources that match the specified search parameters and that have been published to the subscriber organisation's partner services.     
     
 The following  search parameters are supported:     
     
 * *_lastUpdated* is used to get the resources that have been added or changed since the requesting system last synchronised their local data store with PCA Subscriber Agent.  This search parameter supports the ```gt``` and ```lt``` comparison prefixes.     
 * *_type* is used to filter the resources returned by their resource type. The available types are:      
     
   * ```HealthcareService``` (returns resources conforming to the [PCA Healthcare Service](StructureDefinition-pca-healthcare-service.html) profile)     
   * ```PractitionerRole``` (returns resources conforming to the [PCA Practitioner Role](StructureDefinition-pca-practitioner-role.html) profile)     
   * ```Organization``` (returns resources conforming to the [PCA Organization](StructureDefinition-pca-organization.html) profile)     
   * ```Location``` (returns resources conforming to the [PCA Location](StructureDefinition-pca-location.html) profile)     
   * ```Practitioner``` (returns resources conforming to the [PCA Practitioner](StructureDefinition-pca-practitioner.html) profile)     
   * ```Provenance``` (returns resources conforming to the [PCA Provenance](StructureDefinition-pca-provenance.html) profile)     
   * ```Contract``` (returns resources conforming to the [PCA Publication Authorisation Record](StructureDefinition-pca-publication-authorisation-record.html) profile)      
   * ```Task``` (returns resources conforming to the [PCA Match Record](StructureDefinition-pca-match-record.html) profile)      
     
     * Note: a ```Contract``` resource with 'status' of 'revoked' indiates that a service offering that was previously published to a partner service is no longer available to that partner service. In this case the ```Contract``` resource will reference a *target* ```Organization```, ```Location```, ```HealthcareService```, or ```PractitionerRole``` that is not included in the ```Bundle``` (but its *id* will identify the resource which is no longer published to the subscriber).      
            
 * *_include* (and *_include:iterate*) are used in order that all the resources that match the *_lastUpdated* and *_type* filters will be accompanied, in the result ```Bundle```, with all the other resources which they reference (directly or indirectly).  Note that the use of *_include* (and *_include:iterate*) means that a resource may be present in the result ```Bundle``` either because it matched the other query parameters or because it was referenced (possibly indirectly) by another resource that matched the other query parameters.     
 * *_filter* is used to only get the resources that have been published to a specific [PCA Partner Service](StructureDefinition-pca-partner-service.html) of the subscriber organisation.  If the *_filter* parameter is not present, the operation will return resources that have been published to all of the subscriber organisation's partner services.  The *_filter* parameter supports the following query expression:     
     
   * ```HealthcareService: id eq [FHIR ID of partner service]```     
      
 All of the following query parameters are required in order to receive the complete set of resources that have been created, changed or published to all the subscriber organisation's partner services since *[last_synchronized]*:     
     
 ``` _lastUpdated=[last_synchronized] ```     
 ```_type=HealthcareService,PractitionerRole,Practitioner,Organization,Location,Provenance,Contract,Task ```     
 ```_include=Location:organization ```     
 ```_include=HealthcareService:organization ```     
 ```_include=HealthcareService:location ```     
 ```_include=PractitionerRole:organization ```     
 ```_include=PractitionerRole:location ```       
 ```_include=PractitionerRole:service ```     
 ```_include=PractitionerRole:practitioner ```      
 ```_include=Provenance:target ```        
 ```_include=Contract:subject ```        
 ```_include:iterate=HealthcareService:organization  ```       
 ```_include:iterate=HealthcareService:location ```     
 ```_include:iterate=PractitionerRole:organization ```     
 ```_include:iterate=PractitionerRole:location ```     
 ```_include:iterate=PractitionerRole:service ```     
 ```_include:iterate=PractitionerRole:practitioner ```     
      
 As described [above](CapabilityStatement-pca-subscriber-agent.html#common-behaviour) the resources returned in the Bundle exclude any *Identifiers* (e.g. HPI-Is) that cannot be disclosed to the subscriber.     
      
 This operation supports 'paging' ([see FHIR specification](https://www.hl7.org/fhir/search.html#Paging)).  If there are too many resources to return in one ```Bundle```, the FHIR server adds a *next* link to the response ```Bundle```. The contents of the result ```Bundle``` therefore comprise all instances of the requested resource types whose *lastUpdated* timestamp is greater than the &lt;lastSync&gt; in the request query expression and less than or equal to &lt;lastInBundle&gt; where &lt;lastInBundle&gt; is the greatest *lastUpdated* value of all resources included in the result ```Bundle```. The Subscriber Agent FHIR server sets the *next* link to the filter expression that was used in the original request except that &lt;lastSync&gt; is replaced with &lt;lastInBundle&gt;. If the *next* link is not present in the Bundle, then there are no more service offerings available.     
     
 **HTTP interaction signature**:     
 * ```GET [subscriber agent base url]/?[parameters]```     
     
 **Mandatory Request Headers**:     
 * ```Accept: application/fhir+json;charset=utf-8``` or ```Accept: application/fhir+xml;charset=utf-8```     
 * ```Authorization: Bearer [token]```     
     
 **Request Body**:     
 * (No request body)     
     
 **Mandatory Response Headers**:     
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```     
     
 **Response Body**:     
 * Success: ```Bundle``` that contains zero to many resources (where the resource type was included in the filter expression of the search request) that conform to the following [PCA profiles](profiles.html):      
     
   * ```HealthcareService``` resources conforming to the [PCA Healthcare Service](StructureDefinition-pca-healthcare-service.html) profile     
   * ```PractitionerRole``` resources conforming to the [PCA Practitioner Role](StructureDefinition-pca-practitioner-role.html) profile     
   * ```Organization``` resources conforming to the [PCA Organization](StructureDefinition-pca-organization.html) profile     
   * ```Location``` resources conforming to the [PCA Location](StructureDefinition-pca-location.html) profile     
   * ```Practitioner``` resources conforming to the [PCA Practitioner](StructureDefinition-pca-practitioner.html) profile     
   * ```Provenance``` resources conforming to the [PCA Provenance](StructureDefinition-pca-provenance.html) profile     
   * ```Contract``` resources conforming to the [PCA Publication Authorisation Record](StructureDefinition-pca-publication-authorisation-record.html) profile     
     
     * ```Contract``` resources will be returned that reference each of the ```HealthcareService``` or ```PractitionerRole``` resources that are included in the ```Bundle``` - these ```Contract``` resources have a *status* of “executed” if the associated PCA Partner Service is active, or a *status* of “cancelled” if the associated PCA Partner Service is inactive).     
     * ```Contract``` resources will also be returned that reference ```HealthcareService``` or ```PractitionerRole``` resources that are not in the ```Bundle``` but for which authority to publish has been revoked - these ```Contract``` resources will have a *status* of “revoked”).     
     
   * ```Task``` resources conforming to the [PCA Match Record](StructureDefinition-pca-match-record.html) profile      
     
     
 * Error: ```OperationOutcome``` resource (see [FHIR specification](https://www.hl7.org/fhir/operationoutcome.html)).      
     
 **Required OAuth 2.0 scopes**:     
     
 The *scope* string **SHALL** contain the following *scoped role*:     
 * ``` Organization/[id]:SS_Receiver```     
     
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.     
     
 **Response Codes**:         
 * Code 200:  Search successfully performed (but may return zero matching results).     
 * Code 400:  The input query could not be parsed, or the [PCA Partner Service](StructureDefinition-pca-partner-service.html) referenced in the ```_filter``` parameter could not be found or is inactive.      
 * Code 403:  Server refused to perform the search.     

<hr/>

### Organization

     
     
 This section defines the supported interactions for the [PCA Organization](StructureDefinition-pca-organization.html) profile of the ```Organization``` resource type.     

<hr/>

#### Server implementation options

<table border="1">
   <tr>
      <td>
         <b>Option</b>
      </td>
      <td>
         <b>Value</b>
      </td>
   </tr>
   <tr>
      <td>versioning</td>
      <td>versioned</td>
   </tr>
   <tr>
      <td>readHistory</td>
      <td>false</td>
   </tr>
   <tr>
      <td>updateCreate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalCreate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalRead</td>
      <td>not-supported</td>
   </tr>
   <tr>
      <td>conditionalUpdate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalDelete</td>
      <td>not-supported</td>
   </tr>
   <tr>
      <td>referencePolicy</td>
      <td>not specified</td>
   </tr>
   <tr>
      <td>searchInclude</td>
      <td>_include not supported</td>
   </tr>
   <tr>
      <td>searchRevInclude</td>
      <td>_revinclude not supported</td>
   </tr>
</table>

<hr/>

#### RESTful operations



<hr/>

#### read

**Definition**:      
      
 This operation is a RESTful **read** interaction (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#read)) of an ```Organization``` resource.       
      
 **Behaviour**:      
      
 This operation retrieves an identified ```Organization``` resource that conforms to the [PCA Organization](StructureDefinition-pca-organization.html) profile. As described in the [common behaviour](CapabilityStatement-pca-subscriber-agent.html#common-behaviour) section, the ```Organization``` resource excludes any *Identifiers* that cannot be disclosed to the subscriber.      
      
 **HTTP interaction signature**:      
 * ```GET [subscriber agent base url]/Organization/[id]```      
      
 **Mandatory Request Headers**:      
 * ```Accept: application/fhir+json;charset=utf-8``` or ```Accept: application/fhir+xml;charset=utf-8```      
 * ```Authorization: Bearer [token]```      
      
 **Request Body**:      
 * (No request body)      
      
 **Mandatory Response Headers**:      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
 * ```ETag: [version id]``` where [version id] identifies the version of the ```Organization``` resource returned.        
      
 **Response Body**:      
 * Success: ```Organization``` resource that conforms to the [PCA Organization](StructureDefinition-pca-organization.html) profile.      
 * Error: ```OperationOutcome``` resource (see [FHIR specification](https://www.hl7.org/fhir/operationoutcome.html)).      
      
 **Required OAuth 2.0 scopes**:      
      
 The *scope* string **SHALL** contain the following *scoped role*:      
 * ```Organization/[id]:SS_Receiver```      
      
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.      
      
 **Response Codes**:          
 *  Code 200:  The resource was found and returned.      
 *  Code 404:  The resource was not found.      

<hr/>

#### patch

**Definition**:      
      
 This operation is a RESTful **patch** interaction (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#patch)) of an ```Organization``` resource.       
      
 **Behaviour**:      
      
 This operation is used to update an identified ```Organization``` resource that conforms to the [PCA Organization](StructureDefinition-pca-organization.html) profile via a JSON or XML patch document. This operation will only process a patch operation for adding or updating an *identifier* element.      
      
 If any of the following data validation rules are violated, then the request is rejected and the error code 422 is returned:      
      
 * The patch document **SHALL** only contain an operation for adding or updating an *identifier* element.      
 * Only one *identifier* element per *identifier.system* **SHALL** be allowed for the identified ```Organization``` resource.      
 * The patch operation **SHALL** only add or update an *identifier* element where *identifier.system* identifies a subscriber identifier system that is owned by the participating organisation associated with the target PCA Subscriber Agent.      
 * The patch operation **SHALL** only add or update an *identifier* element where a [PCA Match Record](StructureDefinition-pca-match-record.html) exists within the PCA service with:      
   * A *status* of 'completed', and;       
   * A *focus* referencing the ```Organization``` resource, and;      
   * An *owner* referencing a [PCA Partner Service](StructureDefinition-pca-partner-service.html) of the participating organisation associated with the target PCA Subscriber Agent.      
      
 **HTTP interaction signature**:      
 * ```PATCH [subscriber agent base url]/Organization/[id]```      
      
 **Mandatory Request Headers**:      
 * ```Accept: application/json-patch+json;charset=utf-8``` or ```Accept: application/xml-patch+xml;charset=utf-8```      
 * ```Content-Type: application/json-patch+json;charset=utf-8``` or ```Accept: application/xml-patch+xml;charset=utf-8```      
 * ```Authorization: Bearer [token]```      
      
 **Request Body**:      
 * A JSON Patch document that conforms to [https://datatracker.ietf.org/doc/html/rfc6902](https://datatracker.ietf.org/doc/html/rfc6902) with a content type of ```application/json-patch+json```, or;      
 * An XML Patch document that conforms to [https://datatracker.ietf.org/doc/html/rfc5261](https://datatracker.ietf.org/doc/html/rfc5261) with a content type of ```application/xml-patch+xml```      
      
 **Mandatory Response Headers**:      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
 * ```ETag: [version id]``` where [version id] identifies the version of the ```Organization``` resource returned.        
      
 **Response Body**:      
 * Success: ```Organization``` resource that conforms to the [PCA Organization](StructureDefinition-pca-organization.html) profile.      
 * Error: ```OperationOutcome``` resource (see [FHIR specification](https://www.hl7.org/fhir/operationoutcome.html)).      
      
 **Required OAuth 2.0 scopes**:      
      
 The *scope* string **SHALL** contain the following *scoped role*:      
 * ``` Organization/[id]:SS_Updater```      
      
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.      
      
 **Response Codes**:          
 *  Code 200:  The resource was found and returned.      
 *  Code 404:  The resource was not found.      
 *  Code 422:  The input patch document violated applicable FHIR profiles or server data validation rules.      

<hr/>

### Location

       
     
 This section defines the supported interactions for the [PCA Location](StructureDefinition-pca-location.html) profile of the ```Location``` resource type.     
     
 A ```Location``` resource conforming to the [PCA Location](StructureDefinition-pca-location.html) profile represents a location at which healthcare services are offered by a PCA publisher.       

<hr/>

#### Server implementation options

<table border="1">
   <tr>
      <td>
         <b>Option</b>
      </td>
      <td>
         <b>Value</b>
      </td>
   </tr>
   <tr>
      <td>versioning</td>
      <td>versioned</td>
   </tr>
   <tr>
      <td>readHistory</td>
      <td>false</td>
   </tr>
   <tr>
      <td>updateCreate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalCreate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalRead</td>
      <td>not-supported</td>
   </tr>
   <tr>
      <td>conditionalUpdate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalDelete</td>
      <td>not-supported</td>
   </tr>
   <tr>
      <td>referencePolicy</td>
      <td>not specified</td>
   </tr>
   <tr>
      <td>searchInclude</td>
      <td>_include not supported</td>
   </tr>
   <tr>
      <td>searchRevInclude</td>
      <td>_revinclude not supported</td>
   </tr>
</table>

<hr/>

#### RESTful operations



<hr/>

#### read

**Definition**:      
      
 This operation is a RESTful **read** interaction (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#read)) of a ```Location``` resource.       
      
 **Behaviour**:      
      
 This operation retrieves an identified ```Location``` resource that conforms to the [PCA Location](StructureDefinition-pca-location.html) profile. As described in the [common behaviour](CapabilityStatement-pca-subscriber-agent.html#common-behaviour) section, the ```Location``` resource excludes any *Identifiers* that cannot be disclosed to the subscriber.      
      
 **HTTP interaction signature**:      
 * ```GET [subscriber agent base url]/Location/[id]```      
      
 **Mandatory Request Headers**:      
 * ```Accept: application/fhir+json;charset=utf-8``` or ```application/fhir+xml;charset=utf-8```      
 * ```Authorization: Bearer [token]```      
      
 **Request Body**:      
 * (No request body)      
      
 **Mandatory Response Headers**:      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
 * ```ETag: [version id]``` where [version id] identifies the version of the ```Location``` resource returned.        
      
 **Response Body**:      
 * Success: ```Location``` resource that conforms to the [PCA Location](StructureDefinition-pca-location.html) profile.      
 * Error: ```OperationOutcome``` resource (see [FHIR specification](https://www.hl7.org/fhir/operationoutcome.html)).      
      
 **Required OAuth 2.0 scopes**:      
      
 The *scope* string **SHALL** contain the following *scoped role*:      
 * ``` Organization/[id]:SS_Receiver```      
      
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.      
      
 **Response Codes**:          
 *  Code 200:  The resource was found and returned.      
 *  Code 404:  The resource was not found.      

<hr/>

#### patch

**Definition**:      
      
 This operation is a RESTful **patch** interaction (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#patch)) of a ```Location``` resource.       
      
 **Behaviour**:      
      
 This operation is used to update an identified ```Location``` resource that conforms to the [PCA Location](StructureDefinition-pca-location.html) profile via a JSON or XML patch document. This operation will only process a patch operation for adding or updating an *identifier* element.      
      
 If any of the following data validation rules are violated, then the request is rejected and the error code 422 is returned:      
      
 * The patch document **SHALL** only contain an operation for adding or updating an *identifier* element.      
 * Only one *identifier* element per *identifier.system* **SHALL** be allowed for the identified ```Location``` resource.      
 * The patch operation **SHALL** only add or update an *identifier* element where *identifier.system* identifies a subscriber identifier system that is owned by the participating organisation associated with the target PCA Subscriber Agent.      
 * The patch operation **SHALL** only add or update an *identifier* element where a [PCA Match Record](StructureDefinition-pca-match-record.html) exists within the PCA service with:      
   * A *status* of 'completed', and;       
   * A *focus* referencing the ```Location``` resource, and;      
   * An *owner* referencing a [PCA Partner Service](StructureDefinition-pca-partner-service.html) of the participating organisation associated with the target PCA Subscriber Agent.      
      
 **HTTP interaction signature**:      
 * ```PATCH [subscriber agent base url]/Location/[id]```      
      
 **Mandatory Request Headers**:      
 * ```Accept: application/json-patch+json;charset=utf-8``` or ```Accept: application/xml-patch+xml;charset=utf-8```      
 * ```Content-Type: application/json-patch+json;charset=utf-8``` or ```Accept: application/xml-patch+xml;charset=utf-8```      
 * ```Authorization: Bearer [token]```      
      
 **Request Body**:      
 * A JSON Patch document that conforms to [https://datatracker.ietf.org/doc/html/rfc6902](https://datatracker.ietf.org/doc/html/rfc6902) with a content type of ```application/json-patch+json```, or;      
 * An XML Patch document that conforms to [https://datatracker.ietf.org/doc/html/rfc5261](https://datatracker.ietf.org/doc/html/rfc5261) with a content type of ```application/xml-patch+xml```      
      
 **Mandatory Response Headers**:      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
 * ```ETag: [version id]``` where [version id] identifies the version of the ```Location``` resource returned.        
      
 **Response Body**:      
 * Success: ```Location``` resource that conforms to the [PCA Location](StructureDefinition-pca-location.html) profile.      
 * Error: ```OperationOutcome``` resource (see [FHIR specification](https://www.hl7.org/fhir/operationoutcome.html)).      
      
 **Required OAuth 2.0 scopes**:      
      
 The *scope* string **SHALL** contain the following *scoped role*:      
 * ``` Organization/[id]:SS_Updater```      
      
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.      
      
 **Response Codes**:          
 *  Code 200:  The resource was found and returned.      
 *  Code 404:  The resource was not found.      
 *  Code 422:  The input patch document violated applicable FHIR profiles or server data validation rules.      

<hr/>

### HealthcareService

       
     
 This section defines the supported interactions for the [PCA Healthcare Service](StructureDefinition-pca-healthcare-service.html) profile and the [PCA Partner Service](StructureDefinition-pca-partner-service.html) profile of the ```HealthcareService``` resource type.     
     
 A ```HealthcareService``` resource conforming to the [PCA Healthcare Service](StructureDefinition-pca-healthcare-service.html) profile describes a healthcare service offered at a location by a PCA publisher.     
     
 A ```HealthcareService``` resource conforming to the [PCA Partner Service](StructureDefinition-pca-partner-service.html) profile describes a partner service offered by a subscriber to which PCA publishers can publish their service offerings.     

<hr/>

### Server implementation options

<table border="1">
   <tr>
      <td>
         <b>Option</b>
      </td>
      <td>
         <b>Value</b>
      </td>
   </tr>
   <tr>
      <td>versioning</td>
      <td>versioned</td>
   </tr>
   <tr>
      <td>readHistory</td>
      <td>false</td>
   </tr>
   <tr>
      <td>updateCreate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalCreate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalRead</td>
      <td>not-supported</td>
   </tr>
   <tr>
      <td>conditionalUpdate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalDelete</td>
      <td>not-supported</td>
   </tr>
   <tr>
      <td>referencePolicy</td>
      <td>not specified</td>
   </tr>
   <tr>
      <td>searchInclude</td>
      <td>_include not supported</td>
   </tr>
   <tr>
      <td>searchRevInclude</td>
      <td>_revinclude not supported</td>
   </tr>
</table>

<hr/>

#### RESTful operations



<hr/>

#### read

**Definition**:      
      
 This operation is a RESTful **read** interaction (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#read)) of a ```HealthcareService``` resource.       
      
 **Behaviour**:      
      
 This operation is used to retrieve an identified ```HealthcareService``` resource that conforms to one of the following profiles:      
 * [PCA Healthcare Service](StructureDefinition-pca-healthcare-service.html)      
 * [PCA Partner Service](StructureDefinition-pca-partner-service.html)      
      
 As described [above](CapabilityStatement-pca-subscriber-agent.html#common-behaviour) a ```HealthcareService``` resource conforming to the [PCA Healthcare Service](StructureDefinition-pca-healthcare-service.html) profile excludes any *Identifiers* that cannot be disclosed to the subscriber.      
      
 **HTTP interaction signature**:      
 * ```GET [subscriber agent base url]/HealthcareService/[id]```      
      
 **Mandatory Request Headers**:      
 * ```Accept: application/fhir+json;charset=utf-8``` or ```Accept: application/fhir+xml;charset=utf-8```      
 * ```Authorization: Bearer [token]```      
      
 **Request Body**:      
 * (No request body)      
      
 **Mandatory Response Headers**:      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
 * ```ETag: [version id]``` where [version id] identifies the version of the ```HealthcareService``` resource returned.        
      
 **Response Body**:      
 * Success: ```HealthcareService``` resource that conforms to the [PCA Healthcare Service](StructureDefinition-pca-healthcare-service.html) profile.      
 * Error: ```OperationOutcome``` resource (see [FHIR specification](https://www.hl7.org/fhir/operationoutcome.html)).      
      
 **Required OAuth 2.0 scopes**:      
      
 The *scope* string **SHALL** contain the following *scoped role*:      
 * ``` Organization/[id]:SS_Receiver```      
      
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.      
      
 **Response Codes**:          
 * Code 200:  The resource was found and returned.      
 * Code 404:  The resource was not found.      

<hr/>

#### patch

**Definition**:      
      
 This operation is a RESTful **patch** interaction (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#patch)) of a ```HealthcareService``` resource.       
      
 **Behaviour**:      
      
 This operation is used to update an identified ```HealthcareService``` resource that conforms to the [PCA Healthcare Service](StructureDefinition-pca-healthcare-service.html) profile via a JSON or XML patch document. This operation will only process a patch operation for adding or updating an *identifier* element.      
      
 If any of the following data validation rules are violated, then the request is rejected and the error code 422 is returned:      
      
 * The patch document **SHALL** only contain an operation for adding or updating an *identifier* element.      
 * Only one *identifier* element per *identifier.system* **SHALL** be allowed for the identified ```HealthcareService``` resource.      
 * The patch operation **SHALL** only add or update an *identifier* element where *identifier.system* identifies a subscriber identifier system that is owned by the participating organisation associated with the target PCA Subscriber Agent.      
 * The patch operation **SHALL** only add or update an *identifier* element where a [PCA Match Record](StructureDefinition-pca-match-record.html) exists within the PCA service with:      
   * A *status* of 'completed', and;       
   * A *focus* referencing the ```HealthcareService``` resource, and;      
   * An *owner* referencing a [PCA Partner Service](StructureDefinition-pca-partner-service.html) of the participating organisation associated with the target PCA Subscriber Agent.      
      
 **HTTP interaction signature**:      
 * ```PATCH [subscriber agent base url]/HealthcareService/[id]```      
      
 **Mandatory Request Headers**:      
 * ```Accept: application/json-patch+json;charset=utf-8``` or ```Accept: application/xml-patch+xml;charset=utf-8```      
 * ```Content-Type: application/json-patch+json;charset=utf-8``` or ```Accept: application/xml-patch+xml;charset=utf-8```      
 * ```Authorization: Bearer [token]```      
      
 **Request Body**:      
 * A JSON Patch document that conforms to [https://datatracker.ietf.org/doc/html/rfc6902](https://datatracker.ietf.org/doc/html/rfc6902) with a content type of ```application/json-patch+json```, or;      
 * An XML Patch document that conforms to [https://datatracker.ietf.org/doc/html/rfc5261](https://datatracker.ietf.org/doc/html/rfc5261) with a content type of ```application/xml-patch+xml```      
      
 **Mandatory Response Headers**:      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
 * ```ETag: [version id]``` where [version id] identifies the version of the ```HealthcareService``` resource returned.        
      
 **Response Body**:      
 * Success: ```HealthcareService``` resource that conforms to the [PCA Healthcare Service](StructureDefinition-pca-healthcare-service.html) profile.      
 * Error: ```OperationOutcome``` resource (see [FHIR specification](https://www.hl7.org/fhir/operationoutcome.html)).      
      
 **Required OAuth 2.0 scopes**:      
      
 The *scope* string **SHALL** contain the following *scoped role*:      
 * ``` Organization/[id]:SS_Updater```      
      
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.      
      
 **Response Codes**:          
 *  Code 200:  The resource was found and returned.      
 *  Code 404:  The resource was not found.      
 *  Code 422:  The input patch document violated applicable FHIR profiles or server data validation rules.      

<hr/>

#### search-type

<b>Supported search parameters:</b>

<table border="1">
   <tr>
      <td>
         <b>Parameter</b>
      </td>
      <td>
         <b>Type</b>
      </td>
      <td>
         <b>Definition</b>
      </td>
   </tr>
   <tr>
      <td>_profile</td>
      <td>uri</td>
      <td>http://hl7.org/fhir/SearchParameter/Resource-profile</td>
   </tr>
   <tr>
      <td>active</td>
      <td>token</td>
      <td>http://hl7.org/fhir/SearchParameter/HealthcareService-active</td>
   </tr>
   <tr>
      <td>organization</td>
      <td>reference</td>
      <td>http://hl7.org/fhir/SearchParameter/HealthcareService-organization</td>
   </tr>
   <tr>
      <td>service-catgeory</td>
      <td>token</td>
      <td>http://hl7.org/fhir/SearchParameter/HealthcareService-service-category</td>
   </tr>
   <tr>
      <td>coverage-area.location.address-postalcode</td>
      <td>string</td>
      <td/>
   </tr>
</table>**Definition**:      
      
 This operation is a RESTful **search** interaction (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#search)) for ```HealthcareService``` resources that conform to the [PCA Partner Service](StructureDefinition-pca-partner-service.html) profile.       
      
 **Behaviour**:      
      
 The operation is used to find ```HealthcareService``` resources that that conform to the profile [PCA Partner Service](StructureDefinition-pca-partner-service.html) profile and that match the search parameters.      
 Note that all [PCA Partner Service](StructureDefinition-pca-partner-service.html) resources in the PCA service are accessible to all authorised client systems.      
      
 **HTTP interaction signature**:      
 * ```GET [subscriber agent base url]/HealthcareService/{?[parameters]}```      
      
 **Mandatory Request Headers**:      
 * ```Accept: application/fhir+json;charset=utf-8``` or ```Accept: application/fhir+xml;charset=utf-8```      
 * ```Authorization: Bearer [token]```      
      
 **Request Body**:      
 * (No request body)      
      
 **Mandatory Response Headers**:      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
      
 **Response Body**:      
 * Success: Bundle that contains zero to many ```HealthcareService``` resources that conform to the [PCA Partner Service](StructureDefinition-pca-partner-service.html) profile.      
 * Error: ```OperationOutcome``` resource (see [FHIR specification](https://www.hl7.org/fhir/operationoutcome.html)).      
      
 **Required OAuth 2.0 scopes**:      
      
 The *scope* string **SHALL** contain the following *scoped role*:      
 * ``` Organization/[id]:SS_Receiver```      
      
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.      
      
 **Response Codes**:          
 * Code 200:  Search successfully performed (but may return zero matching results).      
 * Code 403:  Server refused to perform the search.      

<hr/>

### PractitionerRole

       
     
 This section defines the supported interactions for the [PCA Practitioner Role](StructureDefinition-pca-practitioner-role.html) profile of the ```PractitionerRole``` resource type.     
Each ```PractitionerRole``` resource conforming to the [PCA Practitioner Role](StructureDefinition-pca-practitioner-role.html) profile describes a practitioner in a role at a healthcare service.     

<hr/>

#### Server implementation options

<table border="1">
   <tr>
      <td>
         <b>Option</b>
      </td>
      <td>
         <b>Value</b>
      </td>
   </tr>
   <tr>
      <td>versioning</td>
      <td>versioned</td>
   </tr>
   <tr>
      <td>readHistory</td>
      <td>false</td>
   </tr>
   <tr>
      <td>updateCreate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalCreate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalRead</td>
      <td>not-supported</td>
   </tr>
   <tr>
      <td>conditionalUpdate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalDelete</td>
      <td>not-supported</td>
   </tr>
   <tr>
      <td>referencePolicy</td>
      <td>not specified</td>
   </tr>
   <tr>
      <td>searchInclude</td>
      <td>PractitionerRole:practitioner</td>
   </tr>
   <tr>
      <td>searchInclude</td>
      <td>PractitionerRole:healthcareService</td>
   </tr>
   <tr>
      <td>searchInclude</td>
      <td>PractitionerRole:organization</td>
   </tr>
   <tr>
      <td>searchInclude</td>
      <td>PractitionerRole:location</td>
   </tr>
   <tr>
      <td>searchRevInclude</td>
      <td>_revinclude not supported</td>
   </tr>
</table>

<hr/>

#### RESTful operations



<hr/>

#### read

**Definition**:      
      
 This operation is a RESTful **read** interaction (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#read)) of a ```PractitionerRole``` resource.       
      
 **Behaviour**:      
      
 This operation retrieves an identified ```PractitionerRole``` resource that conforms to the [PCA Practitioner Role](StructureDefinition-pca-practitioner-role.html) profile. As described in the [common behaviour](CapabilityStatement-pca-subscriber-agent.html#common-behaviour) section, the ```PractitionerRole``` resource excludes any *Identifiers* that cannot be disclosed to the subscriber.      
      
 **HTTP interaction signature**:      
 * ```GET [subscriber agent base url]/PractitionerRole/[id]```      
      
 **Mandatory Request Headers**:      
 * ```Accept: application/fhir+json;charset=utf-8``` or ```Accept: application/fhir+xml;charset=utf-8```      
 * ```Authorization: Bearer [token]```      
      
 **Request Body**:      
 * (No request body)      
      
 **Mandatory Response Headers**:      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
 * ```ETag: [version id]``` where [version id] identifies the version of the ```PractitionerRole``` resource returned.        
      
 **Response Body**:      
 * Success: ```PractitionerRole``` resource that conforms to the [PCA Practitioner Role](StructureDefinition-pca-practitioner-role.html) profile.      
 * Error: ```OperationOutcome``` resource (see [FHIR specification](https://www.hl7.org/fhir/operationoutcome.html)).      
      
 **Required OAuth 2.0 scopes**:      
      
 The *scope* string **SHALL** contain the following *scoped role*:      
 * ```Organization/[id]:SS_Receiver```      
      
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.      
      
 **Response Codes**:          
*  Code 200:  The resource was found and returned.      
*  Code 404:  The resource was not found.      

<hr/>

#### patch

**Definition**:      
      
 This operation is a RESTful **patch** interaction (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#patch)) of a ```PractitionerRole``` resource.       
      
 **Behaviour**:      
      
 This operation is used to update an identified ```PractitionerRole``` resource that conforms to the [PCA Practitioner Role](StructureDefinition-pca-practitioner-role.html) profile via a JSON or XML patch document. This operation will only process a patch operation for adding or updating an *identifier* element.      
      
 If any of the following data validation rules are violated, then the request is rejected and the error code 422 is returned:      
      
 * The patch document **SHALL** only contain an operation for adding or updating an *identifier* element.      
 * Only one *identifier* element per *identifier.system* **SHALL** be allowed for the identified ```PractitionerRole``` resource.      
 * The patch operation **SHALL** only add or update an *identifier* element where *identifier.system* identifies a subscriber identifier system that is owned by the participating organisation associated with the target PCA Subscriber Agent.      
 * The patch operation **SHALL** only add or update an *identifier* element where a [PCA Match Record](StructureDefinition-pca-match-record.html) exists within the PCA service with:      
   * A *status* of 'completed', and;       
   * A *focus* referencing the ```PractitionerRole``` resource, and;      
   * An *owner* referencing a [PCA Partner Service](StructureDefinition-pca-partner-service.html) of the participating organisation associated with the target PCA Subscriber Agent.      
      
 **HTTP interaction signature**:      
 * ```PATCH [subscriber agent base url]/PractitionerRole/[id]```      
      
 **Mandatory Request Headers**:      
 * ```Accept: application/json-patch+json;charset=utf-8``` or ```Accept: application/xml-patch+xml;charset=utf-8```      
 * ```Content-Type: application/json-patch+json;charset=utf-8``` or ```Accept: application/xml-patch+xml;charset=utf-8```      
 * ```Authorization: Bearer [token]```      
      
 **Request Body**:      
 * A JSON Patch document that conforms to [https://datatracker.ietf.org/doc/html/rfc6902](https://datatracker.ietf.org/doc/html/rfc6902) with a content type of ```application/json-patch+json```, or;      
 * An XML Patch document that conforms to [https://datatracker.ietf.org/doc/html/rfc5261](https://datatracker.ietf.org/doc/html/rfc5261) with a content type of ```application/xml-patch+xml```      
      
 **Mandatory Response Headers**:      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
 * ```ETag: [version id]``` where [version id] identifies the version of the ```PractitionerRole``` resource returned.        
      
 **Response Body**:      
 * Success: ```PractitionerRole``` resource that conforms to the [PCA Practitioner Role](StructureDefinition-pca-practitioner-role.html) profile.      
 * Error: ```OperationOutcome``` resource (see [FHIR specification](https://www.hl7.org/fhir/operationoutcome.html)).      
      
 **Required OAuth 2.0 scopes**:      
      
 The *scope* string **SHALL** contain the following *scoped role*:      
 * ``` Organization/[id]:SS_Updater```      
      
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.      
      
 **Response Codes**:          
 *  Code 200:  The resource was found and returned.      
 *  Code 404:  The resource was not found.      
 *  Code 422:  The input patch document violated applicable FHIR profiles or server data validation rules.      

<hr/>

### Practitioner

     
     
 This section defines the supported interactions for the [PCA Practitioner](StructureDefinition-pca-practitioner.html) profile of the ```Practitioner``` resource type.       
     
Each ```Practitioner``` resource conforming the [PCA Practitioner](StructureDefinition-pca-practitioner.html) profile is a read-only proxy for a corresponding ```Practitioner``` resource managed by the PCA portal.     
     
Each 'proxy' has the same *id* and *versionId* as the resource within the PCA portal for which it is a proxy.       

<hr/>

#### Server implementation options

<table border="1">
   <tr>
      <td>
         <b>Option</b>
      </td>
      <td>
         <b>Value</b>
      </td>
   </tr>
   <tr>
      <td>versioning</td>
      <td>versioned</td>
   </tr>
   <tr>
      <td>readHistory</td>
      <td>false</td>
   </tr>
   <tr>
      <td>updateCreate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalCreate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalRead</td>
      <td>not-supported</td>
   </tr>
   <tr>
      <td>conditionalUpdate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalDelete</td>
      <td>not-supported</td>
   </tr>
   <tr>
      <td>referencePolicy</td>
      <td>not specified</td>
   </tr>
   <tr>
      <td>searchInclude</td>
      <td>_include not supported</td>
   </tr>
   <tr>
      <td>searchRevInclude</td>
      <td>_revinclude not supported</td>
   </tr>
</table>

<hr/>

#### RESTful operations



<hr/>

#### read

**Definition**:      
      
 This operation is a RESTful **read** interaction (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#read)) of a ```Practitioner``` resource.       
      
 **Behaviour**:      
      
 This operation retrieves an identified ```Practitioner``` resource that conforms to the [PCA Practitioner](StructureDefinition-pca-practitioner.html) profile. As described in the [common behaviour](CapabilityStatement-pca-subscriber-agent.html#common-behaviour) section, the ```Practitioner``` resource excludes any *Identifiers* that cannot be disclosed to the subscriber.      
      
 **HTTP interaction signature**:      
 * ```GET [subscriber agent base url]/Practitioner/[id]```      
      
 **Mandatory Request Headers**:      
 * ```Accept: application/fhir+json;charset=utf-8``` or ```Accept: application/fhir+xml;charset=utf-8```      
 * ```Authorization: Bearer [token]```      
      
 **Request Body**:      
 * (No request body)      
      
 **Mandatory Response Headers**:      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
 * ```ETag: [version id]``` where [version id] identifies the version of the ```Practitioner``` resource returned.        
      
 **Response Body**:      
 * Success: ```Practitioner``` resource that conforms to the [PCA Practitioner](StructureDefinition-pca-practitioner.html) profile.      
 * Error: ```OperationOutcome``` resource (see [FHIR specification](https://www.hl7.org/fhir/operationoutcome.html)).      
      
 **Required OAuth 2.0 scopes**:      
      
 The *scope* string **SHALL** contain the following *scoped role*:      
 * ``` Organization/[id]:SS_Receiver```      
      
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.      
      
 **Response Codes**:          
 * Code 200:  The resource was found and returned.      
 * Code 404:  The resource was not found.      

<hr/>

### Provenance

     
     
 This section defines the supported interactions for the [PCA Provenance](StructureDefinition-pca-provenance.html) profile of the ```Provenance``` resource type.     
     
 Each ```Provenance``` resource conforming to the [PCA Provenance](StructureDefinition-pca-provenance.html) profile represents an activity performed by the PCA service (conditionally on behalf of a participating organisation when the target element references a PCA Organization resource) whereby specific data elements of a PCA Organization or PCA Practitioner resource have been initially retrieved, updated or verified from an external system of record.   

<hr/>

#### Server implementation options

<table border="1">
   <tr>
      <td>
         <b>Option</b>
      </td>
      <td>
         <b>Value</b>
      </td>
   </tr>
   <tr>
      <td>versioning</td>
      <td>no-version</td>
   </tr>
   <tr>
      <td>readHistory</td>
      <td>false</td>
   </tr>
   <tr>
      <td>updateCreate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalCreate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalRead</td>
      <td>not-supported</td>
   </tr>
   <tr>
      <td>conditionalUpdate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalDelete</td>
      <td>not-supported</td>
   </tr>
   <tr>
      <td>referencePolicy</td>
      <td>not specified</td>
   </tr>
   <tr>
      <td>searchInclude</td>
      <td>_include not supported</td>
   </tr>
   <tr>
      <td>searchRevInclude</td>
      <td>_revinclude not supported</td>
   </tr>
</table>

<hr/>

#### RESTful operations



<hr/>

#### read

**Definition**:      
      
 This operation is a RESTful **read** interaction (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#read)) of a ```Provenance``` resource.       
      
 **Behaviour**:      
      
 This operation retrieves an identified ```Provenance``` resource that conforms to the [PCA Provenance](StructureDefinition-pca-practitioner-role.html) profile.      
      
 **HTTP interaction signature**:      
 * ```GET [subscriber agent base url]/Provenance/[id]```      
      
 **Mandatory Request Headers**:      
 * ```Accept: application/fhir+json;charset=utf-8``` or ```Accept: application/fhir+xml;charset=utf-8```      
 * ```Authorization: Bearer [token]```      
      
 **Request Body**:      
 * (No request body)      
      
 **Mandatory Response Headers**:      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
      
 **Response Body**:      
 * Success: ```Provenance``` resource that conforms to the [PCA Provenance](StructureDefinition-pca-provenance.html) profile.      
 * Error: ```OperationOutcome``` resource (see [FHIR specification](https://www.hl7.org/fhir/operationoutcome.html)).      
      
 **Required OAuth 2.0 scopes**:      
      
  The *scope* string **SHALL** contain the following *scoped role*:      
 * ``` Organization/[id]:SS_Receiver```      
      
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.      
      
 **Response Codes**:          
 * Code 200:  The resource was found and returned.      
 * Code 404:  The resource was not found.      

<hr/>

### Contract

     
     
 This section defines the supported interactions for the [PCA Publication Authorisation Record](StructureDefinition-pca-publication-authorisation-record.html) profile and the [PCA Information Use Statement](StructureDefinition-pca-information-use-statement.html) profile of the ```Contract``` resource type.     
     
     
 A ```Contract``` resource conforming to the [PCA Publication Authorisation Record](StructureDefinition-pca-publication-authorisation-record.html) profile records a decision regarding the publication of a specific service offering (either a ```HealthcareService``` or a ```PractitionerRole```) to a specific partner service. The decision is either to authorise publication or to revoke a previous authority to publish.     
     
 A ```Contract``` resource conforming to the [PCA Information Use Statement](StructureDefinition-pca-information-use-statement.html) profile represents a policy statement by a subscriber organisation's partner service that describes how they will use the information from publisher organisations that is disclosed to the partner service.         

<hr/>

#### Server implementation options

<table border="1">
   <tr>
      <td>
         <b>Option</b>
      </td>
      <td>
         <b>Value</b>
      </td>
   </tr>
   <tr>
      <td>versioning</td>
      <td>versioned</td>
   </tr>
   <tr>
      <td>readHistory</td>
      <td>false</td>
   </tr>
   <tr>
      <td>updateCreate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalCreate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalRead</td>
      <td>not-supported</td>
   </tr>
   <tr>
      <td>conditionalUpdate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalDelete</td>
      <td>not-supported</td>
   </tr>
   <tr>
      <td>referencePolicy</td>
      <td>not specified</td>
   </tr>
   <tr>
      <td>searchInclude</td>
      <td>_include not supported</td>
   </tr>
   <tr>
      <td>searchRevInclude</td>
      <td>_revinclude not supported</td>
   </tr>
</table>

<hr/>

#### RESTful operations



<hr/>

#### read

**Definition**:      
      
 This operation is a RESTful **read** interaction (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#read)) of a ```Contract``` resource.       
      
 This operation is used to retrieve an identified ```Contract``` resource that conforms to one of the following profiles:      
 * [PCA Publication Authorisation Record](StructureDefinition-pca-publication-authorisation-record.html)      
 * [PCA Information Use Statement](StructureDefinition-pca-information-use-statement.html)      
      
 Different authorisation rules apply depending on the profile of the ```Contract``` resource being retrieved; see the 'Required OAuth 2.0 scopes' section       
      
 **HTTP interaction signature**:      
 * ```GET [subscriber agent base url]/Contract/[id]```      
      
 **Mandatory Request Headers**:      
 * ```Accept: application/fhir+json;charset=utf-8``` or ```Accept: application/fhir+xml;charset=utf-8```      
 * ```Authorization: Bearer [token]```      
      
 **Request Body**:      
 * (No request body)      
      
 **Mandatory Response Headers**:      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
      
 **Response Body**:      
 * Success: ```Contract``` resource that conforms to the [PCA Publication Authorisation Record](StructureDefinition-pca-publication-authorisation-record.html) profile.      
 * Error: ```OperationOutcome``` resource (see [FHIR specification](https://www.hl7.org/fhir/operationoutcome.html)).      
      
 **Required OAuth 2.0 scopes**:      
      
 The *scope* string **SHALL** contain the following *scoped role*:      
 * ``` Organization/[id]:SS_Receiver```      
      
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.      
      
 **Response Codes**:          
 * Code 200:  The resource was found and returned.      
 * Code 404:  The resource was not found.      

<hr/>

#### search-type

<b>Supported search parameters:</b>

<table border="1">
   <tr>
      <td>
         <b>Parameter</b>
      </td>
      <td>
         <b>Type</b>
      </td>
      <td>
         <b>Definition</b>
      </td>
   </tr>
   <tr>
      <td>_profile</td>
      <td>uri</td>
      <td>http://hl7.org/fhir/SearchParameter/Resource-profile</td>
   </tr>
   <tr>
      <td>subject</td>
      <td>reference</td>
      <td>http://hl7.org/fhir/SearchParameter/Contract-subject</td>
   </tr>
   <tr>
      <td>version</td>
      <td>string</td>
      <td/>
   </tr>
</table>**Definition**:      
      
 This operation is a RESTful **search** interaction (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#search)) for ```Contract``` resources.       
      
 **Behaviour**:      
      
 This operation is used to find ```Contract``` resources that conform to the [PCA Information Use Statement](StructureDefinition-pca-information-use-statement.html) profile and that match the search parameters.       
      
 Note that all [PCA Information Use Statement](StructureDefinition-pca-information-use-statement.html) resources in the PCA service are accessible to all authorised client systems.      
      
 **HTTP interaction signature**:      
 * ```GET [subscriber agent base url]/Contract/{?[parameters]}```      
      
 **Mandatory Request Headers**:      
 * ```Accept: application/fhir+json;charset=utf-8``` or ```Accept: application/fhir+xml;charset=utf-8```      
 * ```Authorization: Bearer [token]```      
      
 **Request Body**:      
 * (No request body)      
      
 **Mandatory Response Headers**:      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
      
 **Response Body**:      
 * Success: Bundle containing zero to many ```Contract``` resources that conform to the profile [PCA Information Use Statement](StructureDefinition-pca-information-use-statement.html) profile.      
 * Error: ```OperationOutcome``` resource (see [FHIR specification](https://www.hl7.org/fhir/operationoutcome.html)).      
      
 **Required OAuth 2.0 scopes**:      
      
 The *scope* string **SHALL** contain the following *scoped role*:      
 * ``` Organization/[id]:SS_Receiver```      
      
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.      
      
 **Response Codes**:          
 * Code 200:  Search successfully performed (but may return zero matching results).      
 * Code 403:  Server refused to perform the search.      

<hr/>

### Task

     
     
 This section defines the supported interactions for the [PCA Match Record](StructureDefinition-pca-match-record.html) profile of the ```Task``` resource type.     
     
 A ```Task``` resource conforming to the [PCA Match Record](StructureDefinition-pca-match-record.html) profile indicates the matching status of a service offering to a partner service’s local data store.       

<hr/>

### Server implementation options

<table border="1">
   <tr>
      <td>
         <b>Option</b>
      </td>
      <td>
         <b>Value</b>
      </td>
   </tr>
   <tr>
      <td>versioning</td>
      <td>versioned-update</td>
   </tr>
   <tr>
      <td>readHistory</td>
      <td>false</td>
   </tr>
   <tr>
      <td>updateCreate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalCreate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalRead</td>
      <td>not-supported</td>
   </tr>
   <tr>
      <td>conditionalUpdate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalDelete</td>
      <td>not-supported</td>
   </tr>
   <tr>
      <td>referencePolicy</td>
      <td>not specified</td>
   </tr>
   <tr>
      <td>searchInclude</td>
      <td>_include not supported</td>
   </tr>
   <tr>
      <td>searchRevInclude</td>
      <td>_revinclude not supported</td>
   </tr>
</table>

<hr/>

#### RESTful operations



<hr/>

#### read

**Definition**:      
      
 This operation is a RESTful **read** interaction (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#read)) of a ```Task``` resource.       
      
 **Behaviour**:      
      
 This operation retrieves an identified ```Task``` resource that conforms to the [PCA Match Record](StructureDefinition-pca-match-record.html) profile.       
       
 **HTTP interaction signature**:      
 * ```GET [subscriber agent base url]/Task/[id]```      
      
 **Mandatory Request Headers**:      
 * ```Accept: application/fhir+json;charset=utf-8``` or ```Accept: application/fhir+xml;charset=utf-8```      
 * ```Authorization: Bearer [token]```      
      
 **Request Body**:      
 * (No request body)      
      
 **Mandatory Response Headers**:      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
      
 **Response Body**:      
 * Success: ```Task``` resource that conforms to the [PCA Match Record](StructureDefinition-pca-match-record.html) profile.      
 * Error: ```OperationOutcome``` resource (see [FHIR specification](https://www.hl7.org/fhir/operationoutcome.html)).      
      
 **Required OAuth 2.0 scopes**:      
      
 The *scope* string **SHALL** contain the following *scoped role*:      
 * ``` Organization/[id]:SS_Receiver```      
      
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.      
      
 **Response Codes**:          
 * Code 200:  The resource was found and returned.      
 * Code 404:  The resource was not found.      

<hr/>

#### create

**Definition**:      
      
 This operation is a RESTful **create** interaction (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#create)) of a ```Task``` resource conforming to the [PCA Match Record](StructureDefinition-pca-match-record.html) profile.       
      
 **Behaviour**:      
      
 If any of the following data validation rules are violated in the input resource then the request is rejected and the error code 422 is returned:      
      
 * The resource **SHALL** conform to the [PCA Match Record](StructureDefinition-pca-match-record.html) profile.      
 * *id* **SHALL NOT** be present.         
 * *meta.versionId* **SHALL NOT** be present.         
 * *meta.lastUpdated* **SHALL NOT** be present.         
 * *focus* **SHALL** reference a resource that has been published to one or more [Partner Services](StructureDefinition-pca-partner-service.html) associated with the target PCA Subscriber Agent.       
 * *owner* **SHALL** reference a [Partner Service](StructureDefinition-pca-partner-service.html) associated with the target PCA Subscriber Agent.      
 * *note.text* **SHALL** be a maximum of 200 characters.      
       
 **HTTP interaction signature**:      
 * ```POST [subscriber agent base url]/Task```      
      
 **Mandatory Request Headers**:      
 * ```Accept: application/fhir+json;charset=utf-8``` or ```Accept: application/fhir+xml;charset=utf-8```      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
 * ```Authorization: Bearer [token]```      
      
 **Request Body**:      
 * ```Task``` resource that conforms to the [PCA Match Record](StructureDefinition-pca-match-record.html) profile.      
      
 **Mandatory Response Headers**:      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
 * ```ETag: [version id]``` where [version id] identifies the version of the new ```Task``` resource.        
      
 **Response Body**:      
 * Success: ```Task``` resource that conforms to the [PCA Match Record](StructureDefinition-pca-match-record.html) profile.      
*  Error: ```OperationOutcome``` resource (see [FHIR specification](https://www.hl7.org/fhir/operationoutcome.html)).      
      
 **Required OAuth 2.0 scopes**:      
      
 The *scope* string **SHALL** contain the following *scoped role*:      
 * ``` Organization/[id]:SS_Updater```      
      
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.      
      
 **Response Codes**:          
 * Code 201:  The resource was successfully created.      
 * Code 400:  The input resource could not be parsed or failed basic FHIR validation rules.      
 * Code 404:  The resource type is not supported, or not a FHIR end-point.      
 * Code 422:  The input resource violated applicable FHIR profiles or server data validation rules.      

<hr/>

#### update

**Definition**:      
      
 This operation is a RESTful **update** interaction (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#update)) of a ```Task``` resource.       
      
 **Behaviour**:      
      
 If any of the following data validation rules are violated in the input resource then the request is rejected and the error code 422 is returned:      
      
 * The resource **SHALL** conform to the [PCA Match Record](StructureDefinition-pca-match-record.html) profile.      
 * *id* **SHALL** be identical to the corresponding value in the version of the ```Task``` identified in the 'if-match' HTTP header.       
 * *meta.versionId* **SHALL NOT** be present.         
 * *meta.lastUpdated* **SHALL NOT** be present.         
 * *owner* **SHALL** reference a [Partner Service](StructureDefinition-pca-partner-service.html) associated with the target PCA Subscriber Agent.      
 * Only changes to *status* (and optionally *note.text*) will be accepted; changes to any other element values will not be accepted when updating an existing ```Task```.       
 * *note.text* **SHALL** only be updated if *status* is being changed from 'completed' to 'failed'.      
 * *note.text* **SHALL** be a maximum of 200 characters.      
      
 If the 'if-match' HTTP header is not provided in the request, then the PCA Subscriber Agent will reject the request and return the error code 400.        
      
 If the 'if-match' HTTP header does not match the current version, then the PCA Subscriber Agent will reject the request and return the error code 412.            
      
 **HTTP interaction signature**:      
 * ```PUT [subscriber agent base url]/Task/[id]```      
      
 **Mandatory Request Headers**:      
 * ```Accept: application/fhir+json;charset=utf-8``` or ```Accept: application/fhir+xml;charset=utf-8```      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
 * ```Authorization: Bearer [token]```      
 * ```if-match [version id]``` where [version id] is the version of the ```Task``` being updated.      
      
 **Request Body**:      
 ```Task``` resource that conforms to the [PCA Match Record](StructureDefinition-pca-match-record.html) profile.      
      
 **Mandatory Response Headers**:      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
 * ```ETag: [version id]``` where [version id] identifies the version of the ```Task``` resource returned.        
      
 **Response Body**:      
 * Success: ```Task``` resource that conforms to the [PCA Match Record](StructureDefinition-pca-match-record.html) profile.      
*  Error: ```OperationOutcome``` resource (see [FHIR specification](https://www.hl7.org/fhir/operationoutcome.html)).      
      
 **Required OAuth 2.0 scopes**:      
      
 The *scope* string **SHALL** contain the following *scoped role*:      
 * ``` Organization/[id]:SS_Updater```      
      
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.      
      
 **Response Codes**:          
 * Code 200:  The resource was found and returned.      
 * Code 400:  Client error:  request did not include the *if-match* HTTP header.      
 * Code 404:  The resource was not found.      
 * Code 412:  The version id given in the *if-Match* HTTP request header does not match the current version.      
 * Code 422:  The input resource violated applicable FHIR profiles or server data validation rules.      

<hr/>

### Subscription

     
      
 This section defines the supported interactions for the [PCA Subscription](StructureDefinition-pca-subscription.html) profile of the ```Subscription``` resource type.     
     
 The [PCA Subscription](StructureDefinition-pca-subscription.html) profile of ```Subscription``` is used by subscribing systems to register for notifications from the PCA service when new or updated service offerings are available, or when publication of existing service offerings has been newly revoked.       

<hr/>

#### Server implementation options

<table border="1">
   <tr>
      <td>
         <b>Option</b>
      </td>
      <td>
         <b>Value</b>
      </td>
   </tr>
   <tr>
      <td>versioning</td>
      <td>no-version</td>
   </tr>
   <tr>
      <td>readHistory</td>
      <td>false</td>
   </tr>
   <tr>
      <td>updateCreate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalCreate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalRead</td>
      <td>not-supported</td>
   </tr>
   <tr>
      <td>conditionalUpdate</td>
      <td>false</td>
   </tr>
   <tr>
      <td>conditionalDelete</td>
      <td>not-supported</td>
   </tr>
   <tr>
      <td>referencePolicy</td>
      <td>not specified</td>
   </tr>
   <tr>
      <td>searchInclude</td>
      <td>_include not supported</td>
   </tr>
   <tr>
      <td>searchRevInclude</td>
      <td>_revinclude not supported</td>
   </tr>
</table>

<hr/>

#### RESTful operations



<hr/>

#### create

**Definition**:      
      
 This operation is a RESTful **create** interaction (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#create)) of a ```Subscription``` resource conforming to the [PCA Subscription](StructureDefinition-pca-subscription.html) profile.       
      
 **Behaviour**:      
      
 This operation is used to register a callback to receive a notification whenever there are changes in the data set that is published to a subscriber.      
      
 A notification is simply a HTTP POST without any content – the purpose is to trigger the subscriber’s client system to get the available service offerings by invoking the [whole system search operation](CapabilityStatement-pca-subscriber-agent.html#search-system).      
      
 If any of the following data validation rules are violated in the input resource, then the request is rejected and the error code 422 is returned:      
 * *status* **SHALL** be 'requested'      
 * *reason* **SHALL** be 'subscriber change notifications'      
 * *criteria*  **SHALL** be a space separated list of resource types from the list 'HealthcareService', 'PractitionerRole', 'Location', 'Organization', 'Practitioner', 'Provenance', 'Contract', 'Task'       
 * *channel.type*  **SHALL** be 'rest-hook'       
 * *channel.endpoint* **SHALL** be the URL of an endpoint that is operated by (or on behalf of) the subscriber for receiving notifications. The protocol **SHALL** be https.      
 * *channel.payload* **SHALL NOT** be present – its absence indicates that the subscriber will not receive any resources via this endpoint.       
 * *channel.header* **SHALL** be set to 'Authorization: Bearer [secret string]' where [secret string] is the basic authentication string, set by the subscriber organisation, that is used to authenticate the PCA Subscriber Agent to the notification receiving endpoint.      
      
 **HTTP interaction signature**:      
 * ```POST [subscriber agent base url]/Subscription```      
      
 **Mandatory Request Headers**:      
 * ```Accept: application/fhir+json;charset=utf-8``` or ```Accept: application/fhir+xml;charset=utf-8```      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
 * ```Authorization: Bearer [token]```      
      
 **Request Body**:      
 * ```Subscription``` resource that conforms to the [PCA Subscription](StructureDefinition-pca-subscription.html) profile.      
      
 **Mandatory Response Headers**:      
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```      
      
 **Response Body**:      
 * Success: ```Subcription``` resource that conforms to the [PCA Subscription](StructureDefinition-pca-subscription.html) profile.      
 * Error: ```OperationOutcome``` resource (see [FHIR specification](https://www.hl7.org/fhir/operationoutcome.html)).      
      
 **Required OAuth 2.0 scopes**:      
      
 The *scope* string **SHALL** contain the following *scoped role*:      
 * ``` Organization/[id]:SS_Receiver```      
      
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.      
      
 **Response Codes**:          
 * Code 201:  The resource was successfully created.      
 * Code 400:  The input resource could not be parsed or failed basic FHIR validation rules.      
 * Code 404:  The resource type is not supported, or not a FHIR end-point.      
 * Code 422:  The input resource violated applicable FHIR profiles or server data validation rules.      

<hr/>

#### search-type

<b>Supported search parameters:</b>

<table border="1">
   <tr>
      <td>
         <b>Parameter</b>
      </td>
      <td>
         <b>Type</b>
      </td>
      <td>
         <b>Definition</b>
      </td>
   </tr>
</table>**Definition**:       
       
 This operation is a RESTful **search** interaction (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#search)) for a ```Subscription``` resource that is associated with the target Subscriber Agent and that conforms to the [PCA Subscription](StructureDefinition-pca-subscription.html) profile.        
       
 **Behaviour**:       
       
 This operation is used to find a ```Subscription``` resource that is associated with the target Subscriber Agent and that conforms to the [PCA Subscription](StructureDefinition-pca-subscription.html) profile.       
       
 **HTTP interaction signature**:       
        
 * ```GET [subscriber agent base url]/Subscription/```       
       
 **Mandatory Request Headers**:       
 * ```Accept: application/fhir+json;charset=utf-8``` or ```Accept: application/fhir+xml;charset=utf-8```       
 * ```Authorization: Bearer [token]```       
       
 **Request Body**:       
 * (No request body)       
       
 **Mandatory Response Headers**:       
 * ```Content-Type: application/fhir+json;charset=utf-8``` or ```Content-Type: application/fhir+xml;charset=utf-8```       
       
 **Response Body**:       
 * Success: ```Bundle``` that contains zero or one ```Subscription``` resource that that is associated with the target Subscriber Agent and that conforms to the [PCA Subscription](StructureDefinition-pca-subscription.html) profile.       
 * Error: ```OperationOutcome``` resource (see [FHIR specification](https://www.hl7.org/fhir/operationoutcome.html)).       
       
 **Required OAuth 2.0 scopes**:       
       
 The *scope* string **SHALL** contain the following *scoped role*:       
 * ``` Organization/[id]:SS_Receiver```       
       
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.       
       
 **Response Codes**:           
 * Code 200:  Search successfully performed (but may return zero matching results).       
 * Code 403:  Server refused to perform the search.       

<hr/>

#### delete

**Definition**:      
      
 This operation is a RESTful **delete** interaction (as described in the [FHIR specification](https://www.hl7.org/fhir/http.html#delete)) of a ```Subscription``` resource.       
      
 **Behaviour**:      
      
 This operation is used to deregister an existing notification callback.       
      
 **HTTP interaction signature**:      
 * ```DELETE [subscriber agent base url]/Subscription/[id]```      
      
 **Mandatory Request Headers**:      
 * ```Authorization: Bearer [token]```      
      
 **Request Body**:      
 * (No request body)      
      
 **Mandatory Response Headers**:      
 * (No request body)      
      
 **Response Body**:      
 * (No request body)      
      
 **Required OAuth 2.0 scopes**:      
      
 The *scope* string **SHALL** contain the following *scoped role*:      
 * ``` Organization/[id]:SS_Receiver```      
      
 Where [id] **SHALL** identify the ```Organization``` associated with the target PCA Subscriber Agent.      
      
 **Response Codes**:          
 * Code 202:  The deletion was accepted.      
 * Code 404:  The resource was not found.      