﻿/*
 * Copyright 2013 NEHTA
 *
 * Licensed under the NEHTA Open Source (Apache) License; you may not use this
 * file except in compliance with the License. A copy of the License is in the
 * 'license.txt' file, which should be provided with this work.
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or imp201112131459lied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

using System;
using CDA.Generator.Common.Common.Time.Enum;
using Nehta.VendorLibrary.CDA.SCSModel.Common;
using Nehta.VendorLibrary.Common;

namespace Nehta.VendorLibrary.CDA.Common
{
    /// <summary>
    /// Formats the interval for the narrative.
    /// </summary>
    public static class CdaIntervalFormatter
    {
        /// <summary>
        /// Formats the interval.
        /// </summary>
        /// <param name="cdaInterval">Interval</param>
        /// <returns>Narrative text.</returns>
        public static string Format(CdaInterval cdaInterval)
        {
            if (cdaInterval == null)
            {
                return null;
            }

            if (cdaInterval.ShowOngoingInNarrative.HasValue && cdaInterval.ShowOngoingInNarrative.Value)
            {
              return "(ongoing)";
            }

            if (cdaInterval.Type == IntervalType.Low)
            {
                // Low
                return cdaInterval.Low.NarrativeText();
            }
            if (cdaInterval.Type == IntervalType.Width)
            {
                // Width
                return Format(cdaInterval.IntervalWidth);
            }
            if (cdaInterval.Type == IntervalType.LowWidth)
            {
                // Low/Width
              return cdaInterval.Low.NarrativeText() + " (" + Format(cdaInterval.IntervalWidth) + " )";
            }
            if (cdaInterval.Type == IntervalType.LowHigh)
            {
                // Low/High
                return cdaInterval.Low.NarrativeText() + " -> " + cdaInterval.High.NarrativeText();
            }
            if (cdaInterval.Type == IntervalType.Center)
            {
                // Center
                return cdaInterval.Center.NarrativeText();
            }
            if (cdaInterval.Type == IntervalType.CenterWidth)
            {
                // Center/Width
              return cdaInterval.Center.NarrativeText() + " (" + Format(cdaInterval.IntervalWidth) + " )";
            }
            if (cdaInterval.Type == IntervalType.High)
            {
                // High
                return cdaInterval.High.NarrativeText();
            }
            if (cdaInterval.Type == IntervalType.HighWidth)
            {
                // High/Width
              return cdaInterval.High.NarrativeText() + " (" + Format(cdaInterval.IntervalWidth) + " )";
            }

            return null;
        }

        /// <summary>
        /// This is used for sotring purposes only in the narrative
        /// </summary>
        /// <param name="cdaInterval">Interval</param>
        /// <returns>Narrative text.</returns>
        public static DateTime GetFirstDateTimeOfDurrationForNarrativeSorting(CdaInterval cdaInterval)
        {
          if (cdaInterval.ShowOngoingInNarrative.HasValue && cdaInterval.ShowOngoingInNarrative.Value)
          {
            return DateTime.MinValue;
          }

          if (cdaInterval.Type == IntervalType.Low)
          {
            // Low
            return cdaInterval.Low.DateTime;
          }
          if (cdaInterval.Type == IntervalType.LowWidth)
          {
            // Low/Width
            return cdaInterval.Low.DateTime;
          }
          if (cdaInterval.Type == IntervalType.LowHigh)
          {
            if (cdaInterval.Low != null)
            {
              return cdaInterval.Low.DateTime;
            } 
              
            if (cdaInterval.High != null)
            {
              return cdaInterval.Low.DateTime;
            }
          }
          if (cdaInterval.Type == IntervalType.Center)
          {
            // Center
            return cdaInterval.Center.DateTime;
          }
          if (cdaInterval.Type == IntervalType.CenterWidth)
          {
            // Center/Width
            return cdaInterval.Center.DateTime;
          }
          if (cdaInterval.Type == IntervalType.High)
          {
            // High
            return cdaInterval.High.DateTime;
          }
          if (cdaInterval.Type == IntervalType.HighWidth)
          {
            // High/Width
            return cdaInterval.High.DateTime;
          }

          return DateTime.MinValue;
        }


        /// <summary>
        /// Formats the interval width for the narrative.
        /// </summary>
        /// <param name="intervalWidth">CdaIntervalWidth</param>
        /// <returns>Narrative text.</returns>
        public static string Format(CdaIntervalWidth intervalWidth)
        {
            if (intervalWidth == null)
            {
                return null;
            }

            return intervalWidth.Value + " " + intervalWidth.Unit.GetAttributeValue<NameAttribute, string>(x => x.Name) + "(s)";
        }
    }
}
