﻿/*
 * Copyright 2013 NEHTA
 *
 * Licensed under the NEHTA Open Source (Apache) License; you may not use this
 * file except in compliance with the License. A copy of the License is in the
 * 'license.txt' file, which should be provided with this work.
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Runtime.Serialization;
using System.Xml;
using CDA.Generator.Common.Common.Attributes;
using Nehta.VendorLibrary.CDA.CDAModel;
using Nehta.VendorLibrary.CDA.Generator.Enums;
using Nehta.VendorLibrary.CDA.SCSModel;
using Nehta.VendorLibrary.Common;
using Nehta.VendorLibrary.CDA.SCSModel.Common;

namespace Nehta.VendorLibrary.CDA.Common
{
    /// <summary>
    /// This EReferral object is a composition of the context and content objects that define
    /// a CDA EReferral document
    /// 
    /// This object is also responsible for providing the factory methods used to instantiate any E-Referral
    /// objects that are required to build a valid E-Referral CDA document
    /// </summary>
    [DataContract]
    [KnownType(typeof(Content))]
    [KnownType(typeof(Context))]
    [KnownType(typeof(CDAContext))]
    [KnownType(typeof(BaseCDAModel))]
    [TemplatePackedge(DocumentName = "eReferral", TemplatePackages = ".12 (1A) .9 (1B) .10 (2) .11 (3A)")]
    public class EReferral : BaseCDAModel
    {
        #region Properties
        /// <summary>
        /// An ICDAContextEReferral that contains the CDA context for this e-referral record
        /// </summary>
        [DataMember]
        public ICDAContextEReferral CDAContext { get; set; }

        /// <summary>
        /// An IEReferralContent that contains the content for this e-referral record
        /// </summary>
        [DataMember]
        public IEReferralContent SCSContent { get; set; }

        /// <summary>
        /// An IDispenseRecordContext that contains the context for this e-referral record
        /// </summary>
        [DataMember]
        public IEReferralContext SCSContext { get; set; }
        #endregion

        #region Constructors
        /// <summary>
        /// Instantiates an E-Referral model; the status of this CDA document will be 
        /// set to the status passed into this constructor
        /// </summary>
        /// <param name="documentStatus">Document Status</param>
        public EReferral(DocumentStatus documentStatus)
        {
            DocumentStatus = documentStatus;
        }

        /// <summary>
        /// Instantiates an E-Referral model; the status of this document will be 
        /// set to final
        /// </summary>
        public EReferral() : this(DocumentStatus.Final)
        {
        }
        #endregion

        #region Validation
        /// <summary>
        /// Validates this EReferral object and its child objects
        /// </summary>
        /// <param name="path">The path to this object as a string</param>
        /// <param name="messages">the validation messages to date, these may be added to within this method</param>
        public void Validate(string path, List<ValidationMessage> messages)
        {
            var vb = new ValidationBuilder(path, messages);

            if (vb.ArgumentRequiredCheck("CDAContext", CDAContext))
                CDAContext.Validate(vb.Path + "CDAContext", vb.Messages);

            if (vb.ArgumentRequiredCheck("SCSContext", SCSContext))
                SCSContext.Validate(vb.Path + "SCSContext", vb.Messages);
 
            if (vb.ArgumentRequiredCheck("SCSContent", SCSContent))
                SCSContent.Validate(vb.Path + "SCSContent", vb.Messages);
        }
        #endregion

        #region Static Methods
        /// <summary>
        /// Creates an EReferral
        /// </summary>
        /// <returns>EReferral</returns>
        public static EReferral CreateEReferral()
        {
            return new EReferral();
        }

        /// <summary>
        /// Creates a CDA Context
        /// </summary>
        /// <returns>ICDAContextEReferral</returns>
        public static ICDAContextEReferral CreateCDAContext()
        {
            return new CDAContext();
        }

        /// <summary>
        /// Creates a SCS Context
        /// </summary>
        /// <returns>IEReferralContext</returns>
        public static IEReferralContext CreateSCSContext()
        {
            return new Context();
        }

        /// <summary>
        /// Creates a SCS Content
        /// </summary>
        /// <returns>IEReferralContent</returns>
        public static IEReferralContent CreateSCSContent()
        {
            return new Content();
        }

        ///<summary>
        /// Creates a participation for a usual general practitioner
        ///</summary>
        ///<returns>(IParticipationUsualGP) Participation</returns>
        public static IParticipationUsualGP CreateUsualGP()
        {
            return new Participation();
        }

        ///<summary>
        /// Creates a participation for a referee
        ///</summary>
        ///<returns>(IParticipationReferee) Participation</returns>
        public static IParticipationReferee CreateReferee()
        {
            return new Participation();
        }

        ///<summary>
        /// Creates a participation for a usual general practitioner
        ///</summary>
        ///<returns>(IParticipationReferee) Participation</returns>
        public static IParticipationReferee CreateParticipantReferee()
        {
            return new Participation();
        }

        /// <summary>
        /// Creates a participation for patient nominated contact.
        /// </summary>
        /// <returns>IParticipationPatientNominatedContact</returns>
        public static IParticipationPatientNominatedContact CreateParticipationPatientNominatedContact()
        {
            return new Participation();
        }

        /// <summary>
        /// Creates the patient nominated contact.
        /// </summary>
        /// <returns></returns>
        public static IPatientNominatedContacts CreatePatientNominatedContact()
        {
            return new Participant();
        }

        /// <summary>
        /// Creates the person nominated contact.
        /// </summary>
        /// <returns></returns>
        public static IPersonWithRelationship CreatePersonPatientNominatedContacts()
        {
            return new Person();
        }

        /// <summary>
        /// Creates a participant constrained down to an usual general practitioner
        /// </summary>
        /// <returns>(IParticipant) Participant</returns>
        public static IUsualGP CreateParticipantForUsualGP()
        {
            return new Participant();    
        }

        /// <summary>
        /// Creates a participant constrained down to an referee
        /// </summary>
        /// <returns>(IParticipant) Participant</returns>
        public static IReferee CreateParticipantForReferee()
        {
            return new Participant();
        }

        /// <summary>
        /// Creates a medicine
        /// </summary>
        /// <returns>CodableText defining a medicine</returns>
        public static ICodableText CreateMedicine()
        {
            return new CodableText();
        }

        /// <summary>
        /// Creates a medicine
        /// </summary>
        /// <param name="code">Medicine code</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <param name="originalText">Original text, usually applicable in the absence of a code and display name</param>
        /// <param name="translations">Any translations that are associated with this medicine</param>
        /// <returns>CodableText defining a medicine</returns>
        public static ICodableText CreateMedicine(String code, CodingSystem? codeSystem, String displayName, String originalText, List<ICodableTranslation> translations)
        {
            return CreateCodableText(code, codeSystem, displayName, originalText, translations);
        }

        /// <summary>
        /// Creates a image view name
        /// 
        /// This method underpins all of the method that have a Codable text object as a return type
        /// </summary>
        /// <param name="code">code</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <param name="originalText">Original text, usually applicable in the absence of a code and display name</param>
        /// <param name="translations">Any translations that are associated with this codable text</param>
        /// <returns>CodableText</returns>
        public static ICodableText CreateImageViewName(String code, CodingSystem? codeSystem, String displayName, String originalText, List<ICodableTranslation> translations)
        {
            var codableText = new CodableText();

            if (codeSystem.HasValue)
            {
                codableText.DisplayName = displayName;
                codableText.Code = code;
                codableText.CodeSystemCode = codeSystem.Value.GetAttributeValue<NameAttribute, string>(a => a.Code);
                codableText.CodeSystemName = codeSystem.Value.GetAttributeValue<NameAttribute, string>(a => a.Name);
                codableText.CodeSystemVersion = codeSystem.Value.GetAttributeValue<NameAttribute, string>(a => a.Version);
            }

            if (!originalText.IsNullOrEmptyWhitespace())
                codableText.OriginalText = originalText;

            if (translations != null && translations.Any())
                codableText.Translations = translations;

            return codableText;
        }

        /// <summary>
        /// Creates an exclusion statement
        /// </summary>
        /// <returns>Statement</returns>
        public new static Statement CreateExclusionStatement()
        {
            return new Statement();
        }

        /// <summary>
        /// Creates a medications object constrained down to a IMedicationsEReferral
        /// </summary>
        /// <returns>(IMedicationsEReferral) Medications</returns>
        public static IMedicationsEReferral CreateMedications()
        {
            return new Medications();
        }

        /// <summary>
        /// Creates a medication object constrained down to a IMedication
        /// </summary>
        /// <returns>(IMedication) Medication</returns>
        public static IMedicationInstruction CreateMedication()
        {
            return new Medication();
        }

        /// <summary>
        /// Creates a procedure
        /// </summary>
        /// <returns>Procedure</returns>
        public static Procedure CreateProcedure()
        {
            return new Procedure();
        }

        /// <summary>
        /// Creates a procedure name
        /// </summary>
        /// <param name="code">Procedure code</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <param name="originalText">Original text, usually applicable in the absence of a code and display name</param>
        /// <param name="translations">Any translations that are associated with this procedure</param>
        /// <returns>CodableText defining a procedure</returns>
        public static ICodableText CreateProcedureName(String code, CodingSystem? codeSystem, String displayName, String originalText, List<ICodableTranslation> translations)
        {
            return CreateCodableText(code, codeSystem, displayName, originalText, translations);
        }

        /// <summary>
        /// Creates a medical history constrained down to a IMedicalHistoryWithoutExclusions
        /// </summary>
        /// <returns>(IMedicalHistoryWithoutExclusions) MedicalHistoryWithoutExclusions</returns>
        public static IMedicalHistoryWithoutExclusions CreateMedicalHistory()
        {
            return new MedicalHistory();
        }

        /// <summary>
        /// Creates a medical history item
        /// </summary>
        /// <returns>MedicalHistoryItemy</returns>
        public static IMedicalHistoryItem CreateMedicalHistoryItem()
        {
            return new MedicalHistoryItem();
        }

        /// <summary>
        /// Creates a problem diagnosis
        /// </summary>
        /// <returns>ProblemDiagnosis</returns>
        public static IProblemDiagnosis CreateProblemDiagnosis()
        {
            return new ProblemDiagnosis();
        }

        /// <summary>
        /// Creates a problem diagnosis identification
        /// </summary>
        /// <param name="code">Problem diagnosis identification code</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <param name="originalText">Original text, usually applicable in the absence of a code and display name</param>
        /// <returns>CodableText defining a problem diagnosis identification</returns>
        public static ICodableText CreateProblemDiagnosisIdentification(String code, CodingSystem? codeSystem, String displayName, String originalText)
        {
            var codableText = new CodableText();

            if (codeSystem.HasValue)
            {
                codableText.DisplayName = displayName;
                codableText.Code = code;
                codableText.CodeSystemCode = codeSystem.Value.GetAttributeValue<NameAttribute, string>(a => a.Code);
                codableText.CodeSystemName = codeSystem.Value.GetAttributeValue<NameAttribute, string>(a => a.Name);
                codableText.CodeSystemVersion = codeSystem.Value.GetAttributeValue<NameAttribute, string>(a => a.Version);
            }

            if (!originalText.IsNullOrEmptyWhitespace())
                codableText.OriginalText = originalText;

            return codableText;
        }

        /// <summary>
        /// Creates a substance or agent
        /// </summary>
        /// <param name="code">substance or agent code</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <param name="originalText">Original text, usually applicable in the absence of a code and display name</param>
        /// <param name="translations">Any translations that are associated with this substance or agent</param>
        /// <returns>CodableText defining a substance or agent</returns>
        public static ICodableText CreateSubstanceOrAgent(String code, CodingSystem? codeSystem, String displayName, String originalText, List<ICodableTranslation> translations)
        {
            return CreateCodableText(code, codeSystem, displayName, originalText, translations);
        }

        /// <summary>
        /// Creates a reaction
        /// </summary>
        /// <returns>Reaction</returns>
        public static Reaction CreateReaction()
        {
            return new Reaction();
        }

        /// <summary>
        /// Creates a reaction event
        /// </summary>
        /// <returns>ReactionEvent</returns>
        public static ReactionEvent CreateReactionEvent()
        {
            return new ReactionEvent();
        }

        /// <summary>
        /// Creates an adverseSubstance reactions object, constrained down to an IAdverseSubstanceReactions
        /// </summary>
        /// <returns>(IAdverseSubstanceReactions) AdverseSubstanceReactions </returns>
        public static IAdverseReactions CreateAdverseSubstanceReactions()
        {
            return new AdverseReactions();
        }

        /// <summary>
        /// Creates a diagnosticInvestigation
        /// </summary>
        /// <returns>DiagnosticInvestigations</returns>
        public static IDiagnosticInvestigations CreateDiagnosticInvestigations()
        {
            return new DiagnosticInvestigations();
        }

        /// <summary>
        /// Creates a collection procedure
        /// </summary>
        /// <param name="code">collection procedure code</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <param name="originalText">Original text, usually applicable in the absence of a code and display name</param>
        /// <param name="translations">Any translations that are associated with this collection procedure</param>
        /// <returns>CodableText defining a Anatomical Site</returns>
        public static ICodableText CreateCollectionProcedure(String code, CodingSystem? codeSystem, String displayName, String originalText, List<ICodableTranslation> translations)
        {
            return CreateCodableText(code, codeSystem, displayName, originalText, translations);
        }

        /// <summary>
        /// Creates a test result group
        /// </summary>
        /// <returns>(ITestResultGroup) ResultGroup</returns>
        public static ITestResultGroup CreateTestResultGroup()
        {
            return new ResultGroup();
        }

        /// <summary>
        /// Creates a sampling precondition
        /// </summary>
        /// <param name="code">sampling precondition code</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <param name="originalText">Original text, usually applicable in the absence of a code and display name</param>
        /// <param name="translations">Any translations that are associated with this sampling precondition</param>
        /// <returns>CodableText defining a sampling precondition</returns>
        public static ICodableText CreateSamplingPreconditions(String code, CodingSystem? codeSystem, String displayName, String originalText, List<ICodableTranslation> translations)
        {
            return CreateCodableText(code, codeSystem, displayName, originalText, translations);
        }

        /// <summary>
        /// Creates a specimen tissue type
        /// </summary>
        /// <param name="code">specimen tissue type code</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <param name="originalText">Original text, usually applicable in the absence of a code and display name</param>
        /// <param name="translations">Any translations that are associated with this specimen tissue type</param>
        /// <returns>CodableText defining a specimen tissue type</returns>
        public static ICodableText CreateSpecimenTissueType(String code, CodingSystem? codeSystem, String displayName, String originalText, List<ICodableTranslation> translations)
        {
            return CreateCodableText(code, codeSystem, displayName, originalText, translations);
        }

        /// <summary>
        /// Creates an anatomical location
        /// </summary>
        /// <returns>AnatomicalLocation</returns>
        public static AnatomicalLocation CreateAnatomicalLocation()
        {
            return new AnatomicalLocation();
        }
        
        /// <summary>
        /// Creates an anatomic location name
        /// </summary>
        /// <param name="code">anatomic location name code</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <param name="translations">Any translations that are associated with this anatomic location name</param>
        /// <returns>CodableText defining an anatomic location name</returns>
        public static ICodableText CreateAnatomicLocationName(String code, CodingSystem? codeSystem, String displayName, List<ICodableTranslation> translations)
        {
            return CreateCodableText(code, codeSystem, displayName, null, translations);
        }

        /// <summary>
        /// Creates an anatomic location side
        /// </summary>
        /// <param name="code">anatomic location side code</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <param name="originalText">Original text, usually applicable in the absence of a code and display name</param>
        /// <param name="translations">Any translations that are associated with this anatomic location side</param>
        /// <returns>CodableText defining an anatomic location side</returns>
        public static ICodableText CreateAnatomicLocationSide(String code, CodingSystem? codeSystem, String displayName, String originalText, List<ICodableTranslation> translations)
        {
            return CreateCodableText(code, codeSystem, displayName, originalText, translations);
        }

        /// <summary>
        /// Creates an imaging examination result
        /// </summary>
        /// <returns>ImagingExaminationResult</returns>
        public static ImagingExaminationResult CreateImagingExaminationResult()
        {
            return new ImagingExaminationResult();
        }

        /// <summary>
        /// Creates an examination result name
        /// </summary>
        /// <param name="code">examination result name code</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <param name="originalText">Original text, usually applicable in the absence of a code and display name</param>
        /// <param name="translations">Any translations that are associated with this examination result name</param>
        /// <returns>CodableText defining an examination result name</returns>
        public static ICodableText CreateExaminationResultName(String code, CodingSystem? codeSystem, String displayName, String originalText, List<ICodableTranslation> translations)
        {
            return CreateCodableText(code, codeSystem, displayName, originalText, translations);
        }

        /// <summary>
        /// Creates an imaging result
        /// </summary>
        /// <returns>(IImagingResult) Result</returns>
        public static IImagingResult CreateImagingResult()
        {
            return new Result();
        }

        /// <summary>
        /// Creates a modality
        /// </summary>
        /// <param name="code">modality code</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <param name="originalText">Original text, usually applicable in the absence of a code and display name</param>
        /// <param name="translations">Any translations that are associated with this modality</param>
        /// <returns>CodableText defining a modality</returns>
        public static ICodableText CreateModality(String code, CodingSystem? codeSystem, String displayName, String originalText, List<ICodableTranslation> translations)
        {
            return CreateCodableText(code, codeSystem, displayName, originalText, translations);
        }

        /// <summary>
        /// Creates a pathological diagnosis
        /// </summary>
        /// <param name="code">pathological diagnosis code</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <param name="originalText">Original text, usually applicable in the absence of a code and display name</param>
        /// <param name="translations">Any translations that are associated with this pathological diagnosis</param>
        /// <returns>CodableText defining a pathological diagnosis</returns>
        public static ICodableText CreatePathologicalDiagnois(String code, CodingSystem? codeSystem, String displayName, String originalText, List<ICodableTranslation> translations)
        {
            return CreateCodableText(code, codeSystem, displayName, originalText, translations);
        }

        /// <summary>
        /// Creates a test result
        /// </summary>
        /// <returns>(ITestResult) Result</returns>
        public static ITestResult CreateTestResult()
        {
            return new Result();
        }

        /// <summary>
        /// Creates a test request
        /// </summary>
        /// <returns>(ITestRequest) Request</returns>
        public static ITestRequest CreateTestRequest()
        {
            return new Request();
        }

        /// <summary>
        /// Creates a result name
        /// </summary>
        /// <param name="code">result name code</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <param name="originalText">Original text, usually applicable in the absence of a code and display name</param>
        /// <param name="translations">Any translations that are associated with this result name</param>
        /// <returns>CodableText defining a result name</returns>
        public static ICodableText CreateResultName(String code, CodingSystem? codeSystem, String displayName, String originalText, List<ICodableTranslation> translations)
        {
            return CreateCodableText(code, codeSystem, displayName, originalText, translations);
        }

        /// <summary>
        /// Creates a test result name
        /// </summary>
        /// <param name="code">result name</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <param name="originalText">Original text, usually applicable in the absence of a code and display name</param>
        /// <param name="translations">Any translations that are associated with this result name</param>
        /// <returns>CodableText defining a result name</returns>
        public static ICodableText CreateTestResultName(String code, CodingSystem? codeSystem, String displayName, String originalText, List<ICodableTranslation> translations)
        {
            return CreateCodableText(code, codeSystem, displayName, originalText, translations);
        }
        
        /// <summary>
        /// Creates a test result group name
        /// </summary>
        /// <param name="code">result name</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <param name="originalText">Original text, usually applicable in the absence of a code and display name</param>
        /// <param name="translations">Any translations that are associated with this result name</param>
        /// <returns>CodableText defining a result name</returns>
        public static ICodableText CreateTestResultGroupName(String code, CodingSystem? codeSystem, String displayName, String originalText, List<ICodableTranslation> translations)
        {
            return CreateCodableText(code, codeSystem, displayName, originalText, translations);
        }
        

        /// <summary>
        /// Creates a result value
        /// </summary>
        /// <returns>ResultValue</returns>
        public static ResultValue CreateResultValue()
        {
            return new ResultValue();
        }
        
        /// <summary>
        /// Creates a result value reference range detail
        /// </summary>
        /// <returns>ResultValueReferenceRangeDetail</returns>
        public static ResultValueReferenceRangeDetail CreateResultValueReferenceRangeDetail()
        {
            return new ResultValueReferenceRangeDetail();
        }


        /// <summary>
        /// Creates a result value reference range meaning
        /// </summary>
        /// <param name="code">result value reference range meaning code</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <param name="originalText">Original text, usually applicable in the absence of a code and display name</param>
        /// <param name="translations">Any translations that are associated with this result value reference range meaning</param>
        /// <returns>CodableText defining a result value reference range meaning</returns>
        public static ICodableText CreateResultValueReferenceRangeMeaning(String code, CodingSystem? codeSystem, String displayName, String originalText, List<ICodableTranslation> translations)
        {
            return CreateCodableText(code, codeSystem, displayName, originalText, translations);
        }

        /// <summary>
        /// Creates a manifestation
        /// </summary>
        /// <param name="code">role manifestation</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <param name="originalText">Original text, usually applicable in the absence of a code and display name</param>
        /// <param name="translations">Any translations that are associated with this manifestation</param>
        /// <returns>CodableText defining a manifestation</returns>
        public static ICodableText CreateManifestation(String code, CodingSystem? codeSystem, String displayName, String originalText, List<ICodableTranslation> translations)
        {
            return CreateCodableText(code, codeSystem, displayName, originalText, translations);
        }

        /// <summary>
        /// Creates an Image Details
        /// </summary>
        /// <returns>(IImageDetails) ImageDetails</returns>
        public static IImageDetails CreateImageDetails()
        {
            return new ImageDetails();
        }

        /// <summary>
        /// Creates an Image Result Group
        /// </summary>
        /// <returns>(IImagingResultGroup) ResultGroup</returns>
        public static IImagingResultGroup CreateImagingResultGroup()
        {
            return new ResultGroup();
        }

        /// <summary>
        /// Creates a test request name.
        /// </summary>
        /// <param name="code">Test request code</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <returns>CodableText defining a test request name</returns>
        public static ICodableText CreateTestRequestName(String code, CodingSystem? codeSystem, String displayName)
        {
            return CreateCodableText(code, codeSystem, displayName, null, null);
        }

        /// <summary>
        /// Creates a Requested Service Description
        /// </summary>
        /// <param name="code">type of Requested Service Description</param>
        /// <param name="codeSystem">The code system associated with the code</param>
        /// <param name="displayName">The display name associated with the code</param>
        /// <param name="originalText">Original text, usually applicable in the absence of a code and display name</param>
        /// <param name="translations">Any translations that are associated with this participation type</param>
        /// <returns>CodableText defining a Requested Service  Description</returns>
        public static ICodableText CreateRequestedServiceDescription(String code, CodingSystem? codeSystem, String displayName, String originalText, List<ICodableTranslation> translations)
        {
            return CreateCodableText(code, codeSystem, displayName, originalText, translations);
        }

        /// <summary>
        /// Creates a Requested Service
        /// </summary>
        /// <returns>RequestedService</returns>
        public static RequestedService CreateRequestedService()
        {
            return new RequestedService();
        }

        /// <summary>
        /// Create a Person for a Service Provider
        /// </summary>
        /// <returns>(IPerson) Person</returns>
        public static IPersonWithOrganisation CreatePersonForServiceProvider()
        {
            return new Person();
        }

        #endregion

        #region Serialisation Methods
        /// <summary>
        /// This method serializes this model into an XML document and returns this document
        /// </summary>
        /// <returns>XmlDocument</returns>
        public XmlDocument SerializeModel()
        {
            XmlDocument xmlDocument = null;
            var dataContractSerializer = new DataContractSerializer(typeof(EReferral));

            using (var memoryStream = new MemoryStream())
            {
                xmlDocument = new XmlDocument();

                dataContractSerializer.WriteObject(memoryStream, this);

                memoryStream.Seek(0, SeekOrigin.Begin);

                xmlDocument.Load(memoryStream);
            }

            return xmlDocument;
        }

        /// <summary>
        /// This method deserializes the xml document into an eReferral object
        /// </summary>
        /// <returns>XmlDocument</returns>
        public static EReferral DeserializeXmlDocument(XmlDocument xmlDocument)
        {
            EReferral eReferral = null;

            var dataContractSerializer = new DataContractSerializer(typeof(EReferral));

            using (var memoryStream = new MemoryStream())
            {
                xmlDocument.Save(memoryStream);

                memoryStream.Position = 0;

                eReferral = (EReferral)dataContractSerializer.ReadObject(memoryStream);
            }

            return eReferral;
        }
        #endregion
    }
}
