﻿/*
 * Copyright 2013 NEHTA
 *
 * Licensed under the NEHTA Open Source (Apache) License; you may not use this
 * file except in compliance with the License. A copy of the License is in the
 * 'license.txt' file, which should be provided with this work.
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Xml;
using Nehta.HL7.CDA;
using Nehta.VendorLibrary.CDA.Generator.Enums;
using Nehta.VendorLibrary.CDA.SCSModel.Common.Entities;
using Nehta.VendorLibrary.Common;
using Nehta.VendorLibrary.CDA.Generator;
using Nehta.VendorLibrary.CDA.Common.Enums;
using Nehta.VendorLibrary.CDA.Common;
using Nehta.VendorLibrary.CDA.SCSModel;
using Nehta.VendorLibrary.CDA.SCSModel.Common;

namespace Nehta.VendorLibrary.CDA.Sample
{
    /// <summary>
    /// This project is intended to demonstrate how an SpecialistLetter CDA document can be created.
    /// 
    /// The project contains two samples, the first is designed to create a fully populated CDA document, including
    /// all of the optional sections and entries. The second sample only populates the mandatory sections and entries.
    /// 
    /// The CDA model is split into three distinct sections, each of which needs to be created via the 
    /// SpecialistLetter class, and then populated with data as appropriate. The three sections that need to be
    /// created and hydrated with data are:
    /// 
    ///     CDA Context (Clinical Document Architecture - Context)
    ///     SCS Context (Structured Document Template - Context)
    ///     SCS Content (Structured Document Template - Content)
    /// 
    /// The CDA Context typically contains information that is to be represented within the header of the document
    /// that is not encapsulated with the SCS context.
    /// E.g. Generic CDA sections or entries; for example custodian.
    /// 
    /// The SCS Context typically contains information that is to be represented within the header of the document
    /// and relates specifically to the type of document that is to be created.
    /// E.g. E-Referral specific CDA sections or entries; for example Subject of Care.
    /// 
    /// The SCS Content typically contains information that is to be represented with the body of the document.
    /// </summary>
    public class SpecialistLetterSample
    {
        #region Properties

        public static string OutputFolderPath { get; set; }

        public static String OutputFileNameAndPath
        {
            get
            {
                return OutputFolderPath + @"\SpecialistLetter.xml";
            }
        }

        public static String ImageFileNameAndPath
        {
            get
            {
                return OutputFolderPath + @"\x-ray.jpg";
            }
        }

        public static String StructuredFileAttachment
        {
            get
            {
                return OutputFolderPath + @"\attachment.pdf";
            }
        }

        public static String ResultFileNameAndPath
        {
            get
            {
                return OutputFolderPath + @"\path1234.pdf";
            }
        }

        public static String PitNameAndPath
        {
          get
          {
            return OutputFolderPath + @"\PIT.txt";
          }
        }

        // Note: Place this in any string field and and this will insert a break
        private const String DELIMITERBREAK = "<BR>";

        #endregion

        /// <summary>
        /// This example populates only the mandatory Sections / Entries; as a result this sample omits all
        /// of the content within the body of the CDA document; as each of the sections within the body
        /// are optional.
        /// </summary>
        public XmlDocument PopulatedSpecialistLetterSample()
        {
            XmlDocument xmlDoc = null;

            var specialistLetter = PopulateSpecialistLetter(true);

            try
            {
                CDAGenerator.NarrativeGenerator = new CDANarrativeGenerator();

                //Pass the SpecialistLetter model into the GenerateSpecialistLetter method 
                xmlDoc = CDAGenerator.GenerateSpecialistLetter(specialistLetter);

                xmlDoc.Save(OutputFileNameAndPath);
            }
            catch (ValidationException ex)
            {
                //Catch any validation exceptions
                var validationMessages = ex.GetMessagesString();

                //Handle any validation errors as appropriate.
                throw;
            }

            return xmlDoc;
        }

        /// <summary>
        /// This example populates only the mandatory Sections / Entries; as a result this sample omits all
        /// of the content within the body of the CDA document; as each of the sections within the body
        /// are optional.
        /// </summary>
        public XmlDocument PopulateSpecialistLetterSample_1A(string fileName)
        {
            XmlDocument xmlDoc = null;

            var document = PopulateSpecialistLetter(true);
            document.SCSContent = SpecialistLetter.CreateSCSContent();

            document.IncludeLogo = false;

            var structuredBodyFileList = new List<ExternalData>();

            var structuredBodyFile = BaseCDAModel.CreateStructuredBodyFile();
            structuredBodyFile.Caption = "Structured Body File";
            structuredBodyFile.ExternalDataMediaType = MediaType.PDF;
            structuredBodyFile.Path = StructuredFileAttachment;
            structuredBodyFileList.Add(structuredBodyFile);

            document.SCSContent.StructuredBodyFiles = structuredBodyFileList;

            try
            {
                CDAGenerator.NarrativeGenerator = new CDANarrativeGenerator();

                //Pass the document model into the Generate method 
                xmlDoc = CDAGenerator.GenerateSpecialistLetter(document);

                if (!fileName.IsNullOrEmptyWhitespace())
                    xmlDoc.Save(XmlWriter.Create(OutputFolderPath + @"\" + fileName, new XmlWriterSettings() { Indent = true }));
            }
            catch (ValidationException ex)
            {
                //Catch any validation exceptions
                var validationMessages = ex.GetMessagesString();

                //Handle any validation errors as appropriate.
                throw;
            }

            return xmlDoc;
        }

        /// <summary>
        /// This sample populates only the mandatory Sections / Entries
        /// </summary>
        public XmlDocument MinPopulatedSpecialistLetterSample(string fileName)
        {
            XmlDocument xmlDoc = null;

            var specialistLetter = PopulateSpecialistLetter(true);

            try
            {
                CDAGenerator.NarrativeGenerator = new CDANarrativeGenerator();

                //Pass the E-Referral model into the GenerateSpecialistLetter method 
                xmlDoc = CDAGenerator.GenerateSpecialistLetter(specialistLetter);

                using (var writer = XmlWriter.Create(OutputFolderPath + @"\" + fileName, new XmlWriterSettings() { Indent = true }))
                {
                    if (!fileName.IsNullOrEmptyWhitespace()) xmlDoc.Save(writer);
                }
            }
            catch (ValidationException ex)
            {
                //Catch any validation exceptions
                var validationMessages = ex.GetMessagesString();

                //Handle any validation errors as appropriate.
                throw;
            }

            return xmlDoc;
        }
        
        /// <summary>
        /// This sample populates both the mandatory and optional Sections / Entries; as a result this sample
        /// includes all of the sections within the body and each section includes at least one example for 
        /// each of its optional entries.
        /// </summary>
        public XmlDocument MaxPopulatedSpecialistLetterSample(string fileName)
        {
            XmlDocument xmlDoc = null;

            var specialistLetter = PopulateSpecialistLetter(false);

            try
            {
                CDAGenerator.NarrativeGenerator = new CDANarrativeGenerator();

                //Pass the E-Referral model into the GenerateSpecialistLetter method 
                xmlDoc = CDAGenerator.GenerateSpecialistLetter(specialistLetter);

                using (var writer = XmlWriter.Create(OutputFolderPath + @"\" + fileName, new XmlWriterSettings() { Indent = true }))
                {
                    if (!fileName.IsNullOrEmptyWhitespace()) xmlDoc.Save(writer);
                }
            }
            catch (ValidationException ex)
            {
                //Catch any validation exceptions
                var validationMessages = ex.GetMessagesString();

                //Handle any validation errors as appropriate.
                throw;
            }

            return xmlDoc;
        }
        
        #region Populate Methods
        /// <summary>
        /// This method populates a specialistLetter
        /// </summary>
        /// <param name="mandatorySectionsOnly">mandatorySectionsOnly</param>
        /// <returns>SpecialistLetter</returns>
        internal static SpecialistLetter PopulateSpecialistLetter(Boolean mandatorySectionsOnly)
        {
            var specialistLetter = SpecialistLetter.CreateSpecialistLetter();

            // Set Creation Time
            specialistLetter.DocumentCreationTime = new ISO8601DateTime(DateTime.Now);

            // Include Logo
            specialistLetter.IncludeLogo = true;

            #region Setup and populate the CDA context model

            // Setup and populate the CDA context model
            var cdaContext = SpecialistLetter.CreateCDAContext();
            // Document Id
            cdaContext.DocumentId = BaseCDAModel.CreateIdentifier(BaseCDAModel.CreateOid(), null);
            // Set Id  
            cdaContext.SetId = BaseCDAModel.CreateIdentifier(BaseCDAModel.CreateGuid(), null);
            // CDA Context Version
            cdaContext.Version = "1";

            // Custodian
            cdaContext.Custodian = SpecialistLetter.CreateCustodian();
            GenericObjectReuseSample.HydrateCustodian(cdaContext.Custodian, mandatorySectionsOnly);

            // Legal authenticator
            if (!mandatorySectionsOnly)
            {
                cdaContext.LegalAuthenticator = SpecialistLetter.CreateLegalAuthenticator();
                GenericObjectReuseSample.HydrateAuthenticator(cdaContext.LegalAuthenticator, mandatorySectionsOnly);
            }

            // Create information recipient
            if (!mandatorySectionsOnly)
            {
                var recipient1 = SpecialistLetter.CreateInformationRecipient();
                var recipient2 = SpecialistLetter.CreateInformationRecipient();
                GenericObjectReuseSample.HydrateRecipient(recipient1, RecipientType.Primary, mandatorySectionsOnly);
                GenericObjectReuseSample.HydrateRecipient(recipient2, RecipientType.Secondary, mandatorySectionsOnly);
                cdaContext.InformationRecipients = new List<IParticipationInformationRecipient> { recipient1, recipient2 };
            }

            specialistLetter.CDAContext = cdaContext;

            #endregion

            #region Setup and Populate the SCS Context model

            specialistLetter.SCSContext = SpecialistLetter.CreateSCSContext();

            specialistLetter.SCSContext.Author = SpecialistLetter.CreateAuthor();
            GenericObjectReuseSample.HydrateAuthor(specialistLetter.SCSContext.Author, mandatorySectionsOnly);

            specialistLetter.SCSContext.SubjectOfCare = SpecialistLetter.CreateSubjectOfCare();
            GenericObjectReuseSample.HydrateSubjectofCare(specialistLetter.SCSContext.SubjectOfCare, mandatorySectionsOnly);

            specialistLetter.SCSContext.Referrer = CreateReferrer(mandatorySectionsOnly);

            specialistLetter.SCSContext.DateTimeSubjectSeen = new ISO8601DateTime(DateTime.Now);

            if (!mandatorySectionsOnly)
            {
              specialistLetter.SCSContext.UsualGP = CreateUsualGPPerson(mandatorySectionsOnly);
            }
            #endregion

            #region Setup and populate the SCS Content model

            specialistLetter.SCSContent = SpecialistLetter.CreateSCSContent();

            // Response details
            specialistLetter.SCSContent.ResponseDetails = CreateResponseDetails(mandatorySectionsOnly);

            // Recommendations
            specialistLetter.SCSContent.Recommendations = CreateRecommendations(mandatorySectionsOnly);

            // Medications
            specialistLetter.SCSContent.Medications = CreateMedications(mandatorySectionsOnly);

            if (!mandatorySectionsOnly)
            {
                // Adverse reactions
                specialistLetter.SCSContent.AdverseReactions = CreateAdverseReactions();

                // Diagnostic Investigations
                specialistLetter.SCSContent.DiagnosticInvestigations = CreateDiagnosticInvestigations(mandatorySectionsOnly);
            }
            #endregion

            return specialistLetter;
        }

        /// <summary>
        /// Creates and Hydrates the adverse substance reactions section for the Event Summary
        /// 
        /// Note: the data used within this method is intended as a guide and should be replaced.
        /// </summary>
        /// <returns>A Hydrated List of IAdverseReactionsEventSummay object</returns>
        private static IAdverseReactionsWithoutExclusions CreateAdverseReactions()
        {
            var reactions = SpecialistLetter.CreateAdverseReactionsWithoutExclusions();
            
            reactions.AdverseSubstanceReaction = new List<Reaction>
                                                 {
                                                     CreateAdverseReaction("90580008", "Fish"),
                                                     CreateAdverseReaction("256349002", "Peanuts")
                                                 };
            return reactions;
        }

        /// <summary>
        /// Creates an adverse reaction.
        /// </summary>
        /// <param name="code">Code for the adverse reaction.</param>
        /// <param name="name">Name of the adverse reaction.</param>
        /// <returns></returns>
        private static Reaction CreateAdverseReaction(string code, string name)
        {
            Reaction reaction = SpecialistLetter.CreateReaction();

            reaction.SubstanceOrAgent = SpecialistLetter.CreateSubstanceOrAgent
                (
                    code,
                    CodingSystem.SNOMED,
                    name,
                    name,
                    null
                );
            reaction.ReactionEvent = SpecialistLetter.CreateReactionEvent();
            reaction.ReactionEvent.Manifestations = new List<ICodableText>
            {
                SpecialistLetter.CreateManifestation
                (
                    "271807003",
                    CodingSystem.SNOMED,
                    "Skin Rash",
                    null,
                    null
                ),
                SpecialistLetter.CreateManifestation
                (
                    "422290000",
                    CodingSystem.SNOMED,
                    "Skin allergy",
                    null,
                    null
                )
            };

            return reaction;
        }

        /// <summary>
        /// Creates and Hydrates a UsualGP
        /// 
        /// Note: the data used within this method is intended as a guide and should be replaced.
        /// </summary>
        /// <returns>A Hydrated UsualGP object</returns>
        private static IParticipationUsualGP CreateUsualGPPerson(Boolean mandatorySectionsOnly)
        {
            var usualGP = SpecialistLetter.CreateUsualGP();

            var participant = SpecialistLetter.CreateParticipantForUsualGP();

            var personName = SpecialistLetter.CreatePersonName();
            personName.GivenNames = new List<string> { "Information (UsualGPPerson)" };
            personName.FamilyName = "Recipient";
            personName.Titles = new List<string> { "Doctor" };
            personName.NameUsages = new List<NameUsage> { NameUsage.Legal };

            var person = SpecialistLetter.CreatePersonWithOrganisation();
            person.Identifiers = new List<Identifier> { SpecialistLetter.CreateHealthIdentifier(HealthIdentifierType.HPII, "8003610000021101") };

            person.PersonNames = new List<IPersonName>();
            person.PersonNames.Add(personName);

            participant.Person = person;

            var organisation = SpecialistLetter.CreateEmploymentOrganisation();
            organisation.Name = "Bay hill hospital";
            organisation.NameUsage = OrganisationNameUsage.LocallyUsedName;

            participant.Person.Organisation = organisation;
            participant.Person.Organisation.Identifiers = new List<Identifier> { SpecialistLetter.CreateHealthIdentifier(HealthIdentifierType.HPIO, "8003620000020052") };

            var address = SpecialistLetter.CreateAddress();
            address.AddressPurpose = AddressPurpose.Business;
            address.AustralianAddress = SpecialistLetter.CreateAustralianAddress();
            address.AustralianAddress.UnstructuredAddressLines = new List<string> { "1 Clinician Street" };
            address.AustralianAddress.SuburbTownLocality = "Nehtaville";
            address.AustralianAddress.State = AustralianState.QLD;
            address.AustralianAddress.PostCode = "5555";
            address.AustralianAddress.DeliveryPointId = 32568931;

            var addressList = new List<IAddress>
                                {
                                    address
                                };
            participant.Addresses = addressList;

            var electronicCommunicationDetail = SpecialistLetter.CreateElectronicCommunicationDetail
                (
                   "0345754566",
                   ElectronicCommunicationMedium.Telephone,
                   ElectronicCommunicationUsage.WorkPlace
                );

            participant.ElectronicCommunicationDetails = new List<ElectronicCommunicationDetail> { electronicCommunicationDetail };

            if (!mandatorySectionsOnly)
            {
              usualGP.Role = SpecialistLetter.CreateRole(Occupation.GeneralMedicalPractitioner);
            }
            else
            {
              usualGP.Role = SpecialistLetter.CreateRole(NullFlavour.NoInformation);
            }

            usualGP.Participant = participant;

            return usualGP;
        }

        /// <summary>
        /// Creates and Hydrates a Referee
        /// 
        /// Note: the data used within this method is intended as a guide and should be replaced.
        /// </summary>
        /// <returns>A Hydrated Referee object</returns>
        private static IParticipationReferrer CreateReferrer(Boolean mandatorySectionsOnly)
        {
            var referrer = SpecialistLetter.CreateReferrer();

            referrer.Role = SpecialistLetter.CreateRole(Occupation.GeneralMedicalPractitioner);

            referrer.ParticipationPeriod = BaseCDAModel.CreateInterval(
                new ISO8601DateTime(DateTime.Now, ISO8601DateTime.Precision.Day), 
                new ISO8601DateTime(DateTime.Now.AddDays(200), ISO8601DateTime.Precision.Day));

            var participant = SpecialistLetter.CreateParticipantForReferrer();

            var personName = SpecialistLetter.CreatePersonName();
            personName.GivenNames = new List<string> { "Referrer" };
            personName.FamilyName = "1";
            personName.NameUsages = new List<NameUsage> { NameUsage.Legal };
            personName.Titles = new List<string> { "Doctor" };

            var person = SpecialistLetter.CreatePersonWithOrganisation();
            person.Identifiers = new List<Identifier> { SpecialistLetter.CreateHealthIdentifier(HealthIdentifierType.HPII, "8003610000021101") };

            person.PersonNames = new List<IPersonName>();
            person.PersonNames.Add(personName);

            participant.Person = person;

            var organisation = SpecialistLetter.CreateEmploymentOrganisation();
            organisation.Identifiers = new List<Identifier> { SpecialistLetter.CreateHealthIdentifier(HealthIdentifierType.HPIO, "8003620000020052") };
            organisation.Name = "Burrill Lake Medical Centre";
            organisation.NameUsage = OrganisationNameUsage.LocallyUsedName;

            person.Organisation = organisation;

            var address = SpecialistLetter.CreateAddress();
            address.AddressPurpose = AddressPurpose.Business;
            address.AustralianAddress = SpecialistLetter.CreateAustralianAddress();
            address.AustralianAddress.UnstructuredAddressLines = new List<string> { "1 Clinician Street" };
            address.AustralianAddress.SuburbTownLocality = "Nehtaville";
            address.AustralianAddress.State = AustralianState.QLD;
            address.AustralianAddress.PostCode = "5555";
            address.AustralianAddress.DeliveryPointId = 32568931;

            var addressList = new List<IAddress>
                                {
                                    address
                                };
            participant.Addresses = addressList;

            var electronicCommunicationDetail = SpecialistLetter.CreateElectronicCommunicationDetail
                (
                   "0345754566",
                   ElectronicCommunicationMedium.Telephone,
                   ElectronicCommunicationUsage.WorkPlace
                );

            participant.ElectronicCommunicationDetails = new List<ElectronicCommunicationDetail> { electronicCommunicationDetail };

            referrer.Participant = participant;

            return referrer;
        }

        /// <summary>
        /// Creates and Hydrates an natomicalSite
        /// </summary>
        /// <returns>AnatomicalSite</returns>
        private static AnatomicalSite CreateAnatomicalSite(String description, String name, String side, List<ExternalData> images)
        {
            var anatomicalSite = SpecialistLetter.CreateAnatomicalSite();
            anatomicalSite.Description = description;            
            anatomicalSite.AnatomicalLocation = SpecialistLetter.CreateAnatomicalLocation();
            anatomicalSite.AnatomicalLocation.NameOfLocation = SpecialistLetter.CreateAnatomicLocationName(null, null, null, name, null);
            anatomicalSite.AnatomicalLocation.Side = SpecialistLetter.CreateAnatomicLocationSide("7771000", CodingSystem.SNOMED, side, null, null);
            anatomicalSite.Images = images;            
            return anatomicalSite;
        }

        /// <summary>
        /// Creates and Hydrates a Reference Range
        /// </summary>
        /// <returns>ReferenceRange</returns>
        private static ResultValueReferenceRangeDetail CreateReferenceRange(String code, String name, String units, Double? high, Double? low)
        {
            var resultValueReferenceRangeDetail = SpecialistLetter.CreateResultValueReferenceRangeDetail();
            resultValueReferenceRangeDetail.Range = SpecialistLetter.CreateQuantityRange();
            resultValueReferenceRangeDetail.Range.High = high;
            resultValueReferenceRangeDetail.Range.Low = low;
            resultValueReferenceRangeDetail.Range.Units = units;
            resultValueReferenceRangeDetail.ResultValueReferenceRangeMeaning = SpecialistLetter.CreateResultValueReferenceRangeMeaning
                                                                                (
                                                                                    code,
                                                                                    CodingSystem.SNOMED,
                                                                                    name,
                                                                                    null,
                                                                                    null
                                                                                );

            return resultValueReferenceRangeDetail;
        }

        /// <summary>
        /// Creates and Hydrates a PhysicalDetails object
        /// </summary>
        /// <param name="value">The Quantity</param>
        /// <param name="units">The Unit</param>
        /// <returns>A Hydrated PhysicalDetails object</returns>
        private static PhysicalDetails CreatePhysicalDetails(string value, String units)
        {
          PhysicalDetails physicalDetails = null;

          if (!value.IsNullOrEmptyWhitespace() && !units.IsNullOrEmptyWhitespace())
          {
            physicalDetails = EventSummary.CreatePhysicalDetails();

            physicalDetails.Volume = EventSummary.CreateQuantity();
            physicalDetails.Volume.Value = value;
            physicalDetails.Volume.Units = units;
          }


          return physicalDetails;
        }


        /// <summary>
        /// Creates and Hydrates a Response Details object
        /// </summary>
        /// <returns>(IResponseDetails) Response</returns>
        private static IResponseDetails CreateResponseDetails(Boolean mandatorySectionsOnly)
        {
            var responseDetails = SpecialistLetter.CreateResponseDetails();

            if (!mandatorySectionsOnly)
            {
              responseDetails.Diagnoses = new List<ICodableText>
                {
                    SpecialistLetter.CreateDiagnosis("236629009", CodingSystem.SNOMED, "Chronic radiation cystitis", null, null),
                    SpecialistLetter.CreateDiagnosis("73781000", CodingSystem.SNOMED, "Chronic radiation dermatitis", null, null)
                };

              var procedure1 = SpecialistLetter.CreateProcedure();
              procedure1.ProcedureName = SpecialistLetter.CreateProcedureName("397956004", CodingSystem.SNOMED, "Hip replacement", null, null);
              var procedure2 = SpecialistLetter.CreateProcedure();
              procedure2.ProcedureName = SpecialistLetter.CreateProcedureName("30021000", CodingSystem.SNOMED, "Leg", null, null);
              responseDetails.Procedures = new List<IProcedureName>
                {
                    procedure1, procedure2
                };

              responseDetails.OtherDiagnosisEntries = new List<string>()
                {
                    "Text description of first Diagnosis Procedure Entry",
                    "Text description of second Diagnosis Procedure Entry"
                };
            }

            responseDetails.ResponseNarrative = "Response narrative";

            return responseDetails;
        }

        /// <summary>
        /// Creates and Hydrates a Recommendations object
        /// </summary>
        /// <returns>(IRecommendations) Recommendations</returns>
        private static IRecommendations CreateRecommendations(Boolean mandatorySectionsOnly)
        {
            var recommendations = SpecialistLetter.CreateRecommendations();

            if (!mandatorySectionsOnly)
            {
                var recomendationPerson = SpecialistLetter.CreateRecommendation();
                recomendationPerson.Narrative = "Recommendation Narrative Person";
                recomendationPerson.TimeFrame = SpecialistLetter.CreateInterval(
                    new ISO8601DateTime(DateTime.Now, ISO8601DateTime.Precision.Day), 
                    new ISO8601DateTime(DateTime.Now.AddMonths(4), ISO8601DateTime.Precision.Day));
                recomendationPerson.Addressee = CreateAddresseePerson(mandatorySectionsOnly);

                var recomendationOrganisation = SpecialistLetter.CreateRecommendation();
                recomendationOrganisation.Narrative = "Recommendation Narrative Organisation";
                recomendationOrganisation.TimeFrame = SpecialistLetter.CreateInterval("4", TimeUnitOfMeasure.Month);
                recomendationOrganisation.Addressee = CreateAddresseeOrganisation(mandatorySectionsOnly);

                recommendations.Recommendation = new List<Recommendation>
                                                 {
                                                     recomendationPerson,
                                                     recomendationOrganisation
                                                 };
            }
            else
            {
                recommendations.ExclusionStatement = "No Recommendations";
            }

            return recommendations;
        }

        /// <summary>
        /// Create Address Organisation
        /// </summary>
        /// <returns>(IParticipationAddressee) Addressee</returns>
        private static IParticipationAddressee CreateAddresseePerson(Boolean mandatorySectionsOnly)
        {
            var addressee = SpecialistLetter.CreateAddressee();
            addressee.Participant = SpecialistLetter.CreateParticipantAddressee();
   
            var personName = SpecialistLetter.CreatePersonName();
            personName.GivenNames = new List<string> { "Tom" };
            personName.FamilyName = "Smith";
            personName.Titles = new List<string> { "Dr" };
            personName.NameUsages = new List<NameUsage> { NameUsage.Legal };

            var person = SpecialistLetter.CreatePersonWithOrganisation();
            person.PersonNames = new List<IPersonName> { personName };
            
            addressee.Participant.Person = person;
            addressee.Participant.Person.Identifiers = new List<Identifier> { BaseCDAModel.CreateHealthIdentifier(HealthIdentifierType.HPII, "8003610000021101") }; 

            // Employment detail
            addressee.Participant.Person.Organisation = SpecialistLetter.CreateEmploymentOrganisation();
            addressee.Participant.Person.Organisation.Name = "Bay Hill Hospita";
            addressee.Participant.Person.Organisation.NameUsage = OrganisationNameUsage.Other;
            addressee.Participant.Person.Organisation.Identifiers = new List<Identifier> { BaseCDAModel.CreateHealthIdentifier(HealthIdentifierType.HPIO, "8003620000045562") };
            addressee.Participant.Person.Organisation.Department = "Some department person";
            addressee.Participant.Person.Organisation.EmploymentType = SpecialistLetter.CreateCodableText(null, null, null, "Person Casual", null);
            addressee.Participant.Person.Organisation.Occupation = Occupation.ChemicalEngineer;
            addressee.Participant.Person.Organisation.PositionInOrganisation = SpecialistLetter.CreateCodableText(null, null, null, "Person Manager", null);

            var address = SpecialistLetter.CreateAddress();
            address.AddressPurpose = AddressPurpose.Residential;
            address.AustralianAddress = SpecialistLetter.CreateAustralianAddress();
            address.AustralianAddress.PostCode = "4012";
            address.AustralianAddress.StreetName = "Johnson St";
            address.AustralianAddress.StreetNumber = 12;
            address.AustralianAddress.StreetSuffix = StreetSuffix.South;
            address.AustralianAddress.StreetType = StreetType.Street;

            addressee.Participant.Addresses = new List<IAddress>
                                                  {
                                                      address
                                                  };

            addressee.Participant.ElectronicCommunicationDetails = new List<ElectronicCommunicationDetail>
                                                                       {
                                                                           SpecialistLetter.
                                                                               CreateElectronicCommunicationDetail
                                                                               (
                                                                                   "134567891",
                                                                                   ElectronicCommunicationMedium.Telephone,
                                                                                   ElectronicCommunicationUsage.Home
                                                                               ),
                                                                           SpecialistLetter.
                                                                               CreateElectronicCommunicationDetail
                                                                               (
                                                                                   "Tom@optusnet.com.au",
                                                                                   ElectronicCommunicationMedium.Email,
                                                                                   ElectronicCommunicationUsage.Home
                                                                               )
                                                                       };

            if (!mandatorySectionsOnly)
            {
                 addressee.Role = SpecialistLetter.CreateRole(Occupation.GeneralMedicalPractitioner);
            } 
              else
            {
                 addressee.Role = SpecialistLetter.CreateRole(NullFlavour.NotApplicable);
            }

            return addressee;
        }

        /// <summary>
        /// Create Addressee Organisation
        /// </summary>
        /// <returns>(IParticipationAddressee) Addressee</returns>
        private static IParticipationAddressee CreateAddresseeOrganisation(Boolean mandatorySectionsOnly)
        {
            var addressee = SpecialistLetter.CreateAddressee();
            addressee.Participant = SpecialistLetter.CreateParticipantAddressee();

            addressee.Participant.Organisation = SpecialistLetter.CreateOrganisation();
            addressee.Participant.Organisation.Name = "Bay Hill Hospital";
            addressee.Participant.Organisation.NameUsage = OrganisationNameUsage.Other;
            addressee.Participant.Organisation.Identifiers = new List<Identifier> { BaseCDAModel.CreateHealthIdentifier(HealthIdentifierType.HPIO, "8003620000045562") };
            addressee.Participant.Organisation.Department = "Some department organisation";

            var address = SpecialistLetter.CreateAddress();
            address.AddressPurpose = AddressPurpose.Residential;
            address.AustralianAddress = SpecialistLetter.CreateAustralianAddress();
            address.AustralianAddress.PostCode = "4012";
            address.AustralianAddress.StreetName = "Johnson St";
            address.AustralianAddress.StreetNumber = 12;
            address.AustralianAddress.StreetSuffix = StreetSuffix.South;
            address.AustralianAddress.StreetType = StreetType.Street;

            addressee.Participant.Addresses = new List<IAddress>
                                                  {
                                                      address
                                                  };

            addressee.Participant.ElectronicCommunicationDetails = new List<ElectronicCommunicationDetail>
                                                                       {
                                                                           SpecialistLetter.
                                                                               CreateElectronicCommunicationDetail
                                                                               (
                                                                                   "dgdfg@optusnet.com.au",
                                                                                   ElectronicCommunicationMedium.Email,
                                                                                   ElectronicCommunicationUsage.Home
                                                                               ),
                                                                           SpecialistLetter.
                                                                               CreateElectronicCommunicationDetail
                                                                               (
                                                                                   "134567891",
                                                                                   ElectronicCommunicationMedium.Telephone,
                                                                                   new List<ElectronicCommunicationUsage> { ElectronicCommunicationUsage.PrimaryHome, ElectronicCommunicationUsage.MobileContact}
                                                                               ),
                                                                           SpecialistLetter.
                                                                               CreateElectronicCommunicationDetail
                                                                               (
                                                                                   "675675675676",
                                                                                   ElectronicCommunicationMedium.Telephone,
                                                                                   new List<ElectronicCommunicationUsage> { ElectronicCommunicationUsage.MobileContact}
                                                                               ),

                                                                       };

            if (!mandatorySectionsOnly)
            {
              addressee.Role = SpecialistLetter.CreateRole(Occupation.GeneralMedicalPractitioner);
            }
              else
            {
              addressee.Role = SpecialistLetter.CreateRole(Occupation.FreightHandlerRailOrRoad);
            }

            return addressee;
        }    

        /// <summary>
        /// Creates and Hydrates the Medications section for the E-Referral.
        /// 
        /// Note: the data used within this method is intended as a guide and should be replaced.
        /// </summary>
        /// <returns>A Hydrated Medications object</returns>
        private static IMedicationsSpecialistLetter CreateMedications(Boolean mandatorySectionsOnly)
        {
            var medications = SpecialistLetter.CreateMedications();

            if (!mandatorySectionsOnly)
            {
                var medicationList = new List<IMedicationItem>();

                var medication1 = SpecialistLetter.CreateMedication();
                medication1.Directions = "As directed by packaging.";

                medication1.RecommendationOrChange = RecomendationOrChange.TheChangeHasBeenMade;
                medication1.ChangeType = ChangeType.Suspended;
                medication1.ClinicalIndication = "Clinical indication";
                medication1.Comment = "Some comment";
                medication1.ChangeDescription = "Change description";
                medication1.ChangeReason = "Change reason";
                medication1.Medicine = SpecialistLetter.CreateMedicine
                    (
                        "23641011000036102",
                        CodingSystem.AMT,
                        "paracetamol 500 mg + codeine phosphate 30 mg tablet",
                        null,
                        null
                    );
                medicationList.Add(medication1);

                var medication2 = SpecialistLetter.CreateMedication();
                medication2.Directions = "As directed by packaging.";
                medication2.RecommendationOrChangeNullFlavour = NullFlavour.NegativeInfinity;
                medication2.ChangeTypeNullFlavour = NullFlavour.NoInformation;
                medication2.ClinicalIndication = "Clinical indication";
                medication2.Comment = "Some comment";
                medication2.ChangeDescription = "Change description";
                medication2.ChangeReason = "Change reason";
                medication2.Medicine = SpecialistLetter.CreateMedicine
                    (
                        "22589011000036109",
                        CodingSystem.AMT,
                        "paracetamol 240 mg/5 mL oral liquid",
                        null,
                        null
                    );
                medicationList.Add(medication2);

                medications.MedicationsList = medicationList;
            } 
            else
            {
                medications.ExclusionStatement = SpecialistLetter.CreateExclusionStatement();
                medications.ExclusionStatement.Value = NCTISGlobalStatementValues.NoneKnown;
            }

            return medications;
        }

        /// <summary>
        /// Creates a list of requested services.
        /// </summary>
        /// <returns>List of requested services.</returns>
        private static List<RequestedService> CreateRequestedService(Boolean mandatorySectionsOnly)
        {
            var requestedServiceList = new List<RequestedService>();

            // Create Service Provider for a Person
            var requestedServicePerson = SpecialistLetter.CreateRequestedService();
            requestedServicePerson.ServiceScheduled = new ISO8601DateTime(DateTime.Now);

            requestedServicePerson.RequestedServiceDescription = SpecialistLetter.CreateRequestedServiceDescription("399208008", CodingSystem.SNOMED, "chest x-ray", null, null);
            requestedServicePerson.ServiceBookingStatus = EventTypes.Definition;
            // Create Duration
            requestedServicePerson.SubjectOfCareInstructionDescription = "Subject Of Care Instruction Description";
            requestedServicePerson.RequestedServiceDateTime = new ISO8601DateTime(DateTime.Now.AddDays(4));
            // Create Person
            requestedServicePerson.ServiceProvider = CreateServiceProviderPerson(mandatorySectionsOnly);

            // Add to list
            requestedServiceList.Add(requestedServicePerson);

            // Create Service Provider for a Organisation
            var requestedServiceOrganisation = SpecialistLetter.CreateRequestedService();
            requestedServiceOrganisation.RequestedServiceDescription = SpecialistLetter.CreateRequestedServiceDescription("399208008", CodingSystem.SNOMED, "chest x-ray", null, null);
            requestedServiceOrganisation.ServiceBookingStatus = EventTypes.Intent;

            requestedServiceOrganisation.ServiceCommencementWindow = EventSummary.CreateInterval(
                new ISO8601DateTime(DateTime.Now, ISO8601DateTime.Precision.Day),
                new ISO8601DateTime(DateTime.Now.AddMonths(6), ISO8601DateTime.Precision.Day));
            requestedServiceOrganisation.ServiceScheduled = new ISO8601DateTime(DateTime.Now);

            requestedServiceOrganisation.SubjectOfCareInstructionDescription = "Subject Of Care Instruction Description";
            requestedServiceOrganisation.RequestedServiceDateTime = new ISO8601DateTime(DateTime.Now.AddDays(4));
            requestedServiceOrganisation.ServiceProvider = CreateServiceProviderOrganisation(mandatorySectionsOnly);

            // Add to list
            requestedServiceList.Add(requestedServiceOrganisation);

            return requestedServiceList;
        }

        /// <summary>
        /// Creates and Hydrates a Service Provider Person
        /// 
        /// Note: the data used within this method is intended as a guide and should be replaced.
        /// </summary>
        /// <returns>A Hydrated IParticipationHealthProfessional object</returns>
        private static IParticipationServiceProvider CreateServiceProviderPerson(Boolean mandatorySectionsOnly)
        {
            var serviceProvider = SpecialistLetter.CreateServiceProvider();

            var participant = SpecialistLetter.CreateParticipantForServiceProvider();
            participant.Person = SpecialistLetter.CreatePersonForServiceProvider();

            var personName = BaseCDAModel.CreatePersonName();
            personName.FamilyName = "Dr Jane Anderson";
            personName.NameUsages = new List<NameUsage> { NameUsage.Legal };

            participant.Person.PersonNames = new List<IPersonName> { personName };
            participant.Person.Identifiers = new List<Identifier> { SpecialistLetter.CreateHealthIdentifier(HealthIdentifierType.HPII, "8003610000001145") };

            participant.Person.Organisation = BaseCDAModel.CreateEmploymentOrganisation();
            participant.Person.Organisation.Name = "Bay Hill Hospital";
            participant.Person.Organisation.NameUsage = OrganisationNameUsage.Other;
            participant.Person.Organisation.Identifiers = new List<Identifier> { SpecialistLetter.CreateHealthIdentifier(HealthIdentifierType.HPIO, "8003620000045562") };
            participant.Person.Organisation.EmploymentType = SpecialistLetter.CreateCodableText(null, null, null, "Casual", null);
            participant.Person.Organisation.Occupation = Occupation.ChemicalEngineer;
            participant.Person.Organisation.PositionInOrganisation = SpecialistLetter.CreateCodableText(null, null, null, "Manager", null);

            var electronicCommunicationDetail = SpecialistLetter.CreateElectronicCommunicationDetail(
                "0345754566",
                ElectronicCommunicationMedium.Telephone,
                ElectronicCommunicationUsage.WorkPlace);

            var electronicCommunicationDetailEmail = SpecialistLetter.CreateElectronicCommunicationDetail(
                "Jane@Anderson.com.au",
                ElectronicCommunicationMedium.Email,
                ElectronicCommunicationUsage.WorkPlace);

            participant.ElectronicCommunicationDetails = new List<ElectronicCommunicationDetail> { electronicCommunicationDetail, electronicCommunicationDetailEmail };

            var address = SpecialistLetter.CreateAddress();
            address.AddressPurpose = AddressPurpose.Business;
            address.AustralianAddress = SpecialistLetter.CreateAustralianAddress();
            address.AustralianAddress.UnstructuredAddressLines = new List<string> { "1 Clinician Street" };
            address.AustralianAddress.SuburbTownLocality = "Nehtaville";
            address.AustralianAddress.State = AustralianState.QLD;
            address.AustralianAddress.PostCode = "5555";
            address.AustralianAddress.DeliveryPointId = 32568931;

            participant.Addresses = new List<IAddress>
            {
                address, address
            };

            if (!mandatorySectionsOnly)
            {
              serviceProvider.Role = SpecialistLetter.CreateRole(Occupation.GeneralMedicalPractitioner);
            } 
              else
            {
              serviceProvider.Role = SpecialistLetter.CreateRole(Occupation.FreightHandlerRailOrRoad);
            }

            serviceProvider.Participant = participant;

            return serviceProvider;
        }

        /// <summary>
        /// Creates and Hydrates a Service Provider Organisation
        /// 
        /// Note: the data used within this method is intended as a guide and should be replaced.
        /// </summary>
        /// <returns>A Hydrated IParticipationHealthProfessional object</returns>
        private static IParticipationServiceProvider CreateServiceProviderOrganisation(Boolean mandatorySectionsOnly)
        {
            var serviceProvider = SpecialistLetter.CreateServiceProvider();

            serviceProvider.Participant = SpecialistLetter.CreateParticipantForServiceProvider();

            serviceProvider.Participant.Organisation = SpecialistLetter.CreateOrganisation();
            serviceProvider.Participant.Organisation.Name = "Bay Hill Hospital";
            serviceProvider.Participant.Organisation.NameUsage = OrganisationNameUsage.Other;
            serviceProvider.Participant.Organisation.Identifiers = new List<Identifier> { SpecialistLetter.CreateHealthIdentifier(HealthIdentifierType.HPIO, "8003620000045562") };

            var electronicCommunicationDetail = SpecialistLetter.CreateElectronicCommunicationDetail(
                "0345754566",
                ElectronicCommunicationMedium.Telephone,
                ElectronicCommunicationUsage.WorkPlace);

            serviceProvider.Participant.ElectronicCommunicationDetails = new List<ElectronicCommunicationDetail> { electronicCommunicationDetail };

            var address = SpecialistLetter.CreateAddress();
            address.AddressPurpose = AddressPurpose.Residential;
            address.AustralianAddress = SpecialistLetter.CreateAustralianAddress();
            address.AustralianAddress.UnstructuredAddressLines = new List<string> { "1 Clinician Street" };
            address.AustralianAddress.SuburbTownLocality = "Nehtaville";
            address.AustralianAddress.State = AustralianState.QLD;
            address.AustralianAddress.PostCode = "5555";
            address.AustralianAddress.DeliveryPointId = 32568931;

            if (!mandatorySectionsOnly)
            {
              serviceProvider.Role = SpecialistLetter.CreateRole(Occupation.GeneralMedicalPractitioner);
            }
            else
            {
              serviceProvider.Role = SpecialistLetter.CreateRole(NullFlavour.NotAsked);
            }

            serviceProvider.Participant.Addresses = new List<IAddress> 
            { 
                address 
            };

            return serviceProvider;
        }

        /// <summary>
        /// Creates and hydrates the diagnostic investigations section.
        /// 
        /// Note: the data used within this method is intended as a guide and should be replaced.
        /// </summary>
        /// <returns>A hydrated 'DiagnosticInvestigations' object.</returns>
        private static IDiagnosticInvestigations CreateDiagnosticInvestigations(Boolean mandatorySectionsOnly)
        {
            IDiagnosticInvestigations diagnosticInvestigations = SpecialistLetter.CreateDiagnosticInvestigations();

            diagnosticInvestigations.ImagingExaminationResult = new List<IImagingExaminationResult>
                                                                {
                                                                    CreateImagingResults("Leg Image"),
                                                                    CreateImagingResults("Head Image")
                                                                };
            

            // Pathology test results
            diagnosticInvestigations.PathologyTestResult = new List<PathologyTestResult> 
                                                           {
                                                                CreatePathologyResults("Chemisty studies (set)"),
                                                                CreatePathologyResults("Blood studies (set)")
                                                           };

            // Other Test Result 
            diagnosticInvestigations.OtherTestResult = new List<OtherTestResult>
                                                       {
                                                              CreateOtherTestResultAttachment(),
                                                              CreateOtherTestResultText()
                                                       };



            // Requested Service
            diagnosticInvestigations.RequestedService = CreateRequestedService(mandatorySectionsOnly);

            return diagnosticInvestigations;
        }

        /// <summary>
        /// Creates and hydrates the 'Other Test Result' Attachment section.
        /// </summary>
        /// <returns>A hydrated 'OtherTestResult' object.</returns>
        private static OtherTestResult CreateOtherTestResultAttachment()
        {
          var otherTestResult = BaseCDAModel.CreateOtherTestResult();
          otherTestResult.ReportName = BaseCDAModel.CreateCodableText(null, null, null, "Report with Attachment", null);
          otherTestResult.ReportStatus = ResultStatus.Interim;

          // Report ExternalData
          ExternalData report = BaseCDAModel.CreateExternalData();
          report.ExternalDataMediaType = MediaType.PDF;
          report.Path = ResultFileNameAndPath;
          report.Caption = "Path File";

          otherTestResult.ReportContent = BaseCDAModel.CreateEncapsulatedData();
          otherTestResult.ReportContent.ExternalData = report;
          otherTestResult.ReportDate = new ISO8601DateTime(DateTime.Now.AddDays(-2));

          return otherTestResult;
        }

        /// <summary>
        /// Creates and hydrates the 'Other Test Result' text section.
        /// </summary>
        /// <returns>A hydrated 'OtherTestResult' object.</returns>
        private static OtherTestResult CreateOtherTestResultText()
        {
          var otherTestResult = BaseCDAModel.CreateOtherTestResult();
          otherTestResult.ReportName = BaseCDAModel.CreateCodableText(null, null, null, "Report Name", null);
          otherTestResult.ReportStatus = ResultStatus.Interim;

          otherTestResult.ReportContent = BaseCDAModel.CreateEncapsulatedData();
          otherTestResult.ReportContent.Text = "Report Content Text";
          otherTestResult.ReportDate = new ISO8601DateTime(DateTime.Now.AddDays(-2));

          return otherTestResult;
        }

        /// <summary>
        /// Creates and hydrates the 'Imaging Examination Results' section.
        /// </summary>
        /// <param name="name">Name of the imaging examination results.</param>
        /// <returns>A hydrated 'IImagingExaminationResultDischargeSummary' object.</returns>
        private static IImagingExaminationResult CreateImagingResults(string name)
        {
            // Image 1
            ExternalData image1 = SpecialistLetter.CreateExternalData();
            image1.ExternalDataMediaType = MediaType.JPEG;
            image1.Path = ImageFileNameAndPath;
            image1.Caption = "X-Ray";

            // Image 2
            ExternalData image2 = SpecialistLetter.CreateExternalData();
            image2.ExternalDataMediaType = MediaType.JPEG;
            image2.Path = ImageFileNameAndPath;
            image2.Caption = "X-Ray - Left Femur";

            // Image 3
            ExternalData image3 = SpecialistLetter.CreateExternalData();
            image3.ExternalDataMediaType = MediaType.JPEG;
            image3.Path = ImageFileNameAndPath;
            image3.Caption = "X-Ray - Right Arm";

            // Anatomical site 1
            ICodableText nameOfLocation = BaseCDAModel.CreateCodableText("71341001", CodingSystem.SNOMED, "Femur", null, null);
            ICodableText side = BaseCDAModel.CreateCodableText("7771000", CodingSystem.SNOMED, "Left", null, null);
            AnatomicalSite anatomicalSite1 = CreateAnatomicalSite(
                "Femur Length",
                nameOfLocation,
                side,
                new List<ExternalData> { image1, image2 }
                );

            // Anatomical site 2
            AnatomicalSite anatomicalSite2 = CreateAnatomicalSite(
                "Arm Length",
                "182245002",
                CodingSystem.SNOMED,
                "Arm",
                "Left",
                new List<ExternalData> { image3 }
                );

            // Imaging results 1
            IImagingResult imagingResult1 = SpecialistLetter.CreateImagingResult();            
            imagingResult1.Comments = new List<string> { "Femur measured during ultrasound scan.", "Legs measured during ultrasound scan." };
            imagingResult1.ResultName = SpecialistLetter.CreateResultName("16310003", CodingSystem.SNOMED, "ultrasound scan", null, null);
            imagingResult1.ResultValue = SpecialistLetter.CreateResultValue();
            imagingResult1.ResultValue.ValueAsCodableText = BaseCDAModel.CreateCodableText("371573008", CodingSystem.SNOMED, "Ultrasound", null, null);
            imagingResult1.ResultValueStatus = HL7ObservationInterpretationNormality.Normal;
            imagingResult1.ResultValueReferenceRangeDetails = new List<ResultValueReferenceRangeDetail>
                                                                  {
                                                                      CreateReferenceRange("260395002", "Normal range", "ml", 17, 13), 
                                                                      CreateReferenceRange("75540009", "High", "ml", 50, 100)
                                                                  };
            // Imaging results 2
            IImagingResult imagingResult2 = SpecialistLetter.CreateImagingResult();
            imagingResult2.Comments = new List<string> { "Neck measured during ultrasound scan.", "Neck measured during ultrasound scan." };
            imagingResult2.ResultName = SpecialistLetter.CreateResultName("16310040", CodingSystem.SNOMED, "neck ultrasound scan", null, null);
            imagingResult2.ResultValue = SpecialistLetter.CreateResultValue();
            imagingResult2.ResultValue.ValueAsCodableText = SpecialistLetter.CreateCodableText("77777", CodingSystem.SNOMED, "x-ray description", null, null);
            imagingResult2.ResultValueStatus = HL7ObservationInterpretationNormality.Normal;
            imagingResult2.ResultValueReferenceRangeDetails = new List<ResultValueReferenceRangeDetail>
                                                                  {
                                                                      CreateReferenceRange("260395002", "Normal range", "kg", 15, 18), 
                                                                      CreateReferenceRange("75540009", "High", "kg", 60, 110)
                                                                  };

            // Imaging results 3
            IImagingResult imagingResult3 = SpecialistLetter.CreateImagingResult();
            imagingResult3.Comments = new List<string> { "Waist measured during ultrasound scan.", "Waist measured during ultrasound scan." };
            imagingResult3.ResultName = SpecialistLetter.CreateResultName("371573008", CodingSystem.SNOMED, "Ultrasound", null, null);
            imagingResult3.ResultValue = SpecialistLetter.CreateResultValue();
            imagingResult3.ResultValue.ValueAsCodableText = SpecialistLetter.CreateCodableText("371573008", CodingSystem.SNOMED, "Ultrasound", null, null);
            imagingResult3.ResultValueStatus = HL7ObservationInterpretationNormality.Normal;
            imagingResult3.ResultValueReferenceRangeDetails = new List<ResultValueReferenceRangeDetail>
                                                                  {
                                                                      CreateReferenceRange("260395002", "Normal range", "ml", 15, 18), 
                                                                      CreateReferenceRange("75540009", "High", "ml", 60, 110)
                                                                  };


            // Image 4
            ExternalData image4 = SpecialistLetter.CreateExternalData();
            image4.ExternalDataMediaType = MediaType.JPEG;
            image4.Path = ImageFileNameAndPath;
            image4.Caption = "4 X-Ray";

            // Image details 1
            IImageDetails imageDetails1 = SpecialistLetter.CreateImageDetails();
            imageDetails1.DateTime = new ISO8601DateTime(DateTime.Now);
            imageDetails1.SubjectPosition = "Supine";
            imageDetails1.SeriesIdentifier = BaseCDAModel.CreateInstanceIdentifier("1.2.3.1.2.2654654654654564", "Series Identifier");
            imageDetails1.Image = image4;

            // Image 5
            ExternalData image5 = SpecialistLetter.CreateExternalData();
            image5.ExternalDataMediaType = MediaType.JPEG;
            image5.Path = ImageFileNameAndPath;
            image5.Caption = "5 X-Ray";

            // Image details 2
            IImageDetails imageDetails2 = SpecialistLetter.CreateImageDetails();
            imageDetails2.DateTime = new ISO8601DateTime(DateTime.Now);
            imageDetails2.SubjectPosition = "Sublime";
            imageDetails2.SeriesIdentifier = BaseCDAModel.CreateInstanceIdentifier("1.2.3.1.2.2654654654654564", "Series Identifier");
            imageDetails2.Image = image5;

            // Image 6
            ExternalData image6 = SpecialistLetter.CreateExternalData();
            image6.ExternalDataMediaType = MediaType.JPEG;
            image6.Path = ImageFileNameAndPath;
            image6.Caption = "6 X-Ray";

            // Image details 3
            IImageDetails imageDetails3 = SpecialistLetter.CreateImageDetails();
            imageDetails3.DateTime = new ISO8601DateTime(DateTime.Now);
            imageDetails3.SubjectPosition = "Head";
            imageDetails3.SeriesIdentifier = BaseCDAModel.CreateInstanceIdentifier("1.2.3.1.2.2654654654654564", "Series Identifier");
            imageDetails3.Image = image6;

            // Imaging result group 1
            IImagingResultGroup imagingResultGroup1 = SpecialistLetter.CreateImagingResultGroup();
            imagingResultGroup1.Results = new List<IImagingResult>
                                              {
                                                  imagingResult1 
                                              };
            imagingResultGroup1.AnatomicalSite = anatomicalSite1;
            imagingResultGroup1.ResultGroupName = SpecialistLetter.CreateResultGroupName
                (
                    "268445003",
                    CodingSystem.SNOMED,
                    "ultrasound scan - obstetric",
                    null,
                    null
                );

            // Imaging result group 2
            IImagingResultGroup imagingResultGroup2 = SpecialistLetter.CreateImagingResultGroup();
            imagingResultGroup2.Results = new List<IImagingResult>
                                              {
                                                  imagingResult3
                                              };

            imagingResultGroup2.AnatomicalSite = anatomicalSite2;
            imagingResultGroup2.ResultGroupName = SpecialistLetter.CreateResultGroupName
                (
                    "371573008",
                    CodingSystem.SNOMED,
                    "Ultrasound",
                    null,
                    null
                );

            // Anatomical site 3
            AnatomicalSite anatomicalSite3 = CreateAnatomicalSite(
                "Rotia vossa",
                "72098002",
                CodingSystem.SNOMED,
                "Left arm",
                "Left",
                null
                );

            // Imaging examination result 1
            ImagingExaminationResult imagingExaminationResult = SpecialistLetter.CreateImagingExaminationResult();
            imagingExaminationResult.AnatomicalSite = new List<AnatomicalSite>() { anatomicalSite3 };
            imagingExaminationResult.ClinicalInformationProvided = "Patient pregnant. Confirm dates. Estimate from LNMP 18 weeks.";
            imagingExaminationResult.ExaminationRequestDetails = new List<IImagingExaminationRequest>
                                                                     {
                                                                         CreateImagingExaminationRequest("A/U Obstetric - Dating", imageDetails1),
                                                                         CreateImagingExaminationRequest("A/U Skin - Stuff", new List<IImageDetails> { imageDetails2, imageDetails3 })
                                                                     };
            imagingExaminationResult.ExaminationResultName = SpecialistLetter.CreateExaminationResultName("1234", CodingSystem.METEOR, name, name, null);
            imagingExaminationResult.Findings = "Normal obstetric ultrasound with foetal biometry indicating getational age of 18W0D.";
            imagingExaminationResult.Modality = SpecialistLetter.CreateModality("16310003", CodingSystem.SNOMED, "ultrasound", null, null);
            imagingExaminationResult.ResultDateTime = new ISO8601DateTime(DateTime.Now.AddDays(-2));
            imagingExaminationResult.ExaminationResultStatus = ResultStatus.Final;
            imagingExaminationResult.ExaminationResultRepresentation = "ResultRepresentation";
            imagingExaminationResult.ResultGroup = new List<IImagingResultGroup>
                                                       {
                                                           imagingResultGroup1, imagingResultGroup2
                                                       };
            
            return imagingExaminationResult;
        }

        /// <summary>
        /// Creates and hydrates the 'Pathology Test Results' section.
        /// </summary>
        /// <param name="name">Name of the pathology test result.</param>
        /// <returns>A hydrated 'PathologyTestResult' object.</returns>
        private static PathologyTestResult CreatePathologyResults(string name)
        {
            // Image 1
            ExternalData image1 = SpecialistLetter.CreateExternalData();
            image1.ExternalDataMediaType = MediaType.JPEG;
            image1.Path = ImageFileNameAndPath;
            image1.Caption = "X-Ray 5";

            // Image 1
            ExternalData image2 = SpecialistLetter.CreateExternalData();
            image2.ExternalDataMediaType = MediaType.JPEG;
            image2.Path = ImageFileNameAndPath;
            image2.Caption = "X-Ray 5";

            // Anatomical site 1
            AnatomicalSite anatomicalSite1 = CreateAnatomicalSite(
                "Femur Length",
                "71341001",
                CodingSystem.SNOMED,
                "Femur",
                "Left",
                new List<ExternalData> { image1 }
                );

            // Anatomical site 2
            AnatomicalSite anatomicalSite2 = CreateAnatomicalSite(
                "Cubital fossa",
                "71341001",
                CodingSystem.SNOMED,
                "Femur",
                "Left",
                null
                );

            // Anatomical site 3
            AnatomicalSite anatomicalSite3 = CreateAnatomicalSite(
                "Rotia vossa",
                "72098002",
                CodingSystem.SNOMED,
                "Left arm",
                "Left",
                null
                );

            // Specimen detail 1
            SpecimenDetail specimenDetail1 = SpecialistLetter.CreateSpecimenDetail();
            specimenDetail1.AnatomicalSite = new List<AnatomicalSite>
                                                 {
                                                     anatomicalSite1
                                                 };
            specimenDetail1.CollectionDateTime = new ISO8601DateTime(DateTime.Now.AddDays(-22));
            specimenDetail1.CollectionProcedure = SpecialistLetter.CreateCollectionProcedure("82078001", CodingSystem.SNOMED, "blood draw", null, null);
            specimenDetail1.CollectionSetting = "Ward 1A";
            specimenDetail1.ContainerIdentifier = BaseCDAModel.CreateInstanceIdentifier("1.2.36.45364", "CNH45218964");
            specimenDetail1.ParentSpecimenIdentifier = BaseCDAModel.CreateInstanceIdentifier("1.2.36.45364", BaseCDAModel.CreateGuid());
            specimenDetail1.SpecimenDescription = "physicalDetails-Description";
            specimenDetail1.PhysicalDetails = new List<PhysicalDetails> {CreatePhysicalDetails("6", "ml")};
            specimenDetail1.ReceivedDateTime = new ISO8601DateTime(DateTime.Now);
            specimenDetail1.SamplingPreconditions = SpecialistLetter.CreateSamplingPreconditions("16985007", CodingSystem.SNOMED, "fasting", null, null);
            specimenDetail1.SpecimenIdentifier = BaseCDAModel.CreateInstanceIdentifier("1.2.36.45364", BaseCDAModel.CreateGuid());
            specimenDetail1.SpecimenTissueType = SpecialistLetter.CreateSpecimenTissueType("85756007", CodingSystem.SNOMED, "Tissue", null, null);

            // Specimen detail 2
            SpecimenDetail specimenDetail2 = SpecialistLetter.CreateSpecimenDetail();
            specimenDetail2.AnatomicalSite = new List<AnatomicalSite>
                                                 {
                                                     anatomicalSite2, anatomicalSite3
                                                 };
            specimenDetail2.CollectionDateTime = new ISO8601DateTime(DateTime.Now.AddDays(-12));
            specimenDetail2.CollectionProcedure = SpecialistLetter.CreateCollectionProcedure("122575003", CodingSystem.SNOMED, "Urine sample", null, null);
            specimenDetail2.CollectionSetting = "Ward 3B";
            specimenDetail2.ContainerIdentifier = BaseCDAModel.CreateInstanceIdentifier("1.2.36.45364", "AZH45218964");
            specimenDetail2.ParentSpecimenIdentifier = BaseCDAModel.CreateInstanceIdentifier("1.2.36.45364", SpecialistLetter.CreateGuid());
            specimenDetail2.SpecimenDescription = "Urine sample";
            specimenDetail2.PhysicalDetails = new List<PhysicalDetails> {CreatePhysicalDetails("20", "kg")};
            specimenDetail2.ReceivedDateTime = new ISO8601DateTime(DateTime.Now);
            specimenDetail2.SamplingPreconditions = SpecialistLetter.CreateSamplingPreconditions("16985007", CodingSystem.SNOMED, "fasting", null, null);
            specimenDetail2.SpecimenIdentifier = BaseCDAModel.CreateInstanceIdentifier("1.2.36.45364", SpecialistLetter.CreateGuid());

            // Specimen detail 3
            SpecimenDetail specimenDetail3 = SpecialistLetter.CreateSpecimenDetail();
            specimenDetail3.CollectionDateTime = new ISO8601DateTime(DateTime.Now.AddDays(-12));
            specimenDetail3.CollectionProcedure = SpecialistLetter.CreateCollectionProcedure("122575003", CodingSystem.SNOMED, "Urine sample", null, null);
            specimenDetail3.CollectionSetting = "Ward 3B";
            specimenDetail3.ContainerIdentifier = BaseCDAModel.CreateInstanceIdentifier("1.2.36.45364", "AZH45218964");
            specimenDetail3.ParentSpecimenIdentifier = BaseCDAModel.CreateInstanceIdentifier("1.2.36.45364", SpecialistLetter.CreateGuid());
            specimenDetail3.SpecimenDescription = "Urine sample";
            specimenDetail3.PhysicalDetails = new List<PhysicalDetails> {CreatePhysicalDetails("20", "kg")};
            specimenDetail3.ReceivedDateTime = new ISO8601DateTime(DateTime.Now);
            specimenDetail3.SamplingPreconditions = SpecialistLetter.CreateSamplingPreconditions("16985007", CodingSystem.SNOMED, "fasting", null, null);
            specimenDetail3.SpecimenIdentifier = BaseCDAModel.CreateInstanceIdentifier("1.2.36.45364", SpecialistLetter.CreateGuid());

            // Test result 1
            ITestResult testResult1 = SpecialistLetter.CreateTestResult();
            testResult1.Comments = new List<string> { "Sodium test result comments", "More comments", "Another comment" };
            testResult1.ReferenceRangeGuidance = "Reference range guidance comments for Sodium; eg. the quantity should be between the high and low values";
            testResult1.ResultName = SpecialistLetter.CreateResultName("39972003", CodingSystem.SNOMED, "Sodium", null, null);
            testResult1.ResultStatus = ResultStatus.Interim;
            testResult1.ResultValue = SpecialistLetter.CreateResultValue();
            testResult1.ResultValue.ValueAsQuantity = SpecialistLetter.CreateQuantity();
            testResult1.ResultValue.ValueAsQuantity.Value = "12";
            testResult1.ResultValue.ValueAsQuantity.Units = "ml";
            testResult1.ResultValueStatus = HL7ObservationInterpretationNormality.Normal;
            testResult1.ResultValueReferenceRangeDetails = new List<ResultValueReferenceRangeDetail>
                                                               {
                                                                   CreateReferenceRange("75540009", "High", "ml", 50, 100), 
                                                                   CreateReferenceRange("62482003", "Low", "ml", 10, 20), 
                                                               };

            // Test result 2
            ITestResult testResult2 = SpecialistLetter.CreateTestResult();
            testResult2.Comments = new List<string> { "Blood test result comments", "More comments", "Another comment" };
            testResult2.ReferenceRangeGuidance = "Reference range guidance comments for blood; eg. the quantity should be between the high and low values";
            testResult2.ResultName = SpecialistLetter.CreateResultName("87612001", CodingSystem.SNOMED, "Blood", null, null);
            testResult2.ResultStatus = ResultStatus.Final;
            testResult2.ResultValue = SpecialistLetter.CreateResultValue();
            testResult2.ResultValue.ValueAsQuantity = SpecialistLetter.CreateQuantity();
            testResult2.ResultValue.ValueAsQuantity.Value = "5";
            testResult2.ResultValue.ValueAsQuantity.Units = "km";
            testResult2.ResultValueStatus = HL7ObservationInterpretationNormality.Normal;
            testResult2.ResultValueReferenceRangeDetails = new List<ResultValueReferenceRangeDetail>
                                                               {
                                                                   CreateReferenceRange("75540009", "High", "ml", 50, 100), 
                                                                   CreateReferenceRange("62482003", "Low", "ml", 10, 20), 
                                                               };

            // Test result 3
            ITestResult testResult3 = SpecialistLetter.CreateTestResult();
            testResult3.Comments = new List<string> { "Heart test result comments", "More comments", "Another comment" };
            testResult3.ReferenceRangeGuidance = "Reference range guidance comments for heart; eg. the quantity should be between the high and low values";
            testResult3.ResultName = SpecialistLetter.CreateResultName("87612001", CodingSystem.SNOMED, "Blood", null, null);
            testResult3.ResultStatus = ResultStatus.Final;
            testResult3.ResultValue = SpecialistLetter.CreateResultValue();
            testResult3.ResultValue.ValueAsQuantity = SpecialistLetter.CreateQuantity();
            testResult3.ResultValue.ValueAsQuantity.Value = "5";
            testResult3.ResultValue.ValueAsQuantity.Units = "km";
            testResult3.ResultValueStatus = HL7ObservationInterpretationNormality.Normal;
            testResult3.ResultValueReferenceRangeDetails = new List<ResultValueReferenceRangeDetail>
                                                               {
                                                                   CreateReferenceRange("75540009", "High", "psi", 2, 150), 
                                                                   CreateReferenceRange("62482003", "Low", "psi", 15, 200), 
                                                               };


            // Test request 1
            ITestRequest testRequest1 = SpecialistLetter.CreateTestRequest();
            testRequest1.LaboratoryTestResultIdentifier = BaseCDAModel.CreateInstanceIdentifier(BaseCDAModel.CreateGuid(), "Laboratory Test Result Identifier");
            testRequest1.TestsRequested = new List<ICodableText>
                                              {
                                                  SpecialistLetter.CreateTestRequestName("67922002", CodingSystem.SNOMED, "Serum"),
                                                  SpecialistLetter.CreateTestRequestName("75367002", CodingSystem.SNOMED, "Blood pressure"),
                                              };

            // Test request 2
            ITestRequest testRequest2 = SpecialistLetter.CreateTestRequest();
            testRequest2.LaboratoryTestResultIdentifier = BaseCDAModel.CreateInstanceIdentifier(BaseCDAModel.CreateGuid(), "Laboratory Test Result Identifier");
            testRequest2.TestsRequested = new List<ICodableText>
                                              {
                                                  SpecialistLetter.CreateTestRequestName("302509004", CodingSystem.SNOMED, "Heart"),
                                                  SpecialistLetter.CreateTestRequestName("302509004", CodingSystem.SNOMED, "Heart"),
                                              };

            // Test result group 1
            ITestResultGroup testResultGroup1 = SpecialistLetter.CreateTestResultGroup();
            testResultGroup1.Results = new List<ITestResult> { testResult1, testResult2 };
            testResultGroup1.ResultGroupSpecimenDetail = specimenDetail1;
            testResultGroup1.ResultGroupName = SpecialistLetter.CreateResultGroupName
                (
                    "18719-5",
                    CodingSystem.LOINC,
                    "Chemistry Studies (Set)",
                    null,
                    null
                );


            // Test result group 2
            ITestResultGroup testResultGroup2 = SpecialistLetter.CreateTestResultGroup();
            testResultGroup2.Results = new List<ITestResult> { testResult3 };
            testResultGroup2.ResultGroupSpecimenDetail = specimenDetail2;
            testResultGroup2.ResultGroupName = SpecialistLetter.CreateResultGroupName(NullFlavour.NegativeInfinity);

            // Pathology test result
            PathologyTestResult pathologyTestResult = SpecialistLetter.CreatePathologyTestResult();
            pathologyTestResult.Comment = "testResultGroup-Comment";
            pathologyTestResult.Conclusion = "testResultGroup-Conclusion";
            pathologyTestResult.TestResultDateTime = new ISO8601DateTime(DateTime.Now.AddDays(-2));
            pathologyTestResult.PathologicalDiagnosis = new List<ICodableText>
                                                            {
                                                                SpecialistLetter.CreatePathologicalDiagnois("17621005", CodingSystem.SNOMED, "Normal", null, null),
                                                                SpecialistLetter.CreatePathologicalDiagnois("85531003", CodingSystem.SNOMED, "Abnormal", null, null)
                                                            };
            pathologyTestResult.OverallTestResultStatus = ResultStatus.Final;
            pathologyTestResult.ClinicalInformationProvided = "Hepatitus";
            pathologyTestResult.DiagnosticService = DiagnosticServiceSectionID.Chemistry;
            pathologyTestResult.TestRequestDetails = new List<ITestRequest>
                                                         {
                                                             testRequest1, testRequest2
                                                         };
            pathologyTestResult.PathologyTestSpecimenDetail = new List<SpecimenDetail> { specimenDetail3 };
            pathologyTestResult.TestResultName = SpecialistLetter.CreateCodableText("18719-5", CodingSystem.LOINC, name, null, null);
            pathologyTestResult.TestResultRepresentation = BaseCDAModel.CreateEncapsulatedData();
            pathologyTestResult.TestResultRepresentation.Text = "Lipase 150 U/L (RR < 70)";
            pathologyTestResult.PathologyTestResultGroup = new List<ITestResultGroup>
                                                               {
                                                                   testResultGroup1, testResultGroup2
                                                               };

            return pathologyTestResult;
        }

        /// Creates and Hydrates an natomicalSite
        /// </summary>
        /// <returns>AnatomicalSite</returns>
        private static AnatomicalSite CreateAnatomicalSite(String description, ICodableText nameOfLocation, ICodableText side, List<ExternalData> images)
        {
            var anatomicalSite = EReferral.CreateAnatomicalSite();
            anatomicalSite.Description = description;
            anatomicalSite.AnatomicalLocation = EReferral.CreateAnatomicalLocation();
            anatomicalSite.AnatomicalLocation.NameOfLocation = nameOfLocation;
            anatomicalSite.AnatomicalLocation.Side = side;
            anatomicalSite.Images = images;
            return anatomicalSite;
        }

        /// <summary>
        /// Creates and hydrates an 'anatomicalSite'.
        /// </summary>
        /// <param name="description"></param>
        /// <param name="code"></param>
        /// <param name="codingSystem"></param>
        /// <param name="name"></param>
        /// <param name="side"></param>
        /// <param name="images"></param>
        /// <returns>AnatomicalSite</returns>
        private static AnatomicalSite CreateAnatomicalSite(string description, string code, CodingSystem codingSystem, string name, string side, List<ExternalData> images)
        {
            var anatomicalSite = SpecialistLetter.CreateAnatomicalSite();
            anatomicalSite.Description = description;
            anatomicalSite.AnatomicalLocation = SpecialistLetter.CreateAnatomicalLocation();
            anatomicalSite.AnatomicalLocation.NameOfLocation = SpecialistLetter.CreateAnatomicLocationName(code, codingSystem, name, null);
            anatomicalSite.AnatomicalLocation.Side = SpecialistLetter.CreateAnatomicLocationSide("7771000", CodingSystem.SNOMED, side, null, null);
            anatomicalSite.Images = images;
            return anatomicalSite;
        }

        /// <summary>
        /// Creates and hydrates an 'IImagingExaminationRequest' object.
        /// </summary>
        /// <param name="examinationRequestName">Name of the examination request.</param>
        /// <param name="imageDetails">List of 'ImageDetails'.</param>
        /// <returns>A hydrated 'IImagingExaminationRequest' object.</returns>
        private static IImagingExaminationRequest CreateImagingExaminationRequest(string examinationRequestName, List<IImageDetails> imageDetails)
        {
            IImagingExaminationRequest imagingExaminationRequest = SpecialistLetter.CreateImagingExaminationRequest();
            imagingExaminationRequest.ExaminationRequestedName = new List<String>
                                                                     {
                                                                         examinationRequestName, "another name"
                                                                     };

            imagingExaminationRequest.ReportIdentifier = BaseCDAModel.CreateInstanceIdentifier(BaseCDAModel.CreateGuid(), "3355552BHU-23.3");

            if (imageDetails != null && imageDetails.Any())
            {
                int index = 1;
                foreach (IImageDetails imageDetail in imageDetails)
                {
                    imageDetail.ImageViewName = SpecialistLetter.CreateImageViewName(null, null, null, index + "Item " + index + " X-Ray - " + examinationRequestName, null);
                    index++;                    
                }

                imagingExaminationRequest.ImageDetails = imageDetails;
            }

            imagingExaminationRequest.StudyIdentifier = imagingExaminationRequest.StudyIdentifier = BaseCDAModel.CreateInstanceIdentifier(BaseCDAModel.CreateGuid(), "Accession Number Group: 0008  Element: 0050");

            return imagingExaminationRequest;
        }

        /// <summary>
        /// Creates and hydrates a 'IImagingExaminationRequest' object.
        /// </summary>
        /// <param name="examinationRequestName">Name of the examination request.</param>
        /// <param name="imageDetails">Image details.</param>
        /// <returns>A hydrated 'IImagingExaminationRequest' object</returns>
        private static IImagingExaminationRequest CreateImagingExaminationRequest(String examinationRequestName, IImageDetails imageDetails)
        {
            return CreateImagingExaminationRequest(examinationRequestName, new List<IImageDetails> { imageDetails });
        }

        #endregion
    }
}
