﻿/*
 * Copyright 2011 NEHTA
 *
 * Licensed under the NEHTA Open Source (Apache) License; you may not use this
 * file except in compliance with the License. A copy of the License is in the
 * 'license.txt' file, which should be provided with this work.
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.Serialization;
using System.Xml;
using CDA.Generator.Common.CDAModel.Entities;
using Nehta.VendorLibrary.CDA.Common.Enums;
using Nehta.VendorLibrary.CDA.Generator.Enums;
using Nehta.VendorLibrary.CDA.SCSModel.Common;
using Nehta.VendorLibrary.CDA.SCSModel;
using Nehta.VendorLibrary.CDA.CDAModel;
using Nehta.VendorLibrary.Common;

namespace Nehta.VendorLibrary.CDA.Common
{
    /// <summary>
    /// This DispenseRecord object is a composition of the context and content objects that define
    /// a CDA Dispense Record document
    /// 
    /// This object is also responsible for providing the factory methods used to instantiate any DispenseRecord
    /// objects that are required to build a valid Dispense Record CDA document
    /// </summary>
    [Serializable]
    [DataContract]
    [KnownType(typeof(Content))]
    [KnownType(typeof(Context))]
    [KnownType(typeof(CDAContext))]
    public class DispenseRecord : BaseCDAModel
    {
        #region Properties

        /// <summary>
        /// An ICDAContextDispenseRecord that contains the CDA Context for this Dispense Record document
        /// </summary>
        [DataMember]
        public ICDAContextDispenseRecord CDAContext { get; set; }

        /// <summary>
        /// An IDispenseRecordContent that contains the SCS Content for this Dispense Record document
        /// </summary>
        [DataMember]
        public IDispenseRecordContent SCSContent { get; set; }

        /// <summary>
        /// An IDispenseRecordContext that contains the SCS Context for this Dispense Record document
        /// </summary>
        [DataMember]
        public IDispenseRecordContext SCSContext { get; set; }
        #endregion

        #region Constructors
        /// <summary>
        /// Instantiates a Dispense Record model; the status of this CDA document will be 
        /// set to the status passed into this constructor
        /// </summary>
        /// <param name="documentStatus">Document Status</param>
        public DispenseRecord(DocumentStatus documentStatus)
        {
            DocumentStatus = documentStatus;
        }

        /// <summary>
        /// Instantiates a Dispense Record model; the status of this document will be 
        /// set to final
        /// </summary>
        public DispenseRecord() : this(DocumentStatus.Final)
        {
        }
        #endregion

        #region Validation
        /// <summary>
        /// Validates this DispenseRecord object and its child objects
        /// </summary>
        /// <param name="path">The path to this object as a string</param>
        /// <param name="messages">the validation messages to date, these may be added to within this method</param>
        public void Validate(string path, List<ValidationMessage> messages)
        {
            var vb = new ValidationBuilder(path, messages);

            if (vb.ArgumentRequiredCheck("CDAContext", CDAContext))
              CDAContext.Validate(vb.Path + "CDAContext", vb.Messages);

            if (vb.ArgumentRequiredCheck("SCSContext", SCSContext))
              SCSContext.Validate(vb.Path + "SCSContext", vb.Messages);

            if (vb.ArgumentRequiredCheck("SCSContent", SCSContent))
              SCSContent.Validate(vb.Path + "SCSContent", vb.Messages);
        }
        #endregion

        #region Static Methods
        /// <summary>
        /// Creates a DispenseRecord
        /// </summary>
        /// <returns>DispenseRecord</returns>
        public static DispenseRecord CreateDispenseRecord()
        {
            return new DispenseRecord();
        }

        /// <summary>
        /// Creates a DispenseRecord
        /// </summary>
        /// <returns>DispenseRecord</returns>
        public static DispenseRecord CreateDispenseRecord(DocumentStatus documentStatus)
        {
            return new DispenseRecord
            {
                DocumentStatus = documentStatus
            };
        }

        /// <summary>
        /// Creates a CDA Context
        /// </summary>
        /// <returns>(ICDAContextDispenseRecord) Context</returns>
        public static ICDAContextDispenseRecord CreateCDAContext()
        {
            return new CDAContext();
        }

        /// <summary>
        /// Creates an SCS content
        /// </summary>
        /// <returns>(IDispenseRecordContent) Content</returns>
        public static IDispenseRecordContent CreateSCSContent()
        {
            return new Content();
        }

        /// <summary>
        /// Creates an SCS context
        /// </summary>
        /// <returns>(IDispenseRecordContext) Context</returns>
        public static IDispenseRecordContext CreateSCSContext()
        {
            return new Context();
        }

        /// <summary>
        /// Creates a Dispense Item
        /// </summary>
        /// <returns>DispenseItem</returns>
        public static DispenseItem CreateDispenseItem()
        {
            return new DispenseItem();
        }

        /// <summary>
        /// Creates a CreateDispensingInformation 
        /// </summary>
        /// <returns>DispensingInformation</returns>
        public static DispensingInformation CreateDispensingInformation()
        {
          return new DispensingInformation();
        }

        /// <summary>
        /// Creates a ParentDocument  
        /// </summary>
        /// <returns>ParentDocument</returns>
        public static ParentDocument CreateParentDocument()
        {
          return new ParentDocument();
        }

        #endregion

        #region Serialisation Methods
        /// <summary>
        /// This method serializes this model into an XML document and returns this document
        /// </summary>
        /// <returns>XmlDocument</returns>
        public XmlDocument SerializeModel()
        {
            XmlDocument xmlDocument = null;
            var dataContractSerializer = new DataContractSerializer(typeof(DispenseRecord));
            
            using(var memoryStream = new MemoryStream())
            {
                xmlDocument = new XmlDocument();

                dataContractSerializer.WriteObject(memoryStream, this);

                memoryStream.Seek(0, SeekOrigin.Begin);

                xmlDocument.Load(memoryStream);
            }

            return xmlDocument;
        }

        /// <summary>
        /// This method deserializes the xml document into an Dispense Record object
        /// </summary>
        /// <returns>XmlDocument</returns>
        public static DispenseRecord DeserializeXmlDocument(XmlDocument xmlDocument)
        {
            DispenseRecord DispenseRecord = null;

            var dataContractSerializer = new DataContractSerializer(typeof(DispenseRecord));

            using (var memoryStream = new MemoryStream())
            {
                xmlDocument.Save(memoryStream);

                memoryStream.Position = 0;

                DispenseRecord = (DispenseRecord)dataContractSerializer.ReadObject(memoryStream);
            }

            return DispenseRecord;
        }
        #endregion
    }
}