﻿/*
 * Copyright 2011 NEHTA
 *
 * Licensed under the NEHTA Open Source (Apache) License; you may not use this
 * file except in compliance with the License. A copy of the License is in the
 * 'license.txt' file, which should be provided with this work.
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

using System;
using System.Collections.Generic;
using System.Xml;
using Nehta.HL7.CDA;
using Nehta.VendorLibrary.CDA.Generator.Enums;
using Nehta.VendorLibrary.Common;
using Nehta.VendorLibrary.CDA.Generator;
using Nehta.VendorLibrary.CDA.Common.Enums;
using Nehta.VendorLibrary.CDA.Common;
using Nehta.VendorLibrary.CDA.SCSModel;
using Nehta.VendorLibrary.CDA.SCSModel.Common;
using Nehta.VendorLibrary.CDA.SCSModel.Common.Entities;

namespace Nehta.VendorLibrary.CDA.Sample
{
    /// <summary>
    /// This project is intended to demonstrate how a Shared Heath Summary CDA document can be created.
    /// 
    /// The project contains two samples, the first is designed to create a fully populated CDA document, including
    /// all of the optional sections and entries. The second sample only populates the mandatory sections and entries.
    /// 
    /// The CDA model is split into three distinct sections, each of which needs to be created via the 
    /// SharedHealthSummary class, and then populated with data as appropriate. The three sections that need to be
    /// created and hydrated with data are:
    /// 
    ///     CDA Context (Clinical Document Architecture - Context)
    ///     SCS Context (Structured Document Template - Context)
    ///     SCS Content (Structured Document Template - Content)
    /// 
    /// The CDA Context typically contains information that is to be represented within the header of the document
    /// that is not encapsulated with the SCS context.
    /// E.g. Generic CDA sections or entries; for example custodian.
    /// 
    /// The SCS Context typically contains information that is to be represented within the header of the document
    /// and relates specifically to the type of document that is to be created.
    /// E.g. Shared Health Summary specific CDA sections or entries; for example Subject of Care.
    /// 
    /// The SCS Content typically contains information that is to be represented with the body of the document.
    /// </summary>
    public class SharedHealthSummarySample
    {
        #region Properties

        public static string OutputFolderPath { get; set; }

        public static String OutputFileNameAndPath
        {
            get
            {
                return OutputFolderPath + @"\SharedHealthSummary.xml";
            }
        }

        public static String ImageFileNameAndPath
        {
            get
            {
                return OutputFolderPath + @"\x-ray.jpg";
            }
        }

        public static String StructuredFileAttachment
        {
            get
            {
                return OutputFolderPath + @"\attachment.pdf";
            }
        }

        public static String ResultFileNameAndPath
        {
            get
            {
                return OutputFolderPath + @"\path1234.pdf";
            }
        }

        // Note: Place this in any string field and and this will insert a break
        private const String DELIMITERBREAK = "<BR>";

        #endregion

        /// <summary>
        /// This example populates only the mandatory Sections / Entries; as a result this sample omits all
        /// of the content within the body of the CDA document; as each of the sections within the body
        /// are optional.
        /// </summary>
        public XmlDocument PopulatedSharedHealthSummarySample()
        {
            XmlDocument xmlDoc = null;

            var sharedHealthSummary = PopulateSharedHealthSummary(true);

            try
            {
                //Pass the Shared health Summary model into the GenerateSharedHealthSummary method 
                xmlDoc = CDAGenerator.GenerateSharedHealthSummary(sharedHealthSummary);
                xmlDoc.Save(OutputFileNameAndPath);
            }
            catch (ValidationException ex)
            {
                //Catch any validation exceptions
                var validationMessages = ex.GetMessagesString();

                //Handle any validation errors as appropriate.
                throw;
            }

            return xmlDoc;
        }

        /// <summary>
        /// This example populates only the mandatory Sections / Entries; as a result this sample omits all
        /// of the content within the body of the CDA document; as each of the sections within the body
        /// are optional.
        /// </summary>
        public XmlDocument PopulateSharedHealthSummarySample_1A(string fileName)
        {
            XmlDocument xmlDoc = null;

            var sharedHealthSummary = PopulateSharedHealthSummary(true);
            sharedHealthSummary.SCSContent = SharedHealthSummary.CreateSCSContent();

            sharedHealthSummary.IncludeLogo = false;

            var structuredBodyFileList = new List<ExternalData>();

            var structuredBodyFile = BaseCDAModel.CreateStructuredBodyFile();
            structuredBodyFile.Caption = "Structured Body File";
            structuredBodyFile.ExternalDataMediaType = MediaType.PDF;
            structuredBodyFile.Path = StructuredFileAttachment;
            structuredBodyFileList.Add(structuredBodyFile);

            sharedHealthSummary.SCSContent.StructuredBodyFiles = structuredBodyFileList;

            try
            {
                CDAGenerator.NarrativeGenerator = new CDANarrativeGenerator();

                //Pass the E-Referral model into the GenerateEReferral method 
                xmlDoc = CDAGenerator.GenerateSharedHealthSummary(sharedHealthSummary);

                using (var writer = XmlWriter.Create(OutputFolderPath + @"\" + fileName, new XmlWriterSettings() { Indent = true }))
                {
                    if (!fileName.IsNullOrEmptyWhitespace()) xmlDoc.Save(writer);
                }
            }
            catch (ValidationException ex)
            {
                //Catch any validation exceptions
                var validationMessages = ex.GetMessagesString();

                //Handle any validation errors as appropriate.
                throw;
            }

            return xmlDoc;
        }

        /// <summary>
        /// This sample populates only the mandatory Sections / Entries;
        /// </summary>
        public XmlDocument MinPopulatedSharedHealthSummarySample(string fileName)
        {
            XmlDocument xmlDoc = null;

            var sharedHealthSummary = PopulateSharedHealthSummary(true);

            try
            {
                //Pass the Shared health Summary model into the GenerateSharedHealthSummary method 
                xmlDoc = CDAGenerator.GenerateSharedHealthSummary(sharedHealthSummary);

                using (var writer = XmlWriter.Create(OutputFolderPath + @"\" + fileName, new XmlWriterSettings() { Indent = true }))
                {
                    if (!fileName.IsNullOrEmptyWhitespace()) xmlDoc.Save(writer);
                }
            }
            catch (ValidationException ex)
            {
                //Catch any validation exceptions
                var validationMessages = ex.GetMessagesString();

                //Handle any validation errors as appropriate.
                throw;
            }

            return xmlDoc;
        }

        /// <summary>
        /// This sample populates both the mandatory and optional Sections / Entries; as a result this sample
        /// includes all of the sections within the body and each section includes at least one example for 
        /// each of its optional entries
        /// </summary>
        public XmlDocument MaxPopulatedSharedHealthSummarySample(string fileName)
        {
            XmlDocument xmlDoc = null;

            var sharedHealthSummary = PopulateSharedHealthSummary(false);

            try
            {
                //Pass the Shared health Summary model into the GenerateSharedHealthSummary method 
                xmlDoc = CDAGenerator.GenerateSharedHealthSummary(sharedHealthSummary);

                using (var writer = XmlWriter.Create(OutputFolderPath + @"\" + fileName, new XmlWriterSettings() { Indent = true }))
                {
                    if (!fileName.IsNullOrEmptyWhitespace()) xmlDoc.Save(writer);
                }
            }
            catch (ValidationException ex)
            {
                //Catch any validation exceptions
                var validationMessages = ex.GetMessagesString();

                //Handle any validation errors as appropriate.
                throw;
            }

            return xmlDoc;
        }

        #region Private Methods
        /// <summary>
        /// This method populates a shared health summary model with either the mandatory sections only, or both 
        /// the mandatory and optional sections
        /// </summary>
        /// <param name="mandatorySectionsOnly">mandatorySectionsOnly</param>
        /// <returns>SharedHealthSummary</returns>
        internal static SharedHealthSummary PopulateSharedHealthSummary(Boolean mandatorySectionsOnly)
        {
            var sharedHealthSummary = SharedHealthSummary.CreateSharedHealthSummary(DocumentStatus.Final);
            // Include Logo
            sharedHealthSummary.IncludeLogo = true;
            // Set Creation Time
            sharedHealthSummary.DocumentCreationTime = new ISO8601DateTime(DateTime.Now);

            #region Setup and populate the CDA context model
            // Setup and populate the CDA context model
            var cdaContext = SharedHealthSummary.CreateCDAContext();

            // Document Id
            cdaContext.DocumentId = BaseCDAModel.CreateIdentifier(BaseCDAModel.CreateGuid(), null);
            // Set Id  
            cdaContext.SetId = BaseCDAModel.CreateIdentifier(BaseCDAModel.CreateGuid(), null);
            // CDA Context Version
            cdaContext.Version = "1";

            // Custodian
            cdaContext.Custodian = SharedHealthSummary.CreateCustodian();
            GenericObjectReuseSample.HydrateCustodian(cdaContext.Custodian, mandatorySectionsOnly);

            // Legal authenticator
            if (!mandatorySectionsOnly)
            {
                cdaContext.LegalAuthenticator = SharedHealthSummary.CreateLegalAuthenticator();
                GenericObjectReuseSample.HydrateAuthenticator(cdaContext.LegalAuthenticator, mandatorySectionsOnly);
            }

            sharedHealthSummary.CDAContext = cdaContext;
            #endregion

            #region Setup and Populate the SCS Context model
            // Setup and Populate the SCS Context model

            sharedHealthSummary.SCSContext = SharedHealthSummary.CreateSCSContext();

            //Author
            sharedHealthSummary.SCSContext.Author = SharedHealthSummary.CreateAuthor();
            GenericObjectReuseSample.HydrateAuthor(sharedHealthSummary.SCSContext.Author, mandatorySectionsOnly);

            // Subject of Care
            sharedHealthSummary.SCSContext.SubjectOfCare = SharedHealthSummary.CreateSubjectOfCare();
            GenericObjectReuseSample.HydrateSubjectofCare(sharedHealthSummary.SCSContext.SubjectOfCare, mandatorySectionsOnly);

            #endregion

            #region Setup and populate the SCS Content model
            // Setup and populate the SCS Content model

            sharedHealthSummary.SCSContent = SharedHealthSummary.CreateSCSContent();

            // Adverse reactions
            sharedHealthSummary.SCSContent.AdverseReactions = CreateAdverseReactions(mandatorySectionsOnly);

            // Reviewed medications
            sharedHealthSummary.SCSContent.Medications = CreateMedications(mandatorySectionsOnly);

            // Reviewed medical history
            sharedHealthSummary.SCSContent.MedicalHistory = CreateMedicalHistory(mandatorySectionsOnly);

            // Reviewed Immunizations
            sharedHealthSummary.SCSContent.Imunisations = CreateImmunisations(mandatorySectionsOnly);

            #endregion

            return sharedHealthSummary;
        }

        /// <summary>
        /// Creates and Hydrates the reviewed medications section for the Shared Health Summary
        /// 
        /// Note: the data used within this method is intended as a guide and should be replaced.
        /// </summary>
        /// <returns>A Hydrated ReviewedMedications object</returns>
        private static IMedications CreateMedications(Boolean mandatorySectionsOnly)
        {
            var medicationList = new List<IMedication>();

            var medications = SharedHealthSummary.CreateMedications();

            if (!mandatorySectionsOnly)
            {
                var medication = SharedHealthSummary.CreateMedication();
                medication.ClinicalIndication = "Diuretic induced hypokalemia";
                medication.Comment = "Taken with food";
                medication.Directions = "2 tablets once daily oral";
                medication.Medicine = SharedHealthSummary.CreateMedicine("6543456", CodingSystem.AMT, "Slow K (potassium chloride 600mg) tablet", null, null);
                medicationList.Add(medication);

                var medication1 = SharedHealthSummary.CreateMedication();
                medication1.ClinicalIndication = "Arthritis pain management";
                medication1.Comment = "Swallow whole";
                medication1.Directions = "2 tablets three times per day";
                medication1.Medicine = SharedHealthSummary.CreateMedicine("7654567", CodingSystem.AMT, "Panadol Osteo (paracetamol 665 mg) tablet", null, null);
                medicationList.Add(medication1);

                var medication2 = SharedHealthSummary.CreateMedication();
                medication2.ClinicalIndication = "Fluid retention";
                medication2.Comment = "Take in the morning";
                medication2.Directions = "1 tablet once daily oral";
                medication2.Medicine = SharedHealthSummary.CreateMedicine("5678765", CodingSystem.AMT, "Lasix (frusemide 40 mg) tablet", null, null);
                medicationList.Add(medication2);

                var medication3 = SharedHealthSummary.CreateMedication();
                medication3.ClinicalIndication = "COPD";
                medication3.Directions = "1 inhalation per day";
                medication3.Medicine = SharedHealthSummary.CreateMedicine("7654567", CodingSystem.AMT, "Spiriva (tiotropium bromide 18mgper inhalation) inhalant", null, null);
                medicationList.Add(medication3);

                var medication4 = SharedHealthSummary.CreateMedication();
                medication4.ClinicalIndication = "Depression";
                medication4.Directions = "As directed by packaging";
                medication4.Medicine = SharedHealthSummary.CreateMedicine("4567654", CodingSystem.AMT, "St John's Wort", null, null);
                medicationList.Add(medication4);

                var medication5 = SharedHealthSummary.CreateMedication();
                medication5.ClinicalIndication = "Depression";
                medication5.DirectionsNullFlavour = NullFlavour.PositiveInfinity;
                medication5.Medicine = SharedHealthSummary.CreateMedicine("4567654", CodingSystem.AMT, "St John's Wort", null, null);
                medicationList.Add(medication5);
                medications.Medications = medicationList;

            } else
            {
               medications.ExclusionStatement = SharedHealthSummary.CreateExclusionStatement();
               medications.ExclusionStatement.Value = NCTISGlobalStatementValues.NotAsked;  
            }

            return medications;
        }

        /// <summary>
        /// Creates and Hydrates the reviewed medical history section for the Shared Health Summary
        /// 
        /// Note: the data used within this method is intended as a guide and should be replaced.
        /// </summary>
        /// <returns>A Hydrated ReviewedMedicalHistory object</returns>
        private static IMedicalHistory CreateMedicalHistory(Boolean mandatorySectionsOnly)
        {
            var procedureList = new List<Procedure>();

            var history = SharedHealthSummary.CreateMedicalHistory();

            if (!mandatorySectionsOnly)
            {
                var procedure = SharedHealthSummary.CreateProcedure();
                procedure.Comment = "Comment";
                procedure.ProcedureName = SharedHealthSummary.CreateProcedureName("S7446", CodingSystem.AIHW, "Lasik Eye Surgery", null, null);
                procedure.Start = new ISO8601DateTime(DateTime.Now.AddDays(-12));
                procedureList.Add(procedure);

                var prodcedure2 = SharedHealthSummary.CreateProcedure();
                prodcedure2.Comment = "Comment";
                prodcedure2.ProcedureName = SharedHealthSummary.CreateProcedureName("397956004", CodingSystem.SNOMED, "hip replacement", null, null);
                prodcedure2.Start = new ISO8601DateTime(DateTime.Now.AddDays(-40));
                procedureList.Add(prodcedure2);

                var medicalHistoryItems = new List<IMedicalHistoryItem>();
                var medicalHistoryItem = EventSummary.CreateMedicalHistoryItem();
                medicalHistoryItem.DateTimeInterval = BaseCDAModel.CreateInterval(
                    new ISO8601DateTime(DateTime.Now, ISO8601DateTime.Precision.Day), 
                    new ISO8601DateTime(DateTime.Now.AddDays(200), ISO8601DateTime.Precision.Day));
                medicalHistoryItem.ItemDescription = "Medical history item goes here";
                medicalHistoryItem.ItemComment = "Medical history Comment goes here";
                medicalHistoryItems.Add(medicalHistoryItem);

                var diagnosisList = new List<IProblemDiagnosis>();
                var diagnois = SharedHealthSummary.CreateProblemDiagnosis();
                diagnois.ProblemDiagnosisIdentification = SharedHealthSummary.CreateProblemDiagnosisIdentification("F1664", CodingSystem.AIHW, "Hypercholesterolaemia", "Hypercholesterolaemia");
                diagnois.DateOfOnset = new ISO8601DateTime(DateTime.Now.AddYears(-2));
                diagnois.DateOfResolutionRemission = new ISO8601DateTime(DateTime.Now.AddYears(-1), ISO8601DateTime.Precision.Day);
                diagnois.Comment = "Solved this";
                diagnosisList.Add(diagnois);

                var diagnois1 = SharedHealthSummary.CreateProblemDiagnosis();
                diagnois1.ProblemDiagnosisIdentification = SharedHealthSummary.CreateProblemDiagnosisIdentification("F4663", CodingSystem.AIHW, "Chronic Obstructive Airways Disease", "Chronic Obstructive Airways Disease");
                diagnosisList.Add(diagnois1);

                var diagnois2 = SharedHealthSummary.CreateProblemDiagnosis();
                diagnois2.ProblemDiagnosisIdentification = SharedHealthSummary.CreateProblemDiagnosisIdentification("F4673", CodingSystem.AIHW, "Depression", "Depression");
                diagnosisList.Add(diagnois2);

                var diagnois3 = SharedHealthSummary.CreateProblemDiagnosis();
                diagnois3.Comment = "Diuretic induced";
                diagnois3.ProblemDiagnosisIdentification = SharedHealthSummary.CreateProblemDiagnosisIdentification("F7633", CodingSystem.AIHW, "Hypokalemia", "Hypokalemia");
                diagnosisList.Add(diagnois3);

                var diagnois4 = SharedHealthSummary.CreateProblemDiagnosis();
                diagnois4.ProblemDiagnosisIdentification = SharedHealthSummary.CreateProblemDiagnosisIdentification("F1667", CodingSystem.AIHW, "Fluid retention", "Fluid retention");
                diagnosisList.Add(diagnois4);

                var diagnois5 = SharedHealthSummary.CreateProblemDiagnosis();
                diagnois5.ProblemDiagnosisIdentification = SharedHealthSummary.CreateProblemDiagnosisIdentification("F1662", CodingSystem.AIHW, "Osteoartritis", "Osteoartritis");
                diagnois5.DateOfOnset = new ISO8601DateTime(DateTime.Parse("01 Jan 2000"));
                diagnosisList.Add(diagnois5);

                var diagnois6 = SharedHealthSummary.CreateProblemDiagnosis();
                diagnois6.Comment = "Cementless";
                diagnois6.ProblemDiagnosisIdentification = SharedHealthSummary.CreateProblemDiagnosisIdentification("F3564", CodingSystem.AIHW, "Left TKR", "Left TKR");
                diagnois6.DateOfOnset = new ISO8601DateTime(DateTime.Parse("27 Feb 2007"));
                diagnosisList.Add(diagnois6);

                var diagnois7 = SharedHealthSummary.CreateProblemDiagnosis();
                diagnois7.Comment = "T-score less than -3";
                diagnois7.ProblemDiagnosisIdentification = SharedHealthSummary.CreateProblemDiagnosisIdentification("F1662", CodingSystem.AIHW, "Osteoartritis", "Osteoartritis");
                diagnois7.DateOfOnset = new ISO8601DateTime(DateTime.Parse("01 Jan 2007"));
                diagnosisList.Add(diagnois7);

                var diagnois8 = SharedHealthSummary.CreateProblemDiagnosis();
                diagnois8.Comment = "Comment";
                diagnois8.ProblemDiagnosisIdentification = SharedHealthSummary.CreateProblemDiagnosisIdentification("F5664", CodingSystem.AIHW, "RLL pneumonmia", "RLL pneumonmia");
                diagnois8.DateOfOnset = new ISO8601DateTime(DateTime.Parse("01 Aug 2010"));
                diagnois8.DateOfResolutionRemission = new ISO8601DateTime(DateTime.Parse("01 Sep 2010"), ISO8601DateTime.Precision.Day);
                diagnosisList.Add(diagnois8);

                history.MedicalHistoryItems = medicalHistoryItems;
                history.Procedures = procedureList;
                history.ProblemDiagnosis = diagnosisList;

            } 
                else
            {
                history.ProblemDiagnosisExclusionStatement = SharedHealthSummary.CreateExclusionStatement();
                history.ProblemDiagnosisExclusionStatement.Value = NCTISGlobalStatementValues.NotAsked;

                history.ProceduresExclusionStatement = SharedHealthSummary.CreateExclusionStatement();
                history.ProceduresExclusionStatement.Value = NCTISGlobalStatementValues.NotAsked;
            }
            return history;
        }

        /// <summary>
        /// Creates and Hydrates the reviewed immunisations section for the Shared Health Summary
        /// 
        /// Note: the data used within this method is intended as a guide and should be replaced.
        /// </summary>
        /// <returns>A Hydrated Immunisations object</returns>
        private static Immunisations CreateImmunisations(Boolean mandatorySectionsOnly)
        {
            var immunisations = SharedHealthSummary.CreateReviewedImmunisations();

            if (!mandatorySectionsOnly)
            {
                var immunisation = SharedHealthSummary.CreateImmunisation();
                var immunisationList = new List<IImmunisation>();

                immunisation.DateTime = new ISO8601DateTime(DateTime.Parse("22 Dec 2009"));
                immunisation.Medicine = SharedHealthSummary.CreateMedicine("75071011000036107", CodingSystem.SNOMED, "IPV (inactivated poliovirus vaccine)", null, null);
                immunisationList.Add(immunisation);

                var immunisation2 = SharedHealthSummary.CreateImmunisation();
                immunisation2.DateTime = new ISO8601DateTime(DateTime.Parse("9 Feb 2010"));
                immunisation2.Medicine = SharedHealthSummary.CreateMedicine("62813011000036100", CodingSystem.SNOMED,"Engerix B (Hepatitis B)", null, null);
                immunisationList.Add(immunisation2);

                var immunisation3 = SharedHealthSummary.CreateImmunisation();
                immunisation3.DateTime = new ISO8601DateTime(DateTime.Parse("10 Sep 2010"));
                immunisation3.Medicine = SharedHealthSummary.CreateMedicine("92813511000036100", CodingSystem.SNOMED,"Meningitec (Meningococcal)", null, null);
                immunisationList.Add(immunisation3);

                var immunisation4 = SharedHealthSummary.CreateImmunisation();
                immunisation4.DateTime = new ISO8601DateTime(DateTime.Parse("3 Oct 2010"));
                immunisation4.Medicine = SharedHealthSummary.CreateMedicine("58818811000066100", CodingSystem.SNOMED,"Fluvax (Influenza)", null, null);
                immunisation4.SequenceNumber = 1;
                immunisationList.Add(immunisation4);

                immunisations.AdministeredImmunisation = immunisationList;
            }
            else
            {
                immunisations.ExclusionStatement = SharedHealthSummary.CreateExclusionStatement();
                immunisations.ExclusionStatement.Value = NCTISGlobalStatementValues.NotAsked;
            }


            return immunisations;
        }

        /// <summary>
        /// Creates and Hydrates the reviewed adverse substance reactions section for the Shared Health Summary
        /// 
        /// Note: the data used within this method is intended as a guide and should be replaced.
        /// </summary>
        /// <returns>A Hydrated ReviewedAdverseSubstanceReactions object</returns>
        private static IAdverseReactions CreateAdverseReactions(Boolean mandatorySectionsOnly)
        {
            var adverseReactions = SharedHealthSummary.CreateReviewedAdverseSubstanceReactions();

            if (!mandatorySectionsOnly)
            {
                var reaction = SharedHealthSummary.CreateReaction();

                reaction.ReactionEvent = SharedHealthSummary.CreateReactionEvent();
                reaction.SubstanceOrAgent = SharedHealthSummary.CreateSubstanceOrAgent("90580008", CodingSystem.SNOMED, "Penicillin", null, null);

                reaction.ReactionEvent.Manifestations = new List<ICodableText>
                                                            {
                                                                EventSummary.CreateManifestation
                                                                (
                                                                    "271807003",
                                                                    CodingSystem.SNOMED,
                                                                    "Skin Rash",
                                                                    null,
                                                                    null
                                                                ),
                                                                EventSummary.CreateManifestation
                                                                (
                                                                    "271807001",
                                                                    CodingSystem.SNOMED,
                                                                    "Red Sore",
                                                                    null,
                                                                    null
                                                                )
                                                            };

                var reaction2 = SharedHealthSummary.CreateReaction();
                reaction2.SubstanceOrAgent = SharedHealthSummary.CreateSubstanceOrAgent("40588006", CodingSystem.SNOMED, "Metoprolol", null, null);
                reaction2.ReactionEvent = SharedHealthSummary.CreateReactionEvent();
                reaction2.ReactionEvent.Manifestations = new List<ICodableText>
                                                        {
                                                                EventSummary.CreateManifestation
                                                                (
                                                                    "271807003",
                                                                    CodingSystem.SNOMED,
                                                                    "Skin Rash",
                                                                    null,
                                                                    null
                                                                ),
                                                                EventSummary.CreateManifestation
                                                                (
                                                                    "271807001",
                                                                    CodingSystem.SNOMED,
                                                                    "Red Sore",
                                                                    null,
                                                                    null
                                                                )
                                                        };

                adverseReactions.AdverseSubstanceReaction = new List<Reaction> { reaction, reaction2 };
            } 
            else
            {
                adverseReactions.ExclusionStatement = SharedHealthSummary.CreateExclusionStatement();
                adverseReactions.ExclusionStatement.Value = NCTISGlobalStatementValues.NoneKnown;
            }

            return adverseReactions;
        }
        #endregion
    }
}
