﻿using System.ServiceModel;
using System.Transactions;
using HIPS.Client.Proxy;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.ServiceContracts.AckQueue;
using HIPS.ServiceContracts.AckQueue.Message;
using System.Threading.Tasks;
using System.Threading;

namespace HIPS.AppServer.ServiceHost
{
    /// <summary>
    /// This class implements operations for the AckQueue MSMQ queue.
    /// </summary>
    [ServiceBehavior(InstanceContextMode = InstanceContextMode.Single,
                     ConcurrencyMode = ConcurrencyMode.Single,
                     ReleaseServiceInstanceOnTransactionComplete = false,
                     TransactionAutoCompleteOnSessionClose = false)]
    public class AckQueue : BaseService, IAckQueue
    {

        #region Methods

        /// <summary>
        /// This operation is triggered by the Ack Queue MSMQ when there is a 
        /// SendAckRequest ready to process. It stores the acknowledgment in the logs 
        /// and if the MLLP is set up for responses sends the acknowledgment to the MLLP interface for forwarding.
        /// </summary>
        /// <param name="request">The request containing the acknowledgement to be processed.</param>
        [OperationBehavior(TransactionScopeRequired = true, TransactionAutoComplete = true)]
        public void SendAck(SendAckQueueRequest request)
        {
            UserDetails user = null;

            // 1. Log the acknowledgement.
            string logMessage = string.Format(
                "ACK {0}:{1}:{2}:{3}:{4}",
                request.Acknowledgement.MessageControlID,
                request.Acknowledgement.OriginalMessageControlId,
                request.Acknowledgement.AcknowledgementCode,
                request.Acknowledgement.ErrorConditionCode,
                request.Acknowledgement.ErrorConditionText);
            using (new TransactionScope(TransactionScopeOption.Suppress))
            {
                EventLogger.WriteLog(logMessage, null, user, PcehrDataStore.Schemas.Enumerators.LogMessage.HIPS_MESSAGE_187);
            }

            // 2. Invoke SendAck of the Ack Service.
            ThreadPool.QueueUserWorkItem(state =>
            {
                using (var proxy = new AckServiceProxy("AckServiceEndPoint"))
                {
                    var sendAckRequest = new SendAckRequest
                    {
                        Acknowledgement = request.Acknowledgement.AcknowledgementMessage,
                        User = request.User
                    };
                    var response = Task.Run(() => proxy.SendAck(sendAckRequest)).Result;
                }
            });
        }

        #endregion Methods

    }
}
