﻿using System;
using System.Collections.Generic;
using System.IO;

using HIPS.Configuration;
using HIPS.ServiceContracts.AckQueue;
using HIPS.ServiceContracts.AckQueue.Message;
using HIPS.ServiceContracts.Common;

namespace HIPS.AppServer.ServiceHost
{
    /// <summary>
    /// This class implements web services for Ack Queue.
    /// </summary>
    public class AckService : BaseService, IAckService
    {
        #region Methods

        /// <summary>
        /// This operation is invoked by the Ack Queue when there is a SendAckRequest ready to process.
        /// The implementation within eHISC application server is a simple implementation that merely stores the acknowledgement into a file.
        /// When the MLLP interface is configured, the Ack Queue instead invokes the implementation of this service in Mirth®.
        /// The implementation in Mirth® sends the acknowledgement to the configured MLLP endpoint.
        /// </summary>
        /// <param name="request">The request containing the acknowledgement to be processed.</param>
        public SendAckResponse SendAck(SendAckRequest request)
        {
            var response = new SendAckResponse();
            string fileLocation = Settings.Instance.FileLocation;
            string fileName = string.Format("{0}.ack", Guid.NewGuid().ToString());

            try
            {
                // If the acknowledgement file location is configured, store the acknowledgement into a file in the configured location. Otherwise, do nothing.
                if (!string.IsNullOrEmpty(fileLocation))
                {
                    if (!Directory.Exists(fileLocation))
                    {
                        Directory.CreateDirectory(fileLocation);
                    }
                    string filePath = Path.Combine(fileLocation, fileName);
                    File.WriteAllText(filePath, request.Acknowledgement);
                }

                response.Status = ResponseStatus.OK;
            }
            catch (Exception ex)
            {
                response.Status = ResponseStatus.Warning;
                response.Messages = ToMessageList(ex);
            }

            return response;
        }

        /// <summary>
        /// Converts an exception to a message list.
        /// </summary>
        /// <param name="ex"></param>
        /// <returns></returns>
        private List<ServiceContracts.Common.DTO.Message> ToMessageList(Exception ex)
        {
            var list = new List<ServiceContracts.Common.DTO.Message>();
            if (ex != null)
            {
                list.Add(new ServiceContracts.Common.DTO.Message
                {
                    Code = ex.GetType().Name,
                    Description = ex.Message,
                    Level = MessageLevel.Error,
                    Origin = HIPS.CommonSchemas.Constants.Origin.HIPS,
                    Messages = ToMessageList(ex.InnerException)
                });
            }
            return list;
        }

        #endregion Methods
    }
}
