﻿using System;
using HIPS.AppServer.HIPSServiceHost.Services;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.HL7.Common;
using HIPS.HL7.Common.Exceptions;
using HIPS.CommonBusinessLogic.HL7;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.ServiceContracts.ServiceInterfaces;

namespace HIPS.AppServer.ServiceHost
{
    /// <summary>
    /// This class implements web services for processing HL7 messages from the Enterprise Service Bus / Health Integration Broker.
    /// </summary>
    public class DatabaseLoaderService : BaseService, IDatabaseLoaderService
    {
        // Useful for debugging to check that the message was all parsed properly
        private readonly bool IncludeOriginalMessageInReply = false;

        /// <summary>
        /// Notifies HIPS of an event in the Patient Administration System.
        /// </summary>
        /// <param name="messageForm">An HL7 message.</param>
        /// <param name="user">Details of the responsible user.</param>
        /// <returns>An HL7 acknowledgement message.</returns>
        public string NotifyPasEvent(string messageForm, UserDetails user)
        {
            string reply;
            MessageLoader loader = new MessageLoader();
            HL7Message parsedMessage;
            try
            {
                parsedMessage = HL7Message.Parse(messageForm);
            }
            catch (HL7ParseException ex)
            {
                string errorMessage = string.Format(ResponseStrings.UnparsableMessage, messageForm);
                EventLogger.WriteLog(errorMessage, ex, user, LogMessage.HIPS_MESSAGE_114);
                return HL7AcknowledgementFactory.AcknowledgeUnparseableMessage(ex.Message).Encode();
            }
            catch (Exception ex)
            {
                string errorMessage = string.Format(ResponseStrings.UnparsableMessage, messageForm);
                EventLogger.WriteLog(errorMessage, ex, user, LogMessage.HIPS_MESSAGE_115);
                return HL7AcknowledgementFactory.AcknowledgeUnparseableMessage(errorMessage).Encode();
            }
            HL7MessageLog testLog;
            HL7Message ack = loader.Process(parsedMessage, messageForm, user, out testLog);
            if (IncludeOriginalMessageInReply)
            {
                reply = parsedMessage.Encode() + "\r\r" + ack.Encode();
            }
            else
            {
                reply = ack.Encode();
            }
            return reply;
        }

        /// <summary>
        /// Notifies HIPS of a pathology result.
        /// </summary>
        /// <param name="messageForm">The message form.</param>
        /// <param name="user">Details of the responsible user.</param>
        /// <returns>An HL7 acknowledgement message.</returns>
        public string NotifyPathEvent(string messageForm, UserDetails user)
        {
            string reply;
            MessageLoader loader = new MessageLoader();
            HL7MessageLog testLog;
            HL7Message message = HL7Message.Parse(messageForm);
            HL7Message ack = loader.Process(message, messageForm, user, out testLog);
            if (IncludeOriginalMessageInReply)
            {
                reply = message.Encode() + "\r\r" + ack.Encode();
            }
            else
            {
                reply = ack.Encode();
            }
            return reply;
        }
    }
}