﻿#region Using Directives

using System;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.IhiSchemas.Schemas;
using HIPS.CommonBusinessLogic.Ihi;
using HIPS.ServiceContracts.ServiceInterfaces;
using HIPS.ServiceContracts.Common.DTO.ParticipatingIndividual;
using HIPS.AppServer.ServiceHost.Mapping;
using HIPS.Common.DataStore.DataAccess;
using HIPS.AppServer.HIPSServiceHost.Services;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.CommonSchemas.Exceptions;

#endregion Using Directives

namespace HIPS.AppServer.ServiceHost
{
    /// <summary>
    /// This class implements web services for obtaining a validated IHI from the HI Service.
    /// </summary>
    public class IHIService : BaseService, IIHIService
    {
        /// <summary>
        /// Obtains validated IHI information for use in a clinical document.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="dateOfBirth">Patient's date of birth (for identity verification)</param>
        /// <param name="user">The user responsible for this action</param>
        /// <returns>Validated IHI information</returns>
        public ValidatedIhiResponse GetValidatedIhi(PatientIdentifierBase patientIdentifier, DateTime dateOfBirth, UserDetails user)
        {
            return new PatientIhiValidation().GetValidatedIhi(patientIdentifier, dateOfBirth, user);
        }

        /// <summary>
        /// Obtains validated IHI information for use when sending an e-health message.
        /// </summary>
        /// <param name="patientIdentifier">Hospital-level (MRN) patient identifier.</param>
        /// <param name="dateOfBirth">Patient's date of birth (for identity verification)</param>
        /// <param name="user">The user responsible for this action</param>
        /// <returns>Validated IHI information</returns>
        public ValidatedIhiResponse ValidateSendingPatient(Mrn patientIdentifier, DateTime dateOfBirth, UserDetails user)
        {
            // CCA "Use of Healthcare Identifiers in Health Software Systems", UC.330:
            // As per mandatory CR 016832 "Validation of an IHI before inclusion in a new eHealth message/document",
            // an IHI is considered valid only for a maximum of 24 hours.

            // Retrieve the configured validity period (in days).
            // Determine the validity period in hours. If the configured period is less than 24 hours, we
            // make sure we use that value, otherwise the maximum we can use is 24 hours.
            var validityPeriodDays = PatientIhiValidation.GetIhiValidationPeriodDays();
            var validityPeriodHours = validityPeriodDays == 0 ? 0 : 24;

            // Utilise PatientIhiValidation.GetValidatedIhi to perform the validation.
            var response = new PatientIhiValidation(validityPeriodHours).GetValidatedIhi(patientIdentifier, dateOfBirth, user);
            if (response.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                string msg = string.Format(
                    ResponseStrings.ErrorUnableToValidateSendingPatient,
                    patientIdentifier,
                    patientIdentifier.HospitalCode);
                EventLogger.WriteLog(
                    msg,
                    new HipsResponseException(response.HipsResponse),
                    user,
                    LogMessage.HIPS_MESSAGE_191);
            }
            return response;
        }

        /// <summary>
        /// Obtains validated IHI information for use when receiving an e-health message.
        /// </summary>
        /// <param name="consumer">Participating Consumer</param>
        /// <param name="hpio">Health Provider Organisation Id</param>
        /// <param name="user">The user responsible for this action</param>
        /// <returns>Validated IHI information</returns>
        public ValidatedIhiResponse ValidateReceivingPatient(ParticipatingConsumer consumer, string hpio, UserDetails user)
        {
            var ihiValidation = new PatientIhiValidation();
            var modelConsumer = ObjectMapper.Map<HIPS.CommonSchemas.Cda.ParticipatingIndividual.ParticipatingConsumer>(consumer);
            var response = ihiValidation.CheckReceivedValidatedIhi(modelConsumer, hpio, user);
            if (response.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                string msg = string.Format(
                    ResponseStrings.ErrorUnableToValidateReceivingPatient,
                    consumer.Ihi,
                    consumer.FamilyName,
                    consumer.GivenNames);
                EventLogger.WriteLog(
                    msg,
                    new HipsResponseException(response.HipsResponse),
                    user,
                    LogMessage.HIPS_MESSAGE_190);
            }
            return response;
        }
    }
}