﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;

using HIPS.AppServer.ServiceHost.Mapping;
using HIPS.CommonBusinessLogic;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonBusinessLogic.Ihi;
using HIPS.CommonBusinessLogic.Pcehr;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Configuration;
using HIPS.PatientBusinessLogic;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.ServiceContracts.Common;
using HIPS.ServiceContracts.Common.Fault;
using HIPS.ServiceContracts.Patient;
using HIPS.ServiceContracts.Patient.Message;
using HIPS.CommonBusinessLogic.HL7;
using System.Data.SqlClient;

namespace HIPS.AppServer.ServiceHost
{
    /// <summary>
    /// Implementation of web service operations on patients in the HIPS database.
    /// </summary>
    public partial class PatientService : IPatientServiceV2
    {
        #region Methods

        /// <summary>
        /// Lists the episodes for a specific patient in a defined hospital
        /// </summary>
        /// <param name="request">Request containing a patient identifier for retrieving patient episodes.</param>
        /// <returns>Response containing the a list of the admitted or discharged episodes for the patient with in the specified number of days</returns>
        public ListPatientEpisodesInHospitalResponse ListPatientEpisodesInHospital(ListPatientEpisodesInHospitalRequest request)
        {
            this.ValidateRequest(request);

            try
            {
                // Map DTO to Common Model
                UserDetails user = ObjectMapper.Map<UserDetails>(request.User);
                var patientIdentifier = ObjectMapper.Map<HIPS.CommonSchemas.PatientIdentifier.PatientIdentifierBase>(request.PatientIdentifier);

                // Invoke Business Service
                List<PatientInHospitalEpisode> data;

                var logic = new PatientListLogic();
                var hipsResponse = logic.ListPatientEpisodesInHospital(user, patientIdentifier, request.DaysDischarged, out data, request.IncludeDocuments, request.DocumentTypeCode);
                var messages = new List<string>() { hipsResponse.HipsErrorMessage, hipsResponse.ResponseCodeDescription, hipsResponse.ResponseCodeDetails };

                // Map Common Model to DTO
                var serviceResult = ObjectMapper.Map<List<HIPS.ServiceContracts.Common.DTO.PatientEpisode>>(data);
                var serviceMessages = ObjectMapper.Map<List<ServiceContracts.Common.DTO.Message>>(messages.Where(message => !string.IsNullOrEmpty(message)));
                var serviceStatus = (hipsResponse.Status == HipsResponseIndicator.OK ? ServiceContracts.Common.ResponseStatus.OK : ServiceContracts.Common.ResponseStatus.Warning);

                return new ListPatientEpisodesInHospitalResponse() { Status = serviceStatus, Data = serviceResult, Messages = serviceMessages };
            }
            catch (Exception ex)
            {
                throw ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
            }
        }

        /// <summary>
        /// Allows for the creation of a patient within the eHISC system.
        /// </summary>
        /// <param name="request">Request containing a MRN, Demographic and validated IHI for creating a patient.</param>
        /// <returns>Register patient response.</returns>
        public RegisterPatientResponse RegisterPatient(RegisterPatientRequest request)
        {
            this.ValidateRequest(request);

            try
            {
                
                var user = ObjectMapper.Map<UserDetails>(request.User);
                var mrn = ObjectMapper.Map<Mrn>(request.Mrn);
                var demographic = ObjectMapper.Map<Demographic>(request.Demographics);
                var validatedIhi = ObjectMapper.Map<ValidatedIhi>(request.IHI);

                PatientMaster patientMaster;
                PatientMaster oldPatientMaster = null;
                Hospital hospital;
                HospitalPatient hospitalPatient;
                bool isExistingPatientMaster;
                bool isExistingHospitalPatient;

                PatientLoader patientLoader = new PatientLoader(user, null);
                patientLoader.FindOrCreatePatient(null, mrn, demographic, validatedIhi, out patientMaster, out hospital, out hospitalPatient, out oldPatientMaster, out isExistingPatientMaster, out isExistingHospitalPatient);

                if (!Settings.Instance.BypassHIService)
                {
                    if (!isExistingPatientMaster)
                    {
                        oldPatientMaster = patientMaster;

                        var patientRegistration = new PatientRegistration();
                        patientRegistration.RegisterPatient(patientMaster, hospital, user);
                    }

                    if (isExistingPatientMaster && isExistingHospitalPatient)
                    {
                        var patientIhiValidation = new PatientIhiValidation();
                        patientIhiValidation.RevalidateIhi(patientMaster, hospital, user);
                    }
                }
                else
                {
                    var patientDemographicUpdate = new PatientDemographicUpdate();
                    patientDemographicUpdate.UpdatePatient(oldPatientMaster, patientMaster, hospital, user);
                }

                if (validatedIhi != null && !string.IsNullOrEmpty(validatedIhi.Ihi))
                {
                    var doesPcehrExist = new DoesPcehrExist();
                    var result = doesPcehrExist.PcehrExists(mrn, hospital, patientMaster, user);

                    var messages = new List<string>() { result.HipsResponse.HipsErrorMessage, result.HipsResponse.ResponseCodeDescription, result.HipsResponse.ResponseCodeDetails };

                    // Map Common Model to DTO                    
                    var serviceMessages = ObjectMapper.Map<List<ServiceContracts.Common.DTO.Message>>(messages.Where(message => !string.IsNullOrEmpty(message)));
                    var serviceStatus = (result.HipsResponse.Status == HipsResponseIndicator.OK ? ServiceContracts.Common.ResponseStatus.OK : ServiceContracts.Common.ResponseStatus.Warning);

                    return new RegisterPatientResponse() { Status = serviceStatus, Messages = serviceMessages };
                }

                return new RegisterPatientResponse() { Status = ResponseStatus.OK };
            }
            catch (Exception ex)
            {
                throw ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
            }
      
        }

        #endregion Methods
    }
}