﻿#region Using Directives

using HIPS.CommonBusinessLogic;
using HIPS.CommonBusinessLogic.Ihi;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.IhiSchemas.Schemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas.Exceptions;
using HIPS.ServiceContracts.ServiceInterfaces;
using nehta.mcaR3.ConsumerSearchIHI;

#endregion Using Directives

namespace HIPS.AppServer.ServiceHost
{
    /// <summary>
    /// This class implements specialised web services that are used by the PUMA application.
    /// </summary>
    public class PumaInterfaceService : BaseService, IPumaInterfaceService
    {
        #region Methods

        /// <summary>
        ///  Search for an IHI by number, but do not check to see if the PCEHR exists.
        /// </summary>
        /// <param name="hospitalId">The hospital ID.</param>
        /// <param name="searchDetails">The search details.</param>
        /// <param name="user">Information to identify the person responsible for this action.</param>
        /// <returns>HIPS IHI Response.</returns>
        public HipsIhiResponse IHISearchByNumber(int hospitalId, IhiSearchCriteria searchDetails, UserDetails user)
        {
            return IhiSearchByNumber(hospitalId, searchDetails, user, false);
        }

        /// <summary>
        /// Search for an IHI by number, but DO check to see if the PCEHR exists.
        /// </summary>
        /// <param name="hospitalId">The hospital ID.</param>
        /// <param name="searchDetails">The search details.</param>
        /// <param name="user">Information to identify the person responsible for this action.</param>
        /// <returns>HIPS IHI Response.</returns>
        public HipsIhiResponse IHISearchByNumberAndPCEHRCheck(int hospitalId, IhiSearchCriteria searchDetails, UserDetails user)
        {
            return IhiSearchByNumber(hospitalId, searchDetails, user, true);
        }

        /// <summary>
        /// Search for an IHI by number and if need be validate that the PCEHR exists.
        /// </summary>
        /// <param name="searchDetails">The search details.</param>
        /// <param name="user">Information to identify the person responsible for this action.</param>
        /// <param name="checkPcehr">if set to <c>true</c> [check PCEHR].</param>
        /// <returns>HIPS IHI Response.</returns>
        private HipsIhiResponse IhiSearchByNumber(int hospitalId, IhiSearchCriteria searchDetails, UserDetails user, bool checkPcehr)
        {
            Hospital hospital = HospitalSingleton.Value.Find(hospitalId);
            string resultMessage = null;
            HipsIhiResponse serviceResponse = new HipsIhiResponse();
            searchIHIResponse response;
            PatientMaster patientMaster;

            // Can't use data layer directly from the service host
            // because it requires a reference to System.Data for the
            // SqlTransaction parameter.
            //PatientMasterDl dataAccess = new PatientMasterDl(user);
            //dataAccess.Get(searchDetails.PatientMasterId, out patientMaster);

            PatientAccess patientAccess = new PatientAccess(user);
            HospitalPatient hospitalPatient;
            patientAccess.GetPatient(new PatientMasterId(searchDetails.PatientMasterId, null, null, hospitalId), hospital, out hospitalPatient, out patientMaster);

            try
            {
                bool success = Search.ByNumber(hospital, patientMaster, searchDetails, out resultMessage, out response, user, checkPcehr);
            }
            catch (PcehrServiceUnavailableException)
            {
                searchDetails.Ihi = patientMaster.Ihi;
                searchDetails.PcehrExists = null;
                searchDetails.ReturnedIhiRecordStatus = (IhiRecordStatus)patientMaster.IhiRecordStatusId;
                searchDetails.ReturnedIhiStatus = (IhiStatus)patientMaster.IhiStatusId;
                resultMessage = HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.WarningIhiFoundButPcehrUnavailable;

                // It would be a lot of work to recreate the full response - why return this to PUMA anyway?
                response = new searchIHIResponse();
                response.searchIHIResult = new searchIHIResult();
            }

            serviceResponse.SearchIHIResponse = response;

            if (response != null && response.searchIHIResult != null)
            {
                serviceResponse.IhiNumber = searchDetails.Ihi;
                serviceResponse.PcehrExists = searchDetails.PcehrExists;
                serviceResponse.ReturnedIhiRecordStatus = searchDetails.ReturnedIhiRecordStatus;
                serviceResponse.ReturnedIhiStatus = searchDetails.ReturnedIhiStatus;
            }

            serviceResponse.ReturnMessage = resultMessage;

            return serviceResponse;
        }

        #endregion Methods
    }
}