﻿using System;
using System.ServiceModel;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.ConsentSchemas;
using HIPS.ServiceContracts.ServiceInterfaces;

namespace HIPS.Client.Proxy
{
    public class ConsentProxy : ClientBase<IConsentService>, IConsentService
    {
        /// <summary>
        /// Default constructor
        /// </summary>
        /// <param name="endpointName"></param>
        public ConsentProxy(string endpointName)
            : base(endpointName)
        {
        }

        /// <summary>
        /// Finds out whether HIPS has recorded an explicit request to upload or a withdrawal of consent to upload
        /// clinical documents for a specified episode.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="admissionDate">The date and time when the episode started (Episode.AdmissionDate)</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <returns>Response containing the consent status for this episode and an error indicator</returns>
        public EpisodeConsentResponse CheckConsent(PatientIdentifierBase patientIdentifier, DateTime admissionDate, UserDetails user)
        {
            return Channel.CheckConsent(patientIdentifier, admissionDate, user);
        }

        /// <summary>
        /// Gets the participation status for a single hospital patient record. This involves
        /// checking both the HI Service and the PCEHR System, as well as local consent records.
        /// <list type="ordered">
        /// <item>If we do not have a valid IHI then the patient is not considered participating.
        ///       We cannot upload a discharge summary for a patient without a valid IHI.</item>
        /// <item>If the patient has explicitly consented to the upload of any of the discharge
        ///       summaries from a hospital then they are considered participating in that hospital
        ///       only.</item>
        /// <item>If the patient has an advertised PCEHR, then we assume they do want their discharge
        ///       summary to be uploaded to PCEHR.</item>
        /// <item>With no advertised PCEHR and no explicit consents in the hospital, we assume the
        ///       patient will not expect that hospital to upload the discharge summary.</item>
        ///    </list>
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">The user responsible for any IHI and PCEHR checks required</param>
        /// <returns>Participation status for the requested patient</returns>
        public PatientParticipationResponse GetPatientParticipationStatus(PatientIdentifierBase patientIdentifier, UserDetails user)
        {
            return Channel.GetPatientParticipationStatus(patientIdentifier, user);
        }

        /// <summary>
        /// For patients in hospitals that have a code in the given hospital code system (e.g. those which use a particular PAS or CIS),
        /// This service will obtain the participation status for all patients with changes to their episodes since the given date/time.
        /// </summary>
        /// <param name="since">The date/time after which the episode must have been modified to be included</param>
        /// <param name="hospitalCodeSystem">Code that identifies which type of hospital codes should be returned</param>
        /// <param name="user">The user details for IHI validation and PCEHR advertised checking.</param>
        /// <returns>List of participation status for patients</returns>
        public PatientParticipationResponse GetRecentPatientParticipationStatus(DateTime since, string hospitalCodeSystem, UserDetails user)
        {
            return Channel.GetRecentPatientParticipationStatus(since, hospitalCodeSystem, user);
        }

        /// <summary>
        /// Record the withdrawal of consent to upload a discharge summary for a particular episode, or reverse that action.
        /// In order to check or record the consent to upload the discharge summary to PCEHR for an episode,
        /// the clinical system must specify the admission date/time of the episode within a configured tolerance.
        /// If there is no unique match to an episode then an error will be returned and the situation
        /// may be investigated by the system administrator.
        /// </summary>
        /// <param name="consentWithdrawn">True if the patient withdrew consent to upload, false if the patient reinstated consent to upload.</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="admissionDate">The date and time when the episode started (Episode.AdmissionDate)</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="auditInformation">Additional audit information to store</param>
        /// <returns>Indicator of any error that occurred</returns>
        public HipsResponse RecordConsent(bool consentWithdrawn, PatientIdentifierBase patientIdentifier, DateTime admissionDate, UserDetails user, byte[] auditInformation)
        {
            return Channel.RecordConsent(consentWithdrawn, patientIdentifier, admissionDate, user, auditInformation);
        }

        /// <summary>
        /// Record whether the patient has disclosed the existence of a PCEHR.
        /// </summary>
        /// <param name="pcehrDisclosed">True if the patient disclosed the existence of a PCEHR, false if the patient has not disclosed the existence of a PCEHR.</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="auditInformation">Additional audit information to store</param>
        /// <returns>Indicator of any error that occurred</returns>
        public HipsResponse RecordDisclosure(bool pcehrDisclosed, PatientIdentifierBase patientIdentifier, UserDetails user, byte[] auditInformation)
        {
            return Channel.RecordDisclosure(pcehrDisclosed, patientIdentifier, user, auditInformation);
        }
    }
}