﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using AutoMapper;
using ModelSchemas = HIPS.CommonSchemas;
using NehtaSchemas = Nehta.VendorLibrary.CDA;

namespace HIPS.CommonBusinessLogic.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for the Diagnostic Imaging.
    /// </summary>
    internal class DiagnosticImagingProfile : Profile
    {
        /// <summary>
        /// Returns the name of the mapping profile.
        /// </summary>
        public override string ProfileName
        {
            get { return this.GetType().Name; }
        }
        
        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            // Model --> NEHTA
            this.CreateMap<ModelSchemas.Cda.CdaPathDIDocumentDetails, NehtaSchemas.Common.DiagnosticImagingReport>()
                .ForMember(dest => dest.CDAContext, opt => opt.MapFrom(src => src.CdaDocumentDetails))
                .ForMember(dest => dest.DocumentCreationTime, opt => opt.MapFrom(src => src.CdaDocumentDetails.Metadata.DocumentCreationDateTime))
                .ForMember(
                    dest => dest.DocumentStatus,
                    opt => opt.MapFrom(
                        src => src.CdaDocumentDetails.Metadata.SourceDocumentStatus == ModelSchemas.Cda.SourceDocumentStatus.Final
                            ? NehtaSchemas.Generator.Enums.DocumentStatus.Final
                            : src.CdaDocumentDetails.Metadata.SourceDocumentStatus == ModelSchemas.Cda.SourceDocumentStatus.Interim
                            ? NehtaSchemas.Generator.Enums.DocumentStatus.Interim
                            : src.CdaDocumentDetails.Metadata.SourceDocumentStatus == ModelSchemas.Cda.SourceDocumentStatus.Withdrawn
                            ? NehtaSchemas.Generator.Enums.DocumentStatus.Withdrawn
                            : NehtaSchemas.Generator.Enums.DocumentStatus.Final))
                .ForMember(dest => dest.IncludeLogo, opt => opt.MapFrom(src => src.CdaDocumentDetails.Attachments.Exists(a => a.AttachmentType == ModelSchemas.Cda.AttachmentType.OrganisationalLogoPng)))
                .ForMember(dest => dest.LogoPath, opt => opt.MapFrom(src => src.CdaDocumentDetails.Attachments.SingleOrDefault(a => a.AttachmentType == ModelSchemas.Cda.AttachmentType.OrganisationalLogoPng).FileName))
                .ForMember(dest => dest.LogoContent, opt => opt.MapFrom(src => src.CdaDocumentDetails.Attachments.SingleOrDefault(a => a.AttachmentType == ModelSchemas.Cda.AttachmentType.OrganisationalLogoPng).Content))
                .ForMember(dest => dest.SCSContent, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.SCSContext, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.ShowAdministrativeObservationsSection, opt => opt.UseValue(false));

      
            // CDAContext
            // Mapping for Custodian: Facility profile (Diagnostic Imaging - CDAContext - Custodian)
            // Mapping for LegalAuthenticator: Provider profile (Diagnostic Imaging - CDAContext - LegalAuthenticator)
            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.CDAModel.ICDAContextDiagnosticImagingReport>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreateCDAContext())
                .ForMember(dest => dest.DocumentId, opt => opt.MapFrom(src => src.DocumentId))
                .ForMember(dest => dest.SetId, opt => opt.MapFrom(src => src.DocumentSetId))
                .ForMember(dest => dest.Version, opt => opt.MapFrom(src => src.Version))
                .ForMember(dest => dest.Custodian, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.LegalAuthenticator, opt => opt.MapFrom(src => src));
            
            // SCSContext
            // Mapping for Author: Provider profile (Diagnostic Imaging - SCSContext - Author).
            // Mapping for ReportingRadiologist: Provider profile (Diagnostic Imaging - SCSContext - ReportingRadiologist)
            // Mapping for SubjectOfCare: SubjectOfCare profile (Diagnostic Imaging - SCSContext - SubjectOfCare)
            // Mapping for OrderDetails: OrderDetails profile (Diagnostic Imaging - SCSContext - OrderDetails)
            this.CreateMap<ModelSchemas.Cda.CdaPathDIDocumentDetails, NehtaSchemas.SCSModel.IDiagnosticImagingReportContext>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreateSCSContext())
                .ForMember(dest => dest.Author, opt => opt.MapFrom(src => src.CdaDocumentDetails))
                .ForMember(dest => dest.ReportingRadiologist, opt => opt.MapFrom(src => src.CdaDocumentDetails))
                .ForMember(dest => dest.SubjectOfCare, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.OrderDetails, opt => opt.MapFrom(src => src));
            
            // SCSContent
            // Mapping for ImagingExaminationResults: ImagingExamResult profile (Diagnostic Imaging - SCSContent - ImagingExaminationResults)
            // Mapping for RelatedDocument: RelatedDocument profile (Diagnostic Imaging - SCSContent - RelatedDocument)
            this.CreateMap<ModelSchemas.Cda.CdaPathDIDocumentDetails, NehtaSchemas.SCSModel.Pathology.IDiagnosticImagingReportContent>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreateSCSContent())
                .ForMember(dest => dest.ImagingExaminationResults, opt => opt.MapFrom(src => src.ImagingExamResult))
                .ForMember(dest => dest.RelatedDocument, opt => opt.MapFrom(src => src));
        }
    }
}
