﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using AutoMapper;
using ModelSchemas = HIPS.CommonSchemas;
using NehtaSchemas = Nehta.VendorLibrary.CDA;

namespace HIPS.CommonBusinessLogic.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for the Order Details.
    /// </summary>
    internal class OrderDetailsProfile : Profile
    {
        /// <summary>
        /// Returns the name of the mapping profile.
        /// </summary>
        public override string ProfileName
        {
            get { return this.GetType().Name; }
        }
        
        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            // Model --> NEHTA
            
            #region Diagnostic Imaging
            
            // Diagnostic Imaging - SCSContext - OrderDetails
            this.CreateMap<ModelSchemas.Cda.CdaPathDIDocumentDetails, NehtaSchemas.SCSModel.Pathology.OrderDetails>()
                .ForMember(dest => dest.RequesterOrderIdentifier, opt => opt.MapFrom(src => NehtaSchemas.Common.DiagnosticImagingReport.CreateRequesterOrderIdentifier(src.CdaDocumentDetails.Hospital.HpiO, src.OrderDetails.OrderIdentifier)))
                .ForMember(dest => dest.AccessionNumber, opt => opt.MapFrom(src => (src.ImagingExamResult.Count > 0 ? NehtaSchemas.Common.DiagnosticImagingReport.CreateAccessionNumber(src.CdaDocumentDetails.Hospital.HpiO, src.OrderDetails.DiagnosticImagingAccessionNumber) : null)))
                .ForMember(dest => dest.Requester, opt => opt.MapFrom(src => src));

            // Diagnostic Imaging - SCSContext - OrderDetails - Requester
            this.CreateMap<ModelSchemas.Cda.CdaPathDIDocumentDetails, NehtaSchemas.SCSModel.IParticipationRequester>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreateRequester())
                .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src.OrderDetails.Requester))
                .ForMember(dest => dest.ParticipationEndTime, opt => opt.MapFrom(src => src.CdaDocumentDetails.Metadata.DocumentCreationDateTime))
                .ForMember(dest => dest.Role, opt => opt.MapFrom(src => new ModelSchemas.Cda.CdaCode()
                {
                    Code = "253111",
                    CodeSystem = "2.16.840.1.113883.13.62",
                    CodeSystemName = "1220.0 – ANZSCO – Australian and New Zealand Standard Classification of Occupations. First Edition, Revision 1",
                    DisplayName = "General Medical Practitioner"
                }));

            // Diagnostic Imaging - SCSContext - OrderDetails - Requester - Participant
            // Mapping for Person: Diagnostic Imaging - SCSContext - OrderDetails - Requester - Participant - Person
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.SCSModel.IRequester>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreateParticipantForRequester())
                .ForMember(dest => dest.UniqueIdentifier, opt => opt.MapFrom(src => Guid.NewGuid()))
                .ForMember(dest => dest.Person, opt => opt.MapFrom(src => src));

            #endregion

            #region Pathology
            
            // Pathology - SCSContext - OrderDetails
            this.CreateMap<ModelSchemas.Cda.CdaPathDIDocumentDetails, NehtaSchemas.SCSModel.Pathology.OrderDetails>()                
                .ForMember(dest => dest.Requester, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.RequesterOrderIdentifier, opt => opt.MapFrom(src =>  new ModelSchemas.Cda.EntityIdentifier()
                {
                    Root = string.Format("1.2.36.1.2001.1005.52.{0}", src.CdaDocumentDetails.Hospital.HpiO),                    
                    Extension = src.OrderDetails.OrderIdentifier
                }));

            // Pathology - SCSContext - OrderDetails - Requester
            this.CreateMap<ModelSchemas.Cda.CdaPathDIDocumentDetails, NehtaSchemas.SCSModel.IParticipationRequester>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.PathologyResultReport.CreateRequester())
                .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src.OrderDetails.Requester))
                .ForMember(dest => dest.ParticipationEndTime, opt => opt.MapFrom(src => src.CdaDocumentDetails.Metadata.DocumentCreationDateTime))
                .ForMember(dest => dest.Role, opt => opt.MapFrom(src => new ModelSchemas.Cda.CdaCode()
                {
                    Code = "253111",
                    CodeSystem = "2.16.840.1.113883.13.62",
                    CodeSystemName = "1220.0 – ANZSCO – Australian and New Zealand Standard Classification of Occupations. First Edition, Revision 1",
                    DisplayName = "General Medical Practitioner"
                }));

            // Pathology - SCSContext - OrderDetails - Requester - Participant
            // Mapping for Person: Pathology - SCSContext - OrderDetails - Requester - Participant - Person
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.SCSModel.IRequester>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.PathologyResultReport.CreateParticipantForRequester())
                .ForMember(dest => dest.UniqueIdentifier, opt => opt.MapFrom(src => Guid.NewGuid()))
                .ForMember(dest => dest.Person, opt => opt.MapFrom(src => src));
            
            #endregion
        }
    }
}
