﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using AutoMapper;
using HIPS.PcehrDataStore.Schemas.Enumerators;

using ModelSchemas = HIPS.CommonSchemas;
using NehtaSchemas = Nehta.VendorLibrary.CDA;

namespace HIPS.CommonBusinessLogic.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for the Person.
    /// </summary>
    internal class PersonProfile : Profile
    {
        /// <summary>
        /// Returns the name of the mapping profile.
        /// </summary>
        public override string ProfileName
        {
            get { return this.GetType().Name; }
        }
        
        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            #region Discharge Summary

            // Discharge Summary - CDAContext - LegalAuthenticator - Participant - Person
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.SCSModel.IPerson>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreatePerson())
                .ForMember(dest => dest.Identifiers, opt => opt.MapFrom(src => ObjectMapper.Map<List<ModelSchemas.Cda.EntityIdentifier>>(src)))
                .ForMember(dest => dest.PersonNames, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider>() { src }));

            // Discharge Summary - CDAContext - LegalAuthenticator - Participant - Person - PersonNames            
            // Discharge Summary - SCSContext - Author - Participant - Person - PersonNames
            // Discharge Summary - SCSContent - Event - Encounter - ResponsibleHealthProfessional - Participant - Person - PersonNames
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.IPersonName>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreatePersonName())
                .ForMember(dest => dest.FamilyName, opt => opt.MapFrom(src => src.FamilyName))
                .ForMember(dest => dest.GivenNames, opt => opt.MapFrom(src => src.GivenNames.Split(' ')))
                .ForMember(dest => dest.NameSuffix, opt => opt.MapFrom(src => src.Suffix.Split(' ')))
                .ForMember(dest => dest.Titles, opt => opt.MapFrom(src => (src.Title.Length > 0 ? src.Title.Split(' ') : null)));

            // Discharge Summary - SCSContext - Author - Participant - Person
            // Discharge Summary - SCSContent - Event - Encounter - ResponsibleHealthProfessional - Participant - Person
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.SCSModel.IPersonWithOrganisation>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreatePersonWithOrganisation())
                .ForMember(dest => dest.PersonNames, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider>() { src }))
                .ForMember(dest => dest.Identifiers, opt => opt.MapFrom(src => ObjectMapper.Map<List<ModelSchemas.Cda.EntityIdentifier>>(src)))
                .ForMember(dest => dest.Organisation, opt => opt.MapFrom(src => src));

            // Discharge Summary - SCSContent - Event - Encounter - ResponsibleHealthProfessional - Person - Organisation
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.IEmploymentOrganisation>()
               .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateEmploymentOrganisation())
               .ForMember(dest => dest.Name, opt => opt.MapFrom(src => src.EmployerName))
               .ForMember(dest => dest.Identifiers, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.EntityIdentifier>()
               {
                   new ModelSchemas.Cda.EntityIdentifier
                   {
                       AssigningAuthorityName = "HPI-O",
                       AssigningGeographicArea = "National Identifier",
                       Root = string.Format("1.2.36.1.2001.1003.0.{0}", src.EmployerHpio),
                   }
               }));

            // Discharge Summary - SCSContext - SubjectOfCare - Participant - Person
            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.IPersonSubjectOfCare>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreatePersonForSubjectOfCare())
                .ForMember(dest => dest.DateOfBirth, opt => opt.MapFrom(src => src.PatientMaster.RegisteredDateOfBirth))
                .ForMember(dest => dest.DateOfBirthIncludesTime, opt => opt.MapFrom(src => false))
                .ForMember(dest => dest.Gender, opt => opt.MapFrom(src => (SexEnumerator)src.PatientMaster.CurrentSexId))
                .ForMember(dest => dest.PersonNames, opt => opt.MapFrom(src => src.PatientNames))
                .ForMember(dest => dest.Identifiers, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.EntityIdentifier>()
                {
                    new ModelSchemas.Cda.EntityIdentifier
                    {
                        AssigningAuthorityName = "IHI",
                        AssigningGeographicArea = "National Identifier",
                        Root = string.Format("1.2.36.1.2001.1003.0.{0}", src.PatientMaster.Ihi)
                    },
                    new ModelSchemas.Cda.EntityIdentifier
                    {
                        AssigningAuthorityName = src.Hospital.Name,
                        Root = string.Format("1.2.36.1.2001.1005.49.3.{0}.{1}", src.Hospital.HpiO, src.Hospital.HospitalId),
                        Extension = src.HospitalPatient.Mrn,
                        Code = new ModelSchemas.Cda.CdaCode
                        {
                            Code = "MR",
                            CodeSystem = "2.16.840.1.113883.12.203",
                            CodeSystemName = "Identifier Type (HL7)"
                        }
                    },
                    new ModelSchemas.Cda.EntityIdentifier
                    {
                        AssigningAuthorityName = "Medicare card number",
                        Root = string.IsNullOrEmpty(src.PatientMaster.MedicareIrn) ? "1.2.36.1.5001.1.0.7.1" : "1.2.36.1.5001.1.0.7",
                        Extension = src.PatientMaster.MedicareNumber + src.PatientMaster.MedicareIrn,
                        Code = new ModelSchemas.Cda.CdaCode
                        {
                            Code = "MC",
                            CodeSystem = "2.16.840.1.113883.12.203",
                            CodeSystemName = "Identifier Type (HL7)"
                        }
                    },
                }.Where(a => a.AssigningAuthorityName == "IHI" || !string.IsNullOrEmpty(a.Extension))));

            // Discharge Summary - SCSContext - SubjectOfCare - Participant - Person - PersonNames
            this.CreateMap<ModelSchemas.Cda.IndividualName, NehtaSchemas.IPersonName>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreatePersonName())
                .ForMember(dest => dest.FamilyName, opt => opt.MapFrom(src => src.FamilyName))
                .ForMember(dest => dest.GivenNames, opt => opt.MapFrom(src => src.GivenNames))
                .ForMember(dest => dest.NameSuffix, opt => opt.MapFrom(src => src.Suffixes))
                .ForMember(dest => dest.Titles, opt => opt.MapFrom(src => src.Titles))
                .ForMember(dest => dest.NameUsages, opt => opt.ResolveUsing((ModelSchemas.Cda.IndividualName src) =>
                {
                    var list = new List<NehtaSchemas.Common.Enums.NameUsage>();
                  
                    switch (src.Usage)
                    {
                        case ModelSchemas.Cda.IndividualNameUsage.MaidenName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.MaidenName);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.NewbornName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.NewbornName);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.OtherName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.OtherName);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.ProfessionalName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.ProfessionalOrBusinessName);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.RegisteredName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.Legal);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.ReportingName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.ReportingName);
                            break;
                    }

                    return list;
                }));
            
            #endregion

            #region Diagnostic Imaging

            // Diagnostic Imaging - CDAContext - LegalAuthenticator - Participant - Person
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.SCSModel.IPerson>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreatePerson())
                .ForMember(dest => dest.Identifiers, opt => opt.MapFrom(src => ObjectMapper.Map<List<ModelSchemas.Cda.EntityIdentifier>>(src)))
                .ForMember(dest => dest.PersonNames, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider>() { src }));
      
            // Diagnostic Imaging - SCSContext - Author - Participant - Person
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, CDA.Generator.Common.SCSModel.Interfaces.IPersonHealthcareProvider>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreatePersonHealthcareProvider())
                .ForMember(dest => dest.PersonNames, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider>() { src }))                
                .ForMember(dest => dest.Identifiers, opt => opt.MapFrom(src => ObjectMapper.Map<List<ModelSchemas.Cda.EntityIdentifier>>(src)))
                .ForMember(dest => dest.Organisation, opt => opt.MapFrom(src => src));                   

            // Diagnostic Imaging - SCSContext - SubjectOfCare - Participant - Person
            this.CreateMap<ModelSchemas.Cda.CdaPathDIDocumentDetails, NehtaSchemas.IPersonSubjectOfCare>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreatePersonForSubjectOfCare())
                .ForMember(dest => dest.DateOfBirth, opt => opt.MapFrom(src => src.CdaDocumentDetails.PatientMaster.RegisteredDateOfBirth))
                .ForMember(dest => dest.DateOfBirthIncludesTime, opt => opt.UseValue(false))
                .ForMember(dest => dest.Gender, opt => opt.MapFrom(src => (SexEnumerator)src.CdaDocumentDetails.PatientMaster.CurrentSexId))
                .ForMember(dest => dest.PersonNames, opt => opt.MapFrom(src => src.CdaDocumentDetails.PatientNames))
                .ForMember(dest => dest.IndigenousStatus, opt => opt.ResolveUsing<IndigenousStatusResolver>().FromMember(src => src.CdaDocumentDetails.PatientIndigenousStatus))
                .ForMember(dest => dest.Identifiers, opt => opt.ResolveUsing<IdentifierResolver>().FromMember(src => src));

            // Diagnostic Imaging - SCSContext - SubjectOfCare - Participant - Person - PersonNames
            this.CreateMap<ModelSchemas.Cda.IndividualName, NehtaSchemas.IPersonName>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreatePersonName())
                .ForMember(dest => dest.FamilyName, opt => opt.MapFrom(src => src.FamilyName))
                .ForMember(dest => dest.GivenNames, opt => opt.MapFrom(src => src.GivenNames))
                .ForMember(dest => dest.NameSuffix, opt => opt.MapFrom(src => src.Suffixes))
                .ForMember(dest => dest.Titles, opt => opt.MapFrom(src => (src.Titles.Count > 0 ? src.Titles : null)))
                .ForMember(dest => dest.NameUsages, opt => opt.ResolveUsing((ModelSchemas.Cda.IndividualName src) =>
                {
                    var list = new List<NehtaSchemas.Common.Enums.NameUsage>();
                   
                    switch (src.Usage)
                    {
                        case ModelSchemas.Cda.IndividualNameUsage.MaidenName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.MaidenName);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.NewbornName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.NewbornName);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.OtherName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.OtherName);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.ProfessionalName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.ProfessionalOrBusinessName);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.RegisteredName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.Legal);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.ReportingName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.ReportingName);
                            break;
                    }

                    return list;
                }));

            // Diagnostic Imaging - SCSContext - OrderDetails - Participant - Person
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.SCSModel.IPersonWithOrganisation>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreatePersonWithOrganisation())
                .ForMember(dest => dest.PersonNames, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider>() { src }))                
                .ForMember(dest => dest.Identifiers, opt => opt.MapFrom(src => ObjectMapper.Map<List<ModelSchemas.Cda.EntityIdentifier>>(src)))
                .ForMember(dest => dest.Organisation, opt => opt.MapFrom(src => src));

            // Diagnostic Imaging - SCSContext - ReportingRadiologist - Participant - Person
            // Diagnostic Imaging - SCSContext - OrderDetails - Requester - Participant - Person
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.SCSModel.IPersonWithOrganisation>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreatePersonWithOrganisation())
                .ForMember(dest => dest.PersonNames, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider>() { src }))
                .ForMember(dest => dest.Identifiers, opt => opt.MapFrom(src => ObjectMapper.Map<List<ModelSchemas.Cda.EntityIdentifier>>(src)))
                .ForMember(dest => dest.Organisation, opt => opt.MapFrom(src => src));

            // Diagnostic Imaging - CDAContext - LegalAuthenticator - Participant - PersonNames
            // Diagnostic Imaging - SCSContext - Author - Participant - Person - PersonNames
            // Diagnostic Imaging - SCSContext - ReportingRadiologist - Participant - PersonNames
            // Diagnostic Imaging - SCSContext - OrderDetails - Participant - Person - PersonNames
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.IPersonName>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreatePersonName())
                .ForMember(dest => dest.FamilyName, opt => opt.MapFrom(src => src.FamilyName))
                .ForMember(dest => dest.GivenNames, opt => opt.MapFrom(src => src.GivenNames.Split(' ')))
                .ForMember(dest => dest.NameSuffix, opt => opt.MapFrom(src => src.Suffix.Split(' ')))
                .ForMember(dest => dest.Titles, opt => opt.MapFrom(src => (src.Title.Length > 0 ? src.Title.Split(' ') : null)));

            // Diagnostic Imaging - SCSContext - Author - Participant - Person - Organisation
            // Diagnostic Imaging - SCSContext - ReportingRadiologist - Participant - Organisation
            // Diagnostic Imaging - SCSContext - OrderDetails - Requester - Participant - Person - Organisation
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.IEmploymentOrganisation>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreateEmploymentOrganisation())
                .ForMember(dest => dest.Name, opt => opt.MapFrom(src => src.EmployerName))
                .ForMember(dest => dest.Identifiers, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.EntityIdentifier>()
                {
                    new ModelSchemas.Cda.EntityIdentifier
                    {
                        AssigningAuthorityName = "HPI-O",
                        AssigningGeographicArea = "National Identifier",
                        Root = string.Format("1.2.36.1.2001.1003.0.{0}", src.EmployerHpio)
                    }
                }));

            #endregion

            #region Pathology

            // Pathology - CDAContext - LegalAuthenticator - Participant - Person
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.SCSModel.IPerson>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.PathologyResultReport.CreatePerson())
                .ForMember(dest => dest.Identifiers, opt => opt.MapFrom(src => ObjectMapper.Map<List<ModelSchemas.Cda.EntityIdentifier>>(src)))
                .ForMember(dest => dest.PersonNames, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider>() { src }));

            // Pathology - SCSContext - Author - Participant - Person
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, CDA.Generator.Common.SCSModel.Interfaces.IPersonHealthcareProvider>()
               .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.PathologyResultReport.CreatePersonHealthcareProvider())
                .ForMember(dest => dest.PersonNames, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider>() { src }))
                .ForMember(dest => dest.Identifiers, opt => opt.MapFrom(src => ObjectMapper.Map<List<ModelSchemas.Cda.EntityIdentifier>>(src)))
                .ForMember(dest => dest.Organisation, opt => opt.MapFrom(src => src));     

            // Pathology - SCSContext - SubjectOfCare - Participant - Person
            this.CreateMap<ModelSchemas.Cda.CdaPathDIDocumentDetails, NehtaSchemas.IPersonSubjectOfCare>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.PathologyResultReport.CreatePersonForSubjectOfCare())
                .ForMember(dest => dest.DateOfBirth, opt => opt.MapFrom(src => src.CdaDocumentDetails.PatientMaster.RegisteredDateOfBirth))
                .ForMember(dest => dest.DateOfBirthIncludesTime, opt => opt.UseValue(false))
                .ForMember(dest => dest.Gender, opt => opt.MapFrom(src => (SexEnumerator)src.CdaDocumentDetails.PatientMaster.CurrentSexId))
                .ForMember(dest => dest.PersonNames, opt => opt.MapFrom(src => src.CdaDocumentDetails.PatientNames))
                .ForMember(dest => dest.IndigenousStatus, opt => opt.ResolveUsing<IndigenousStatusResolver>().FromMember(src => src.CdaDocumentDetails.PatientIndigenousStatus))
                .ForMember(dest => dest.Identifiers, opt => opt.ResolveUsing<IdentifierResolver>().FromMember(src => src));

            // Pathology - SCSContext - SubjectOfCare - Participant - Person - PersonNames
            this.CreateMap<ModelSchemas.Cda.IndividualName, NehtaSchemas.IPersonName>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.PathologyResultReport.CreatePersonName())
                .ForMember(dest => dest.FamilyName, opt => opt.MapFrom(src => src.FamilyName))
                .ForMember(dest => dest.GivenNames, opt => opt.MapFrom(src => src.GivenNames))
                .ForMember(dest => dest.NameSuffix, opt => opt.MapFrom(src => src.Suffixes))
                .ForMember(dest => dest.Titles, opt => opt.MapFrom(src => src.Titles))
                .ForMember(dest => dest.NameUsages, opt => opt.ResolveUsing((ModelSchemas.Cda.IndividualName src) =>
                {
                    var list = new List<NehtaSchemas.Common.Enums.NameUsage>();
                    
                    switch (src.Usage)
                    {
                        case ModelSchemas.Cda.IndividualNameUsage.MaidenName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.MaidenName);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.NewbornName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.NewbornName);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.OtherName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.OtherName);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.ProfessionalName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.ProfessionalOrBusinessName);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.RegisteredName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.Legal);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.ReportingName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.ReportingName);
                            break;
                    }

                    return list;
                }));

            // Pathology - SCSContext - ReportingPathologist - Participant - Person
            // Pathology - SCSContext - OrderDetails - Requester - Participant - Person
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.SCSModel.IPersonWithOrganisation>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.PathologyResultReport.CreatePersonWithOrganisation())
                .ForMember(dest => dest.PersonNames, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider>() { src }))
                .ForMember(dest => dest.Identifiers, opt => opt.MapFrom(src => ObjectMapper.Map<List<ModelSchemas.Cda.EntityIdentifier>>(src)))
                .ForMember(dest => dest.Organisation, opt => opt.MapFrom(src => src));

            // Pathology - CDAContext - LegalAuthenticator - Participant - PersonNames
            // Pathology - SCSContext - Author - Participant - Person - PersonNames
            // Pathology - SCSContext - ReportingPathologist - Participant - PersonNames
            // Pathology - SCSContext - OrderDetails - Requester - Participant - Person - PersonNames
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.IPersonName>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.PathologyResultReport.CreatePersonName())
                .ForMember(dest => dest.FamilyName, opt => opt.MapFrom(src => src.FamilyName))
                .ForMember(dest => dest.GivenNames, opt => opt.MapFrom(src => src.GivenNames.Split(' ')))
                .ForMember(dest => dest.NameSuffix, opt => opt.MapFrom(src => src.Suffix.Split(' ')))
                .ForMember(dest => dest.Titles, opt => opt.MapFrom(src => src.Title.Split(' ')));

            // Pathology - SCSContext - Author - Participant - Person - Organisation
            // Pathology - SCSContext - ReportingPathologist - Participant - Organisation
            // Pathology - SCSContext - OrderDetails - Requester - Participant - Organisation
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.IEmploymentOrganisation>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.PathologyResultReport.CreateEmploymentOrganisation())
                .ForMember(dest => dest.Name, opt => opt.MapFrom(src => src.EmployerName))
                .ForMember(dest => dest.Identifiers, opt => opt.MapFrom(src => (!string.IsNullOrEmpty(src.EmployerHpio) ? new List<ModelSchemas.Cda.EntityIdentifier>()
                {
                    new ModelSchemas.Cda.EntityIdentifier
                    {
                        AssigningAuthorityName = "HPI-O",
                        AssigningGeographicArea = "National Identifier",
                        Root = string.Format("1.2.36.1.2001.1003.0.{0}", src.EmployerHpio)
                    }
                }: null)));      
            
            #endregion
        }

        /// <summary>
        /// Resolves the value of IndigenousStatus.
        /// </summary>
        internal class IndigenousStatusResolver : ValueResolver<string, Nehta.VendorLibrary.CDA.Common.Enums.IndigenousStatus>
        {
            #region Methods

            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">IndigenousStatus representing the source.</param>
            /// <returns>IndigenousStatus representing the destination.</returns>
            protected override Nehta.VendorLibrary.CDA.Common.Enums.IndigenousStatus ResolveCore(string source)
            {
                Nehta.VendorLibrary.CDA.Common.Enums.IndigenousStatus result;

                switch (source)
                {
                    case "1":
                        result = Nehta.VendorLibrary.CDA.Common.Enums.IndigenousStatus.AboriginalButNotTorresStraitIslanderOrigin;
                        break;

                    case "2":
                        result = Nehta.VendorLibrary.CDA.Common.Enums.IndigenousStatus.TorresStraitIslanderButNotAboriginalOrigin;
                        break;

                    case "3":
                        result = Nehta.VendorLibrary.CDA.Common.Enums.IndigenousStatus.BothAboriginalAndTorresStraitIslanderOrigin;
                        break;

                    case "4":
                        result = Nehta.VendorLibrary.CDA.Common.Enums.IndigenousStatus.NeitherAboriginalNorTorresStraitIslanderOrigin;
                        break;

                    case "9":
                        result = Nehta.VendorLibrary.CDA.Common.Enums.IndigenousStatus.NotStatedOrInadequatelyDescribed;
                        break;

                    default:
                         result = Nehta.VendorLibrary.CDA.Common.Enums.IndigenousStatus.Undefined;
                         break;
                }

                return result;
            }

            #endregion Methods
        }

        /// <summary>
        /// Resolves the value of Identifiers.
        /// </summary>
        internal class IdentifierResolver : ValueResolver<ModelSchemas.Cda.CdaPathDIDocumentDetails, List<ModelSchemas.Cda.EntityIdentifier>>
        {
            #region Methods

            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">cdaPathDIDocumentDetails representing the source.</param>
            /// <returns>List of Patient Identifiers.</returns>
            protected override List<ModelSchemas.Cda.EntityIdentifier> ResolveCore(ModelSchemas.Cda.CdaPathDIDocumentDetails source)
            {
                List<ModelSchemas.Cda.EntityIdentifier> listIdentifier = new List<ModelSchemas.Cda.EntityIdentifier>()
                {
                    new ModelSchemas.Cda.EntityIdentifier()
                    {
                        AssigningAuthorityName = "IHI",
                        AssigningGeographicArea = "National Identifier",
                        Root = string.Format("1.2.36.1.2001.1003.0.{0}", source.CdaDocumentDetails.PatientMaster.Ihi)
                    },
                    new ModelSchemas.Cda.EntityIdentifier()
                    {
                        AssigningAuthorityName = source.CdaDocumentDetails.Hospital.Name,
                        Root = string.Format("1.2.36.1.2001.1005.49.3.{0}.{1}", source.CdaDocumentDetails.Hospital.HpiO, source.CdaDocumentDetails.Hospital.HospitalId),
                        Extension = source.CdaDocumentDetails.HospitalPatient.Mrn,
                        Code = new ModelSchemas.Cda.CdaCode()
                        {
                            Code = "MR",
                            CodeSystem = "2.16.840.1.113883.12.203",
                            CodeSystemName = "Identifier Type (HL7)"                                    
                        }
                    }
                };

                // Only map if the RequestingHospitalPatientMrn is supplied
                if (!string.IsNullOrEmpty(source.RequestingHospitalPatientMrn) && !string.IsNullOrEmpty(source.OrderDetails.Requester.EmployerName) && !string.IsNullOrEmpty(source.OrderDetails.Requester.EmployerHpio))
                {
                    listIdentifier.Add(new ModelSchemas.Cda.EntityIdentifier()
                    {
                        AssigningAuthorityName = source.OrderDetails.Requester.EmployerName,
                        Root = "1.2.36.1.2001.1005.49.3." + source.OrderDetails.Requester.EmployerHpio,
                        Extension = source.RequestingHospitalPatientMrn,
                        Code = new ModelSchemas.Cda.CdaCode()
                        {
                            Code = "MR",
                            CodeSystem = "2.16.840.1.113883.12.203",
                            CodeSystemName = "Identifier Type (HL7®)"
                        }
                    });
                }
                return listIdentifier;
            }

            #endregion Methods
        }
    }
}
