﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using AutoMapper;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.PcehrDataStore.Schemas;

using ModelSchemas = HIPS.CommonSchemas;
using NehtaSchemas = Nehta.VendorLibrary.CDA;
using Nehta.VendorLibrary.Common;

namespace HIPS.CommonBusinessLogic.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for the Test Results.
    /// </summary>
    internal class ResultProfile : Profile
    {
        /// <summary>
        /// Returns the name of the mapping profile.
        /// </summary>
        public override string ProfileName
        {
            get { return this.GetType().Name; }
        }

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            // Model --> NEHTA

            #region Diagnostic Imaging

            // Diagnostic Imaging - SCSContent - ImagingExaminationResults
            this.CreateMap<ModelSchemas.Cda.ImagingExamResult, NehtaSchemas.SCSModel.IDiagnosticImagingExaminationResult>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreateDiagnosticImagingExaminationResult())
                //.ForMember(dest => dest.CustomNarrativeImagingExaminationResult, opt => opt.MapFrom(src => src.TestResult.NarrativeText)) - Removed but need to decide where this should go (perhap Admin Observation)
                .ForMember(dest => dest.ExaminationResultName, opt => opt.ResolveUsing<ResultNameResolver>().FromMember(src => src.TestResult))
                .ForMember(dest => dest.Modality, opt => opt.ResolveUsing<ImagingModalityResolver>().FromMember(src => src.Modality))
                .ForMember(dest => dest.AnatomicalSite, opt => opt.MapFrom(src => src.AnatomicalSite))
                .ForMember(dest => dest.ExaminationProcedure, opt => opt.MapFrom(src => src.Procedure))
                .ForMember(dest => dest.ExaminationDetails, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.ObservationDateTime, opt => opt.MapFrom(src => src.TestResult.ObservationDateTime))
                //.ForMember(dest => dest.RelatedImage, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.OverallResultStatus, opt => opt.ResolveUsing<ResultStatusResolver>().FromMember(src => src.TestResult));

            // Diagnostic Imaging - SCSContent - ImagingExaminationResults - AnatomicalSite
            this.CreateMap<string, NehtaSchemas.SCSModel.Common.IAnatomicalSiteExtended>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreateAnatomicalSiteExtended())
                .ForMember(dest => dest.Description, opt => opt.MapFrom(src => Nehta.HL7.CDA.NullFlavor.UNK));

            // Diagnostic Imaging - SCSContent - ImagingExaminationResults - ExaminationDetails
            this.CreateMap<ModelSchemas.Cda.ImagingExamResult, NehtaSchemas.SCSModel.Pathology.ExaminationDetails>()
                .ForMember(dest => dest.ImageDateTime, opt => opt.MapFrom(src => src.ImageDateTime));

            // Diagnostic Imaging - SCSContent - ImagingExaminationResults - RelatedImage
            //this.CreateMap<ModelSchemas.Cda.ImagingExamResult, CDA.Generator.Common.SCSModel.DiagnosticImagingReport.Entities.RelatedImage>()
                //.ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreateRelatedImage())
                //.ForMember(dest => dest.ImageUrl, opt => opt.MapFrom(src => src.RelatedImageUrl));

            #endregion

            #region Pathology

            // Pathology - SCSContent - PathologyTestResult
            this.CreateMap<ModelSchemas.Cda.PathologyTestResult, NehtaSchemas.SCSModel.Pathology.PathologyTestResult>()
                //.ForMember(dest => dest.CustomNarrative, opt => opt.MapFrom(src => src.TestResult.NarrativeText)) - Removed but need to decide where this should go (perhap Admin Observation)
                .ForMember(dest => dest.ObservationDateTime, opt => opt.MapFrom(src => src.TestResult.ObservationDateTime))
                .ForMember(dest => dest.TestResultName, opt => opt.ResolveUsing<ResultNameResolver>().FromMember(src => src.TestResult))
                .ForMember(dest => dest.TestSpecimenDetail, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.PathologyDiscipline, opt => opt.ResolveUsing<PathologyDisciplineResolver>().FromMember(src => src.PathologyDiscipline))
                .ForMember(dest => dest.OverallTestResultStatus, opt => opt.ResolveUsing<ResultStatusResolver>().FromMember(src => src.TestResult));

            
            // Pathology - SCSContent - PathologyTestResult - TestSpecimenDetail
            this.CreateMap<ModelSchemas.Cda.PathologyTestResult, NehtaSchemas.SCSModel.Pathology.TestSpecimenDetail>()
                .ForMember(dest => dest.CollectionDateTime, opt => opt.MapFrom(src => src.CollectionDateTime));

            #endregion

            #region Common

            // Diagnostic Imaging - SCSContent - ImagingExaminationResults - ExaminationResultName - Translations
            // Pathology - SCSContent - PathologyTestResult - TestResultName - Translations
            this.CreateMap<ModelSchemas.Cda.TestResultName, NehtaSchemas.SCSModel.ICodableTranslation>()
                .ForMember(dest => dest.Code, opt => opt.MapFrom(src => src.Code))
                .ForMember(dest => dest.CodeSystemCode, opt => opt.MapFrom(src => src.CodeSystem))
                .ForMember(dest => dest.DisplayName, opt => opt.MapFrom(src => src.Name));

            #endregion
        }

       
        /// <summary>
        /// Resolves the value of Result Name. 
        /// </summary>
        /// <summary>
        /// Resolves the value of ResultStatus.
        /// </summary>
        internal class ResultNameResolver : ValueResolver<ModelSchemas.Cda.TestResult, NehtaSchemas.SCSModel.ICodableText>
        {
            #region Methods

            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">TestResult representing the source.</param>
            /// <returns>ICodeableText representing the destination.</returns>
            protected override NehtaSchemas.SCSModel.ICodableText ResolveCore(ModelSchemas.Cda.TestResult source)
            {
                // Is there a matching code system
                CodeSystem localCodeSystem = this.GetCodeSystem(source.PrimaryName.CodeSystem, source.PrimaryName.HL7Field);
                NehtaSchemas.Generator.Enums.CodingSystem? codeSystem = null;
                string displayName = string.Empty;
                // If the local CodeSystem is returned then try and load the matching one from Nehta library
                if (localCodeSystem != null)
                {
                    codeSystem = Enum.GetValues(typeof(NehtaSchemas.Generator.Enums.CodingSystem))
                                         .Cast<NehtaSchemas.Generator.Enums.CodingSystem>()
                                         .Where(c => c.GetAttributeValue<NehtaSchemas.Common.NameAttribute, string>(a => a.Code) == localCodeSystem.Oid)
                                         .FirstOrDefault();
                    // Either pass in the display name if there is a matching ode system, otehrwise original text if no codeSystem is returned
                    if (codeSystem != NehtaSchemas.Generator.Enums.CodingSystem.Undefined)
                        displayName = source.PrimaryName.Name;
                }
                // If no matching code system is found then simply set the original text
                string originalText = (codeSystem == null ? source.PrimaryName.Name : string.Empty);
                
                return NehtaSchemas.Common.BaseCDAModel.CreateResultName(source.PrimaryName.Code, codeSystem, displayName, originalText, GetTransalationsList(source.AlternateNames.ToList()));

            }
            /// <summary>
            /// Gets the CodeSystem.
            /// </summary>
            /// <param name="codeSystem">The Code.</param>
            /// /// <param name="hl7Field">The HL7 Field.</param>
            /// <returns>Code System.</returns>
            private CodeSystem GetCodeSystem(string codeSystem, string hl7Field)
            {
                var lists = ListSingleton.Instance;
                var result = lists.AllCodeSystems.Where(cs => cs.Code == codeSystem && cs.HL7FieldPosition == hl7Field).FirstOrDefault();

                return result;
            }

            /// <summary>
            /// Return a iCodableTranslation list.
            /// </summary>
            /// <param name="source">ResolutionResult representing the source.</param>
            /// <returns>ResolutionResult representing the destination.</returns>
            private List<NehtaSchemas.SCSModel.ICodableTranslation> GetTransalationsList(List<ModelSchemas.Cda.TestResultName> source)
            {
                var translations = new List<NehtaSchemas.SCSModel.ICodableTranslation>();

                foreach (var item in source)
                {
                    CodeSystem localCodeSystem = this.GetCodeSystem(item.CodeSystem, item.HL7Field);
                    NehtaSchemas.Generator.Enums.CodingSystem? codeSystem = null;
                    string displayName = string.Empty;
                    // If the local CodeSystem is returned then try and load the matching one from Nehta library
                    if (localCodeSystem != null)
                    {
                        codeSystem = Enum.GetValues(typeof(NehtaSchemas.Generator.Enums.CodingSystem))
                                             .Cast<NehtaSchemas.Generator.Enums.CodingSystem>()
                                             .Where(c => c.GetAttributeValue<NehtaSchemas.Common.NameAttribute, string>(a => a.Code) == localCodeSystem.Oid)
                                             .FirstOrDefault();
                        // Add the transaltion if we have a valid codesystem
                        if (codeSystem != NehtaSchemas.Generator.Enums.CodingSystem.Undefined)
                        {
                            var translation = NehtaSchemas.Common.BaseCDAModel.CreateCodableTranslation(item.Code, codeSystem, item.Name);
                            translations.Add(translation);
                        }
                    }
                }

                return translations;
            }
            #endregion Methods

        }

        /// <summary>
        /// Resolves the value of OverallTestResultStatus.
        /// </summary>
        /// <summary>
        /// Resolves the value of ResultStatus.
        /// </summary>
        internal class ResultStatusResolver : ValueResolver<ModelSchemas.Cda.TestResult, NehtaSchemas.SCSModel.ICodableText>
        {
            #region Methods

            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">ResolutionResult representing the source.</param>
            /// <returns>ResolutionResult representing the destination.</returns>
            protected override NehtaSchemas.SCSModel.ICodableText ResolveCore(ModelSchemas.Cda.TestResult source)
            {
                switch (source.Status)
                {
                    case "C":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.CorrectionToResults);
                        
                    case "F":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.FinalResults);
                        
                    case "I":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.NoResultsAvailableSpecimenReceivedProcedureIncomplete);
                                           
                    case "O":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.OrderReceived);
                        
                    case "P":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.Preliminary);
                                            
                    case "R":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.ResultsStored);
                                           
                    case "S":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.NoResultsAvailableProcedureScheduledButNotDone);
                                            
                    case "A":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.SomeButNotAllResultsAvailable);
                        
                    case "X":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.NoResultsAvailableOrderCanceled);
                        
                    case "Y":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.NoOrderOnRecordForThisTest);
                        
                    case "Z":
                        return NehtaSchemas.Common.BaseCDAModel.CreateResultStatus(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus.NoRecordOfThisPatient);
                   
                }

                // If we're here then it didn't map correctly so add a Null Flavour for the codeable text
                return NehtaSchemas.Common.BaseCDAModel.CreateCodableText(NehtaSchemas.Common.Enums.NullFlavour.NoInformation);
            }


            #endregion Methods
       
        }

        /// <summary>
        /// Resolves the value of PathologyDiscipline.
        /// </summary>
        internal class PathologyDisciplineResolver : ValueResolver<ModelSchemas.PathologyDiscipline, NehtaSchemas.Common.Enums.DiagnosticService>
        {
            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">ResolutionResult representing the source.</param>
            /// <returns>ResolutionResult representing the destination.</returns>
            protected override NehtaSchemas.Common.Enums.DiagnosticService ResolveCore(ModelSchemas.PathologyDiscipline source)
            {
                var values = Enum.GetValues(typeof(NehtaSchemas.Common.Enums.DiagnosticService))
                                 .Cast<NehtaSchemas.Common.Enums.DiagnosticService>()
                                 .Where(v => v.GetAttributeValue<NehtaSchemas.Common.NameAttribute, string>(a => a.Code) == source.ToString());
                
                if (values == null || values.Count() < 1)
                {
                    return NehtaSchemas.Common.Enums.DiagnosticService.Other;
                }

                return values.SingleOrDefault();
            }
        }

        /// <summary>
        /// Resolves the value of Modality.
        /// </summary>
        internal class ImagingModalityResolver : ValueResolver<string, NehtaSchemas.SCSModel.ICodableText>
        {
            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">ResolutionResult representing the source.</param>
            /// <returns>ResolutionResult representing the destination.</returns>
            protected override NehtaSchemas.SCSModel.ICodableText ResolveCore(string source)
            {
                var values = Enum.GetValues(typeof(NehtaSchemas.Common.Enums.DiagnosticService))
                                 .Cast<NehtaSchemas.Common.Enums.DiagnosticService>()
                                 .Where(v => v.GetAttributeValue<NehtaSchemas.Common.NameAttribute, string>(a => a.Code) == source);

                if (values == null || values.Count() < 1)
                {
                    return NehtaSchemas.Common.BaseCDAModel.CreateCodableText(source);
                }

                return NehtaSchemas.Common.BaseCDAModel.CreateCodableText(source, NehtaSchemas.Generator.Enums.CodingSystem.HL7DiagnosticServiceSectionID, ((NehtaSchemas.Common.Enums.DiagnosticService)values.SingleOrDefault()).GetAttributeValue<NehtaSchemas.Common.NameAttribute, string>(a => a.Name)); 
            }
        }

    }
}
