﻿using System;
using System.Collections.Generic;
using System.Linq;

using AutoMapper;
using DatabaseSchemas = HIPS.PcehrDataStore.Schemas;
using ModelSchemas = HIPS.CommonSchemas;
using NehtaSchemas = Nehta.VendorLibrary.CDA;

namespace HIPS.CommonBusinessLogic.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for the SubjectOfCare (patient) entity in a CDA document.
    /// </summary>
    internal class SubjectOfCareProfile : Profile
    {
        #region Methods

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            // Database --> Model
            //
            // The database schema has given names joined by spaces, so they will be split apart.
            //
            // The database schema has a single Suffix and Title which will be an empty string
            // rather than null if not known. This empty string needs to be replaced by an empty
            // list, and must not be a list with one element containing an empty string!
            //
            // The database schema has NameType which differentiates the registered name (legal)
            // from the current name in PAS (preferred) and previous names. The registered name
            // will be mapped to RegisteredName, all others to OtherName.

            this.CreateMap<DatabaseSchemas.PatientMasterName, ModelSchemas.Cda.IndividualName>()
                .ForMember(dest => dest.FamilyName, opt => opt.MapFrom(src => src.FamilyName))

                .ForMember(
                    dest => dest.GivenNames,
                    opt => opt.MapFrom(src => string.IsNullOrWhiteSpace(src.GivenNames)
                        ? new string[0]
                        : src.GivenNames.Split(' ')))
                .ForMember(
                    dest => dest.Suffixes,
                    opt => opt.MapFrom(src => string.IsNullOrWhiteSpace(src.Suffix)
                        ? new string[0]
                        : new string[] { src.Suffix }))
                .ForMember(
                    dest => dest.Titles,
                    opt => opt.MapFrom(src => string.IsNullOrWhiteSpace(src.Title)
                        ? new string[0]
                        : new string[] { src.Title }))
                .ForMember(dest => dest.Usage, opt => opt.ResolveUsing((DatabaseSchemas.PatientMasterName src) =>
                {
                    if (src.NameTypeId == (int)HIPS.Base.Schemas.Enumerators.NameTypes.Legal)
                    {
                        return ModelSchemas.Cda.IndividualNameUsage.RegisteredName;
                    }
                    return ModelSchemas.Cda.IndividualNameUsage.OtherName;
                }));

            // Model --> NEHTA

            #region Discharge Summary

            // Discharge Summary - SCSContext - SubjectOfCare
            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.IParticipationSubjectOfCare>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateSubjectOfCare())
                .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src));
            
            // Discharge Summary - SCSContext - SubjectOfCare - Participant
            // Mapping for Person: Person Profile (Discharge Summary - SCSContext - SubjectOfCare - Participant - Person)
            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.ISubjectOfCare>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateParticipantForSubjectOfCare())
                .ForMember(dest => dest.Person, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.Addresses, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.Address>() { src.Metadata.PatientAddress }))
                .ForMember(dest => dest.ElectronicCommunicationDetails, opt => opt.MapFrom(src => src.Metadata.PatientContactDetails == null ? null : new List<ModelSchemas.Cda.ElectronicCommunicationDetail>() { src.Metadata.PatientContactDetails }))
                .ForMember(dest => dest.UniqueIdentifier, opt => opt.ResolveUsing((ModelSchemas.Cda.CdaDocumentDetails src) => Guid.NewGuid()));

            #endregion

            #region Diagnostic Imaging

            // Diagnostic Imaging - SCSContext - SubjectOfCare
            this.CreateMap<ModelSchemas.Cda.CdaPathDIDocumentDetails, NehtaSchemas.SCSModel.IParticipationSubjectOfCare>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreateSubjectOfCare())
                .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src));

            // Diagnostic Imaging - SCSContext - SubjectOfCare - Participant
            // Mapping for Person: Person Profile (Diagnostic Imaging - SCSContext - SubjectOfCare - Participant - Person)
            this.CreateMap<ModelSchemas.Cda.CdaPathDIDocumentDetails, NehtaSchemas.SCSModel.ISubjectOfCare>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreateParticipantForSubjectOfCare())
                .ForMember(dest => dest.UniqueIdentifier, opt => opt.MapFrom(src => Guid.NewGuid()))
                .ForMember(dest => dest.Addresses, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.Address>() { src.CdaDocumentDetails.Metadata.PatientAddress }))
                .ForMember(dest => dest.ElectronicCommunicationDetails, opt => opt.MapFrom(src => src.CdaDocumentDetails.Metadata.PatientContactDetails == null ? null : new List<ModelSchemas.Cda.ElectronicCommunicationDetail>() { src.CdaDocumentDetails.Metadata.PatientContactDetails }))
                .ForMember(dest => dest.Person, opt => opt.MapFrom(src => src));
            
            #endregion

            #region Pathology

            // Pathology - SCSContext - SubjectOfCare
            this.CreateMap<ModelSchemas.Cda.CdaPathDIDocumentDetails, NehtaSchemas.SCSModel.IParticipationSubjectOfCare>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.PathologyResultReport.CreateSubjectOfCare())
                .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src));

            // Pathology - SCSContext - SubjectOfCare - Participant
            // Mapping for Person: Person Profile (Pathology - SCSContext - SubjectOfCare - Participant - Person)
            this.CreateMap<ModelSchemas.Cda.CdaPathDIDocumentDetails, NehtaSchemas.SCSModel.ISubjectOfCare>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.PathologyResultReport.CreateParticipantForSubjectOfCare())
                .ForMember(dest => dest.UniqueIdentifier, opt => opt.MapFrom(src => Guid.NewGuid()))
                .ForMember(dest => dest.Addresses, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.Address>() { src.CdaDocumentDetails.Metadata.PatientAddress }))
                .ForMember(dest => dest.ElectronicCommunicationDetails, opt => opt.MapFrom(src => src.CdaDocumentDetails.Metadata.PatientContactDetails == null ? null : new List<ModelSchemas.Cda.ElectronicCommunicationDetail>() { src.CdaDocumentDetails.Metadata.PatientContactDetails }))
                .ForMember(dest => dest.Person, opt => opt.MapFrom(src => src));

            #endregion
        }

        #endregion Methods
    }
}