﻿using System;
using System.Linq;
using System.Net;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel;
using System.ServiceModel.Channels;
using System.Transactions;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.Configuration;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using Nehta.VendorLibrary.PCEHR;
using Nehta.VendorLibrary.PCEHR.RemoveDocument;

namespace HIPS.CommonBusinessLogic.Pcehr
{
    /// <summary>
    /// Handles invoking the PCEHR document removal web service.
    /// </summary>
    public class DocumentRemovalInvoker
    {
        /// <summary>
        /// Invokes the web service to remove a document from the PCEHR system.
        /// This sets the document status to "Deprecated" within the XDS
        /// repository. The document can be set back to the "Approved" status
        /// by uploading a new version. After uploading a new version, all
        /// historical versions are visible once more, including the version
        /// that was "removed".
        /// </summary>
        /// <param name="operation">The queued document removal operation.</param>
        /// <param name="documentId">The ID of the latest version of the document to be removed.</param>
        /// <exception cref="System.InvalidOperationException">When the operation should be retried.</exception>
        /// <returns>Whether the document was removed.</returns>
        public static bool Remove(QueuedRemoveOperation operation, string documentId)
        {
            // Create PCEHR header and web service client object
            CommonPcehrHeader header = Helpers.GetHeader(operation.PatientIdentifier, operation.PatientMaster.Ihi, operation.User, operation.Hospital);
            RemoveDocumentClient removeDocumentClient = CreateRemoveDocumentClient(operation);
            FaultAction action;
            HipsResponse pcehrResponse = new HipsResponse(HipsResponseIndicator.SystemError);
            try
            {
                Guid guid;
                if (operation.DocumentType.RepositoryId == (int)Repository.PCEHR && Guid.TryParseExact(documentId, "D", out guid))
                {
                    documentId = XdsMetadataHelper.UuidToOid(documentId);
                }

                var request = new removeDocument()
                {
                    documentID = documentId,
                    reasonForRemoval = operation.RemovalReason
                };

                // Invoke the service
                var responseStatus = removeDocumentClient.RemoveDocument(header, request);

                // Store the code, description and details from the response.
                pcehrResponse.ResponseCode = responseStatus.responseStatus.code;
                pcehrResponse.ResponseCodeDescription = responseStatus.responseStatus.description;
                pcehrResponse.ResponseCodeDetails = responseStatus.responseStatus.details;

                // Classify the response as Completed, CompletedWithWarnings, TransientFailure or PermanentFailure
                action = FaultHelper.ClassifyMessages(pcehrResponse.ResponseCode, pcehrResponse.ResponseCodeDescription);
            }
            catch (FaultException<Nehta.VendorLibrary.PCEHR.RemoveDocument.StandardErrorType> ex)
            {
                // Store the code and description from the [ATS 5820-2010] SOAP fault.
                pcehrResponse.ResponseCode = ex.Detail.errorCode.ToString();
                pcehrResponse.ResponseCodeDescription = ex.Detail.message;

                // Classify the fault as CompletedWithWarnings, TransientFailure or PermanentFailure.
                action = FaultHelper.ClassifyMessages(pcehrResponse.ResponseCode, pcehrResponse.ResponseCodeDescription);
            }
            catch (Exception ex)
            {
                // Store the exception message, inner exception message and stack trace.
                pcehrResponse.ResponseCode = ex.Message;
                if (ex.InnerException != null)
                {
                    pcehrResponse.ResponseCodeDescription = ex.InnerException.Message;
                }

                // Classify the exception as TransientFailure or PermanentFailure.
                action = FaultHelper.ClassifyMessages(pcehrResponse.ResponseCode, pcehrResponse.ResponseCodeDescription);
            }
            finally
            {
                removeDocumentClient.Close();
            }
            bool wasSuccessful = HandlePcehrResponse(operation, removeDocumentClient.SoapMessages, pcehrResponse, action);
            return wasSuccessful;
        }

        /// <summary>
        /// Creates the web service client object for the document removal operation.
        /// </summary>
        /// <param name="operation">The queued document remove operation.</param>
        /// <returns>The web service client object.</returns>
        private static RemoveDocumentClient CreateRemoveDocumentClient(QueuedRemoveOperation operation)
        {
            // Obtain the certificate for use with TLS and signing
            X509Certificate2 certificate = Helpers.GetConnectionCertificate(operation.Hospital);
            Uri url = Helpers.GetRemoveDocumentUrl(operation.DocumentType);
            RemoveDocumentClient removeDocumentClient = new RemoveDocumentClient(url, certificate, certificate, Settings.Instance.MockPcehrServiceDocumentRemoveWaitSeconds);

            System.Reflection.FieldInfo clientField = removeDocumentClient.GetType().GetField("removeDocumentClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
            RemoveDocumentPortTypeClient rdptc = clientField.GetValue(removeDocumentClient) as RemoveDocumentPortTypeClient;
            CustomBinding binding = rdptc.Endpoint.Binding as CustomBinding;
            HttpsTransportBindingElement https = binding.Elements.First(a => a.GetType() == typeof(HttpsTransportBindingElement)) as HttpsTransportBindingElement;

            //Set the connection timeout for the GetDocument service
            binding.OpenTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentProductionTimeoutSeconds);
            binding.ReceiveTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentProductionTimeoutSeconds);
            binding.SendTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentProductionTimeoutSeconds);

            // Avoid using the default proxy if set
            if (Settings.Instance.AvoidProxy)
            {
                https.UseDefaultWebProxy = false;
            }

            // Add server certificate validation callback
            ServicePointManager.ServerCertificateValidationCallback += DocumentHelper.ValidateServiceCertificate;

            return removeDocumentClient;
        }

        /// <summary>
        /// Performs the actions required after invoking the web service to
        /// remove a document from the PCEHR system. This includes writing an
        /// audit record to the PcehrAudit table and populating the operation
        /// status information in the PcehrMessageQueue item.
        ///
        /// In the case where HIPS was unable to connect to the PCEHR system,
        /// or the PCEHR system returned a message that the service was
        /// temporarily unavailable, this method throws the exception
        /// InvalidOperationException that causes the queue transaction to
        /// be rolled back and therefore the removal will be tried again
        /// according to the MSMQ configuration.
        ///
        /// Otherwise returns whether the clinical document was removed.
        /// </summary>
        /// <param name="operation">The queued remove document operation.</param>
        /// <param name="uploadDocumentClient">The web service client.</param>
        /// <param name="pcehrResponse">Details of the responses received.</param>
        /// <param name="action">The classification of the responses received.</param>
        /// <exception cref="System.InvalidOperationException">When the queued operation should be tried again.</exception>
        /// <returns>Whether the clinical document was removed.</returns>
        private static bool HandlePcehrResponse(QueuedRemoveOperation operation,
            SoapMessages soapMessages, HipsResponse pcehrResponse,
            FaultAction action)
        {
            bool wasRemoved;
            pcehrResponse.Status = FaultHelper.ConvertToHipsResponseIndicator(action);
            string codeAndDetails = string.Format(ResponseStrings.ErrorCodeAndDetailsFormat,
                pcehrResponse.ResponseCode, pcehrResponse.ResponseCodeDescription);
            operation.PendingItem.Request = soapMessages.SoapRequest;
            operation.PendingItem.Response = soapMessages.SoapResponse;
            operation.PendingItem.Details = pcehrResponse.ToString();

            Helpers.InsertAudit(operation.PatientMaster,
                operation.User,
                operation.Hospital,
                AuditOperationNames.RemoveDocument,
                pcehrResponse,
                soapMessages);
            switch (action)
            {
                case FaultAction.Completed:
                case FaultAction.CompletedWithWarnings:
                    wasRemoved = true;
                    operation.PendingItem.QueueStatusId = (int)QueueStatus.Success;
                    break;

                case FaultAction.TransientFailure:

                    // HIPS was unable to connect to the PCEHR system or
                    // the PCEHR system returned a message indicating that
                    // the service was temporarily unavailable. In this case
                    // the pending operation will not be updated. HIPS will
                    // write to the event log and fail the queued operation,
                    // so that MSMQ will retry the removal.
                    using (new TransactionScope(TransactionScopeOption.Suppress))
                    {
                        EventLogger.WriteLog(ResponseStrings.PcehrSystemTemporarilyUnavailable,
                            new Exception(codeAndDetails), operation.User, LogMessage.HIPS_MESSAGE_105);
                    }
                    throw new InvalidOperationException();

                default:
                    operation.PendingItem.QueueStatusId = (int)QueueStatus.Failure;
                    wasRemoved = false;
                    break;
            }
            return wasRemoved;
        }
    }
}