﻿using System;
using System.Transactions;
using System.Linq;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic.Ihi;
using HIPS.CommonSchemas;
using HIPS.IhiSchemas.Schemas;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrQueueLogic;
using HIPS.PcehrSchemas;
using HIPS.HL7.Common;
using HIPS.HL7.Common.Message;
using HIPS.HL7.Common.Enumerators;
using HIPS.HL7.Common.DataStructure;
using HIPS.CommonBusinessLogic.HL7;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.CommonBusinessLogic.Cda;
using System.IO;
using System.Runtime.Serialization;
using System.Text;

namespace HIPS.CommonBusinessLogic.Pcehr
{
    public class DocumentUpload
    {
        #region private members

        /// <summary>
        /// The document set for the instance we are uploading in this operation.
        /// </summary>
        private ClinicalDocument clinicalDocument;

        /// <summary>
        /// A previously-uploaded document instance with the same source system document ID as the one that we are being asked to upload.
        /// </summary>
        private ClinicalDocumentVersion duplicateVersion;

        /// <summary>
        /// The document instance that are uploading in this operation.
        /// </summary>
        private ClinicalDocumentVersion newCurrentVersion;

        /// <summary>
        /// The most recently uploaded document instance in the document set. This will be the parent document for the current upload.
        /// </summary>
        private ClinicalDocumentVersion oldCurrentVersion;

        /// <summary>
        /// The queued upload operation object that was stored in the MSMQ.
        /// </summary>
        private QueuedUploadOperation operation;

        /// <summary>
        /// Data access helper.
        /// </summary>
        private PatientAccess patientAccess;

        /// <summary>
        /// Data access for PCEHR message queue pending item.
        /// </summary>
        private PcehrMessageQueueDl pendingItemDataAccess;

        #endregion private members

        #region constructor

        /// <summary>
        /// Initializes a new instance of the <see cref="DocumentUpload" /> class.
        /// </summary>
        /// <param name="queuedUploadOperation">The queued upload operation.</param>
        public DocumentUpload(QueuedUploadOperation queuedUploadOperation)
        {
            this.operation = queuedUploadOperation;
            this.patientAccess = new PatientAccess(queuedUploadOperation.User);
            this.pendingItemDataAccess = new PcehrMessageQueueDl(operation.User);
        }

        #endregion constructor

        #region Methods

        /// <summary>
        /// Performs the upload or supersede operation as specified in the queued upload operation.
        /// </summary>
        /// <exception cref="System.InvalidOperationException">When the MSMQ operation should be retried</exception>
        public void PerformUpload()
        {
            DocumentQueueTransactionHandler handler = new DocumentQueueTransactionHandler(operation.User);

            if (!handler.RestorePackageInQueuedUploadOperation(operation))
            {
                return;
            }
            if (operation.PendingItem.QueueStatusId != (int)QueueStatus.Pending)
            {
                string message = string.Format(ResponseStrings.OperationAbortedDetail,
                    operation.PendingItem.PcehrMessageQueueId,
                    operation.PendingItem.QueueOperationName,
                    operation.SourceSystemDocumentId,
                    operation.PatientIdentifier,
                    operation.Hospital.Description);
                EventLogger.WriteLog(ResponseStrings.InfoQueuedOperationAborted, new Exception(message), operation.User, LogMessage.HIPS_MESSAGE_086);
                if (operation.HL7MessageLogId.HasValue)
                {
                    if (string.IsNullOrEmpty(operation.PendingItem.Details))
                    {
                        operation.PendingItem.Details = message;
                    }
                    handler.PlaceAcknowledgementOnQueue(operation.PendingItem, operation.HL7MessageLogId.Value);
                }
                return;
            }

            //ALL Documents in the same Document Set must be performed in order. This is determined from the SourceSystemSetId Id in the
            //PcehrMessageQueue table. If there are any PENDING messages that are from the same DocumentSet that are earlier than this
            //message then this message must log that there are earlier messages still to be performed and this must wait.
            PcehrMessageQueueDl dataAcces = new PcehrMessageQueueDl(operation.User);
            using (new TransactionScope(TransactionScopeOption.Suppress))
            {
                if (dataAcces.CountEarlierPendingMessages(operation.SourceSystemSetId, operation.PendingItem.DateCreated, operation.PendingItem.Id) > 0)
                {
                    string message = string.Format(ResponseStrings.OperationOutOfOrderUpload,
                        operation.PendingItem.PcehrMessageQueueId,
                        operation.PendingItem.QueueOperationName,
                        operation.SourceSystemSetId,
                        operation.SourceSystemDocumentId,
                        operation.PatientIdentifier,
                        operation.Hospital.Description);
                    EventLogger.WriteLog(ResponseStrings.InfoQueuedOperationRetriedDueToMessagesOutofOrder, new Exception(message), operation.User, LogMessage.HIPS_MESSAGE_136);
                    throw new InvalidOperationException();
                }
            }

            // Ensure the IHI has been validated within the configured period.
            // This is always done before the upload operation is placed on the
            // queue, however if the HI Service was unavailable at the time then
            // it must be done now. If the HI Service is still unavailable now
            // then the queue transaction will be rolled back and the upload
            // will stay on the queue, being retried, until the HI service is
            // once more available.
            PatientIhiValidation patientIhiValidation = new PatientIhiValidation();
            IhiSearchResponse ihiResponse;

            using (new TransactionScope(TransactionScopeOption.Suppress))
            {
                ihiResponse = patientIhiValidation.GetValidatedIhi(operation.PatientIdentifier, operation.Hospital, operation.User, operation.PatientMaster);
            }
            if (ihiResponse.HipsResponse.Status == HipsResponseIndicator.HiServiceError)
            {
                // If the IHI cannot be validated because the HI service is unavailable, HIPS will
                // roll back the queue transaction, and the operation will be tried again.
                using (new TransactionScope(TransactionScopeOption.Suppress))
                {
                    EventLogger.WriteLog(ResponseStrings.InfoUploadWithStaleIhiBeingRetried, new Exception(ihiResponse.HipsResponse.HipsErrorMessage), operation.User, LogMessage.HIPS_MESSAGE_087);
                }
                throw new InvalidOperationException(ResponseStrings.InfoUploadWithStaleIhiBeingRetried);
            }
            if (ihiResponse.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                operation.PendingItem.QueueStatusId = (int)QueueStatus.Failure;
                operation.PendingItem.Details = ihiResponse.HipsResponse.HipsErrorMessage;
                handler.UpdateOperationForFailure(operation.PendingItem);
                return;
            }
            this.patientAccess.ValidateLocalIhiInformation(operation.PatientMaster, ihiResponse.HipsResponse);
            if (ihiResponse.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                operation.PendingItem.QueueStatusId = (int)QueueStatus.Failure;
                operation.PendingItem.Details = ihiResponse.HipsResponse.HipsErrorMessage;
                handler.UpdateOperationForFailure(operation.PendingItem);
                return;
            }

            LoadClinicalDocumentAndVersions();

            if (this.duplicateVersion != null)
            {
                operation.PendingItem.QueueStatusId = (int)QueueStatus.Failure;
                operation.PendingItem.Details = string.Format(ResponseStrings.DuplicateSourceSystemDocumentId, operation.SourceSystemDocumentId);
                handler.UpdateOperationForFailure(operation.PendingItem);
                return;
            }

            if (UploadIsSuccessOrDuplicate())
            {
                if (oldCurrentVersion != null)
                {
                    oldCurrentVersion.SupersededDate = DateTime.Now;
                }
                this.newCurrentVersion = new ClinicalDocumentVersion();
                this.newCurrentVersion.Package = this.operation.Package;
                this.newCurrentVersion.UploadedDate = DateTime.Now;
                this.newCurrentVersion.SourceSystemDocumentId = this.operation.SourceSystemDocumentId;
                handler.SaveAfterSuccessfulUpload(clinicalDocument, oldCurrentVersion, newCurrentVersion, operation.PendingItem);
            }
            else
            {
                handler.UpdateOperationForFailure(operation.PendingItem);
            }

            if (operation.HL7MessageLogId.HasValue)
            {
                handler.PlaceAcknowledgementOnQueue(operation.PendingItem, operation.HL7MessageLogId.Value);
            }
        }

        /// <summary>
        /// Loads or creates the clinical document set, and attempts to load two versions.
        /// One based on source system document ID that we are uploading, and one which is
        /// the latest version in the set.
        /// </summary>
        private void LoadClinicalDocumentAndVersions()
        {
            // Get the version for duplicate checking (which could be attached to any document set, on any patient!)
            using (new TransactionScope(TransactionScopeOption.Suppress))
            {
                patientAccess.VersionDataAccess.GetVersion(this.operation.SourceSystemDocumentId, out this.duplicateVersion);
            }
            if (this.duplicateVersion != null && !this.duplicateVersion.ClinicalDocumentVersionId.HasValue)
            {
                this.duplicateVersion = null;
            }

            // Load or create the document set.
            using (new TransactionScope(TransactionScopeOption.Suppress))
            {
                this.clinicalDocument = this.patientAccess.DocumentDataAccess.Get(this.operation.Episode.EpisodeId.Value, this.operation.SourceSystemSetId);
            }
            this.clinicalDocument.EpisodeId = operation.Episode.EpisodeId.Value;
            this.clinicalDocument.SourceSystemSetId = operation.SourceSystemSetId;
            this.clinicalDocument.DocumentTypeId = operation.DocumentType.DocumentTypeId.Value;
            this.clinicalDocument.ClinicalDocumentStatusId = (int)HIPS.PcehrDataStore.Schemas.Enumerators.ClinicalDocumentStatus.Active;
            this.clinicalDocument.RemovalReasonId = -1;
            this.clinicalDocument.RemovedDate = null;

            // If the document set already existed, get the most recently uploaded version in the document set.
            if (this.clinicalDocument.ClinicalDocumentId.HasValue)
            {
                using (new TransactionScope(TransactionScopeOption.Suppress))
                {
                    patientAccess.VersionDataAccess.GetLatestVersion(this.clinicalDocument.ClinicalDocumentId.Value, out this.oldCurrentVersion);
                }
                if (this.oldCurrentVersion != null && !this.oldCurrentVersion.ClinicalDocumentVersionId.HasValue)
                {
                    this.oldCurrentVersion = null;
                }
            }
        }

        /// <summary>
        /// Uploads the document instance to the PCEHR.
        /// </summary>
        /// <returns>True if there was a duplicate ID error, or the upload was successful. False if another failure.</returns>
        private bool UploadIsSuccessOrDuplicate()
        {
            string parentDocumentId = this.oldCurrentVersion == null ? null : this.oldCurrentVersion.SourceSystemDocumentId;
            return DocumentUploadInvoker.Upload(this.operation, parentDocumentId);
        }

        #endregion Methods
    }
}