﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Security.Cryptography;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel;
using System.Text;
using System.Transactions;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.Configuration;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using Nehta.VendorLibrary.PCEHR;
using Nehta.VendorLibrary.PCEHR.DocumentRepository;

namespace HIPS.CommonBusinessLogic.Pcehr
{
    /// <summary>
    /// Handles invoking the PCEHR document upload web service.
    /// </summary>
    public class DocumentUploadInvoker
    {
        private const string PCEHR_CONFIDENTIALITY_CODE = "N/A";
        private const string XDS_AUTHOR_PERSON_SLOT_NAME = "authorPerson";
        private const string XDS_DOCUMENT_ENTRY_AUTHOR = "urn:uuid:93606bcf-9494-43ec-9b4e-a7748d1a838d";
        private const string XDS_DOCUMENT_ENTRY_CLASS_CODE = "urn:uuid:41a5887f-8865-4c09-adf7-e362475b143a";
        private const string XDS_DOCUMENT_ENTRY_CONFIDENTIALITY_CODE = "urn:uuid:f4f85eac-e6cb-4883-b524-f2705394840f";
        private const string XDS_DOCUMENT_ENTRY_TYPE_CODE = "urn:uuid:f0306f51-975f-434e-a61c-c59651d33983";
        private const string XDS_SUBMISSION_SET_AUTHOR = "urn:uuid:a7058bb9-b4e4-4307-ba5b-e3f0ab85e12d";
        private const string XDS_SUBMISSION_SET_CONTENT_TYPE_CODE = "urn:uuid:aa543740-bdda-424e-8c96-df4873be8500";

        /// <summary>
        /// <para>Builds the author string from CDA details using the hospital's HPI-O and the Approver.</para>
        ///
        /// <para>Previously, this also overrode the sender ID in the PCEHR header with the ID of the author, which should
        /// not be necessary, but was necessary due to the PCEHR System builder's misinterpretation of the
        /// requirements. The header no longer needs to be overridden because the fix to allow the uploader to
        /// be different from the author of the documents was deployed to SVT and Prod by 21/12/2012.</para>
        ///
        /// <para>The formatting of the XDS metadata authorPerson value field is described below:</para>
        /// <code>
        /// Field  Field Name           Content
        /// -----  ----------           -------
        /// 2      Family Name          Author Family name
        /// 3      Given Name           Author Given name
        /// 5      Suffix               Author suffix
        /// 6      Prefix               Author prefix
        /// 9      Assigning Authority  OID form of the HPI-O (or PAI-O) of Author (* field not previously used)
        /// 10     Name Type Code       [IS data type]
        /// 10.2   Field Value          The local ID of the user within the organisation
        /// </code>
        /// <para>Source: "FAQ Implementation clarification for relaxation of HPII in DS rev001", NEHTA, 5/12/2012.</para>
        /// </summary>
        /// <param name="hospital">The hospital.</param>
        /// <param name="author">The approver.</param>
        /// <returns>The XDS.b metadata authorPerson string</returns>
        public static string BuildAuthorString(Hospital hospital, HIPS.PcehrSchemas.Participant author)
        {
            List<String> emptyList = new List<string>();
            string familyName = author.FamilyName;
            string firstName = String.Empty;
            string middleNames = String.Empty;
            string nameSuffixes = string.Join(" ", author.Suffixes ?? emptyList);
            string nameTitles = string.Join(" ", author.Titles ?? emptyList);
            string idRoot = author.Identifiers[0].Root;
            string idExtension = author.Identifiers[0].Extension;
            if (author.GivenNames.Count > 0)
            {
                firstName = author.GivenNames[0];
            }
            if (author.GivenNames.Count > 1)
            {
                middleNames = string.Join(" ", author.GivenNames.GetRange(1, author.GivenNames.Count - 1));
            }

            if (idExtension == null)
            {
                // HPII identifier:
                //
                // Originally based on FAQ_Implementation_clarification_for_relaxation_of_HPII_in_DS_rev001.pdf
                // This had 9 carets before the first ampersand.
                // Example: ^Button^Henry^^Jr^Dr^^^^&1.2.36.1.2001.1003.0.8003619900015717&ISO
                //
                // Was changed in NEHTA_1483_2013_PCEHROverviewandGuides_ImplementationGuide_v1.3.pdf
                // Now has only 8 carets before the first ampersand.
                // Example: ^Button^Henry^^^^^^&1.2.36.1.2001.1003.0.8003618334357646&ISO
                const string format = "^{0}^{1}^{2}^{3}^{4}^^^&{5}&ISO";
                return string.Format(format, familyName, firstName, middleNames, nameSuffixes, nameTitles, idRoot);
            }
            else
            {
                // Local System Identifier
                // Example: ^Button^Henry^^Jr^Dr^^^1.2.36.1.2001.1005.41.8003621566684455^&buttonh001&ISO
                const string format = "^{0}^{1}^{2}^{3}^{4}^^^{5}^&{6}&ISO";
                return string.Format(format, familyName, firstName, middleNames, nameSuffixes, nameTitles, idRoot, idExtension);
            }
        }

        /// <summary>
        /// Uploads the specified package to the PCEHR.
        /// Populates the QueueStatusId, Request, Response and Details in the PendingItem in the operation.
        /// </summary>
        /// <param name="operation">The queued operation, containing the package, patient, user, approver and pending item.</param>
        /// <param name="parentDocumentId">The parent document id.</param>
        /// <returns>Whether the clinical document should be saved. True on success or failure due to document instance already uploaded. False when PCEHR System rejects the document.</returns>
        /// <exception cref="System.InvalidOperationException">When the MSMQ operation should be retried</exception>
        public static bool Upload(QueuedUploadOperation operation, string parentDocumentId)
        {
            CommonPcehrHeader header = Helpers.GetHeader(operation.PatientIdentifier, operation.PatientMaster.Ihi, operation.User, operation.Hospital);
            UploadDocumentClient uploadDocumentClient = CreateUploadDocumentClient(operation);

            // This HIPS Response is never returned to a calling system because
            // the upload only occurs when processing a queued operation. It
            // exists here to store the response codes and details for the
            // benefit of the HandlePcehrResponse and InsertAudit helper methods.
            HipsResponse pcehrResponse = new HipsResponse(HipsResponseIndicator.OK);
            FaultAction action;
            try
            {
                // Create the upload request. This can throw exceptions if the CDA document is invalid.
                ProvideAndRegisterDocumentSetRequestType request = CreateUploadRequest(operation, parentDocumentId, uploadDocumentClient);

                // Invoke the service
                RegistryResponseType registryResponse = uploadDocumentClient.UploadDocument(header, request);
                StringBuilder serviceMessage = new StringBuilder();

                // The upload operation can return no errors or one or more registry errors
                // whose errorCode is either XDSRepositoryError or XDSRegistryMetadataError.
                // Since we only store one response code / description / details, this code
                // joins them together separated by commas if there is more than one error.
                RegistryError[] errors = new RegistryError[0];
                if (registryResponse.RegistryErrorList != null)
                {
                    errors = registryResponse.RegistryErrorList.RegistryError;
                }
                string[] allErrorCodes = (from error in errors select error.errorCode).Distinct().ToArray();
                string[] allCodeContexts = (from error in errors select error.codeContext).ToArray();
                pcehrResponse.ResponseCode = string.Join(", ", allErrorCodes);
                pcehrResponse.ResponseCodeDescription = string.Join(", ", allCodeContexts);

                // Because HIPS checks if it has uploaded a document before
                // attempting to upload it again, in the case where the PCEHR
                // system gives the error that the document was already
                // uploaded, this must either be:
                // 1. A retry due to an error accessing the database to store
                //    the results after the upload completed successfully, or
                // 2. A document that was uploaded to PCEHR outside of this
                //    instance of HIPS.
                // This will be classified as CompletedWithWarnings and therefore
                // HIPS will attempt to store the document in the database.

                action = FaultHelper.Classify(registryResponse.RegistryErrorList);
            }
            catch (FaultException<Nehta.VendorLibrary.PCEHR.GetTemplate.StandardErrorType> ex)
            {
                // The NEHTA library declares the StandardErrorType in most of
                // the services but not for the UploadDocument, this is because
                // ITI-41 Provide & Register Document Set – b services typically
                // do not return [ATS 5820-2010] SOAP faults.
                //
                // Just in case we will attempt to catch them using the definition
                // in the GetTemplate package and handle appropriately.
                pcehrResponse.ResponseCode = ex.Detail.errorCode.ToString();
                pcehrResponse.ResponseCodeDescription = ex.Detail.message;
                action = FaultHelper.ClassifyMessages(pcehrResponse.ResponseCode, pcehrResponse.ResponseCodeDescription);
            }
            catch (ArgumentException ex)
            {
                // The upload request could not be created because the CDA document was invalid.
                action = FaultAction.PermanentFailure;
                pcehrResponse.HipsErrorMessage = ex.Message;
                if (ex.InnerException != null)
                {
                    pcehrResponse.ResponseCodeDescription = ex.InnerException.Message;
                }
            }
            catch (Exception ex)
            {
                // Some other kind of exception, such as a failure to connect to PCEHR.
                action = FaultAction.TransientFailure;
                pcehrResponse.HipsErrorMessage = ex.Message;
                if (ex.InnerException != null)
                {
                    pcehrResponse.ResponseCodeDescription = ex.InnerException.Message;
                }
                pcehrResponse.ResponseCodeDetails = ex.StackTrace;
            }
            finally
            {
                uploadDocumentClient.Close();
            }
            bool shouldSaveClinicalDocument = HandlePcehrResponse(operation,
                uploadDocumentClient.SoapMessages,
                pcehrResponse,
                action);
            return shouldSaveClinicalDocument;
        }

        /// <summary>
        /// Computes the SHA1 hash of the binary object and formats it as uppercase hexadecimal digits.
        /// </summary>
        /// <param name="binary">The object to compute the hash of.</param>
        /// <returns>The SHA1 has of the input binary object.</returns>
        private static string ComputeHexadecimalSHA1Hash(byte[] binary)
        {
            using (SHA1Managed sha1 = new SHA1Managed())
            {
                return BitConverter.ToString(sha1.ComputeHash(binary)).Replace("-", "");
            }
        }

        /// <summary>
        /// Creates a web service client object for the upload document operation.
        /// </summary>
        /// <param name="operation">The queued upload operation.</param>
        /// <returns>The web service client object.</returns>
        private static UploadDocumentClient CreateUploadDocumentClient(QueuedUploadOperation operation)
        {
            Uri url = Helpers.GetUploadDocumentUrl(operation.DocumentType);
            X509Certificate2 certificate = Helpers.GetConnectionCertificate(operation.Hospital);
            UploadDocumentClient uploadDocumentClient = new UploadDocumentClient(url, certificate, certificate, Settings.Instance.MockPcehrServiceDocumentUploadWaitSeconds);

            System.Reflection.FieldInfo clientField = uploadDocumentClient.GetType().GetField("client", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
            object drc = clientField.GetValue(uploadDocumentClient);
            System.Reflection.FieldInfo repositoryClientField = drc.GetType().GetField("repositoryClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
            DocumentRepository_PortTypeClient ptc = repositoryClientField.GetValue(drc) as DocumentRepository_PortTypeClient;
            WSHttpBinding wsbinding = ptc.Endpoint.Binding as WSHttpBinding;

            //Set the connection timeout for the GetDocument service
            wsbinding.OpenTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentProductionTimeoutSeconds);
            wsbinding.ReceiveTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentProductionTimeoutSeconds);
            wsbinding.SendTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentProductionTimeoutSeconds);

            // Avoid using the default proxy if set
            if (Settings.Instance.AvoidProxy)
            {
                wsbinding.UseDefaultWebProxy = false;
            }

            // Add server certificate validation callback
            ServicePointManager.ServerCertificateValidationCallback += DocumentHelper.ValidateServiceCertificate;
            return uploadDocumentClient;
        }

        /// <summary>
        /// Creates an upload request from the package, with the specified
        /// document format (or the default specified in the application
        /// configuration).
        /// </summary>
        /// <param name="operation">The queued upload operation.</param>
        /// <param name="parentDocumentId">The document ID to replace, or null if a new document</param>
        /// <param name="uploadDocumentClient">The web service client object.</param>
        /// <returns>The upload request.</returns>
        private static ProvideAndRegisterDocumentSetRequestType CreateUploadRequest(QueuedUploadOperation operation, string parentDocumentId, UploadDocumentClient uploadDocumentClient)
        {
            // Create a request from the package. Here we specify the document format code and
            // description.
            ProvideAndRegisterDocumentSetRequestType request;
            Guid guid;
            if (Guid.TryParseExact(parentDocumentId, "D", out guid))
            {
                parentDocumentId = XdsMetadataHelper.UuidToOid(parentDocumentId);
            }
            if (string.IsNullOrEmpty(parentDocumentId))
            {
                request = uploadDocumentClient.CreateRequestForNewDocument(
                operation.Package,
                operation.DocumentFormat.Code,
                operation.DocumentFormat.Description,
                Nehta.VendorLibrary.PCEHR.HealthcareFacilityTypeCodes.Hospitals,
                Nehta.VendorLibrary.PCEHR.PracticeSettingTypes.GeneralHospital
               );
            }
            else
            {
                request = uploadDocumentClient.CreateRequestForReplacement(
                operation.Package,
                operation.DocumentFormat.Code,
                operation.DocumentFormat.Description,
                Nehta.VendorLibrary.PCEHR.HealthcareFacilityTypeCodes.Hospitals,
                Nehta.VendorLibrary.PCEHR.PracticeSettingTypes.GeneralHospital,
                parentDocumentId
               );
            }

            // Override the author strings with the compliant version.
            string s = BuildAuthorString(operation.Hospital, operation.Author);
            ClassificationType xdsDocumentEntryAuthor = request.SubmitObjectsRequest.RegistryObjectList.ExtrinsicObject[0].Classification.First(a => a.classificationScheme == XDS_DOCUMENT_ENTRY_AUTHOR);
            SlotType1 xdsDocumentEntryAuthorPerson = xdsDocumentEntryAuthor.Slot.First(a => a.name == XDS_AUTHOR_PERSON_SLOT_NAME);
            xdsDocumentEntryAuthorPerson.ValueList.Value[0] = s;
            ClassificationType xdsSubmissionSetAuthor = request.SubmitObjectsRequest.RegistryObjectList.RegistryPackage[0].Classification.First(a => a.classificationScheme == XDS_SUBMISSION_SET_AUTHOR);
            SlotType1 xdsSubmissionSetAuthorPerson = xdsSubmissionSetAuthor.Slot.First(a => a.name == XDS_AUTHOR_PERSON_SLOT_NAME);
            xdsSubmissionSetAuthorPerson.ValueList.Value[0] = s;

            // Regarding PCEHR NOC TEST 35
            // N states in error:
            //  "The XDSDocumentEntry.typeCodeDisplayName is the same value as
            //   the XDSDocumentEntry.classCodeDisplayName field"
            //
            // NEHTA have published a clarification regarding test 'N'. Please see
            // https://vendors.nehta.gov.au/public/fileServer.cfm?activityContentId=272
            // N should state:
            //  "The XDSDocumentEntry.typeCodeDisplayName is set to the appropriate typeCodeDisplayName
            //   and the XDSDocumentEntry.classCodeDisplayName is set to the appropriate classCodeDisplayName"
            //
            // Accordingly these overrides to the class code name (e.g. from
            // "Discharge Summarization Note" to "Discharge Summary") are not
            // necessary.
            //
            // Note from Accenture:
            // As per HL7 terminologies it is supposed to be "Discharge Summarization Note" with a z.
            // However, if you use the vendor library and that is generating the incorrect spelling we’ll consider it a pass.
            //
            //ClassificationType classCode = request.SubmitObjectsRequest.RegistryObjectList.ExtrinsicObject[0].Classification.First(a => a.classificationScheme == XDS_DOCUMENT_ENTRY_CLASS_CODE);
            //classCode.Name.LocalizedString[0].value = operation.DocumentType.Description;
            //ClassificationType typeCode = request.SubmitObjectsRequest.RegistryObjectList.ExtrinsicObject[0].Classification.First(a => a.classificationScheme == XDS_DOCUMENT_ENTRY_TYPE_CODE);
            //typeCode.Name.LocalizedString[0].value = operation.DocumentType.Description;
            //ClassificationType contentTypeCode = request.SubmitObjectsRequest.RegistryObjectList.RegistryPackage[0].Classification.First(a => a.classificationScheme == XDS_SUBMISSION_SET_CONTENT_TYPE_CODE);
            //contentTypeCode.Name.LocalizedString[0].value = operation.DocumentType.Description;

            // Change the displayName for XDSDocumentEntry.confidentialityCode from 'NA' to 'N/A' in line with TSS specification.
            ClassificationType confidentialityCode = request.SubmitObjectsRequest.RegistryObjectList.ExtrinsicObject[0].Classification.First(a => a.classificationScheme == XDS_DOCUMENT_ENTRY_CONFIDENTIALITY_CODE);
            confidentialityCode.Name.LocalizedString[0].value = PCEHR_CONFIDENTIALITY_CODE;

            if (operation.DocumentType.RepositoryId == (int)Repository.NPDR)
            {
                List<SlotType1> slots = request.SubmitObjectsRequest.RegistryObjectList.ExtrinsicObject[0].Slot.ToList();
                slots.Add(new SlotType1 { name = "hash", ValueList = new ValueListType { Value = new string[] { ComputeHexadecimalSHA1Hash(operation.Package) } } });
                slots.Add(new SlotType1 { name = "size", ValueList = new ValueListType { Value = new string[] { operation.Package.Length.ToString() } } });
                slots.Add(new SlotType1 { name = "repositoryUniqueId", ValueList = new ValueListType { Value = new string[] { operation.DocumentType.RepositoryUniqueId } } });
                request.SubmitObjectsRequest.RegistryObjectList.ExtrinsicObject[0].Slot = slots.ToArray();
            }

            return request;
        }

        /// <summary>
        /// Performs the actions required after invoking the web service to
        /// upload a document to the PCEHR system. This includes writing an
        /// audit record to the PcehrAudit table and populating the operation
        /// status information in the PcehrMessageQueue item.
        ///
        /// In the case where HIPS was unable to connect to the PCEHR system,
        /// or the PCEHR system returned a message that the service was
        /// temporarily unavailable, this method throws the exception
        /// InvalidOperationException that causes the queue transaction to
        /// be rolled back and therefore the upload will be tried again
        /// according to the MSMQ configuration.
        ///
        /// Otherwise returns whether the clinical document should be saved.
        /// </summary>
        /// <param name="operation">The queued upload document operation.</param>
        /// <param name="uploadDocumentClient">The web service client.</param>
        /// <param name="pcehrResponse">Details of the responses received.</param>
        /// <param name="action">The classification of the responses received.</param>
        /// <exception cref="System.InvalidOperationException">When the queued operation should be tried again.</exception>
        /// <returns>Whether the clinical document should be saved.</returns>
        private static bool HandlePcehrResponse(QueuedUploadOperation operation,
            SoapMessages soapMessages, HipsResponse pcehrResponse,
            FaultAction action)
        {
            bool shouldSaveClinicalDocument;
            pcehrResponse.Status = FaultHelper.ConvertToHipsResponseIndicator(action);
            string codeAndDetails = string.Format(ResponseStrings.ErrorCodeAndDetailsFormat,
                pcehrResponse.ResponseCode, pcehrResponse.ResponseCodeDescription);
            operation.PendingItem.Request = soapMessages.SoapRequest;
            operation.PendingItem.Response = soapMessages.SoapResponse;
            operation.PendingItem.Details = pcehrResponse.ToString();

            Helpers.InsertAudit(operation.PatientMaster,
                operation.User,
                operation.Hospital,
                AuditOperationNames.UploadDocument,
                pcehrResponse,
                soapMessages);
            switch (action)
            {
                case FaultAction.Completed:
                case FaultAction.CompletedWithWarnings:
                    shouldSaveClinicalDocument = true;
                    operation.PendingItem.QueueStatusId = (int)QueueStatus.Success;
                    break;

                case FaultAction.TransientFailure:

                    // HIPS was unable to connect to the PCEHR system or
                    // the PCEHR system returned a message indicating that
                    // the service was temporarily unavailable. In this case
                    // the pending operation will not be updated. HIPS will
                    // write to the event log and fail the queued operation,
                    // so that MSMQ will retry the upload.
                    using (new TransactionScope(TransactionScopeOption.Suppress))
                    {
                        EventLogger.WriteLog(ResponseStrings.PcehrSystemTemporarilyUnavailable,
                            new Exception(codeAndDetails), operation.User, LogMessage.HIPS_MESSAGE_106);
                    }
                    throw new InvalidOperationException();

                default:
                    operation.PendingItem.QueueStatusId = (int)QueueStatus.Failure;
                    shouldSaveClinicalDocument = false;
                    break;
            }
            return shouldSaveClinicalDocument;
        }
    }
}