﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Net.Security;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel;
using HIPS.CommonBusinessLogic.Ihi;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Configuration;
using HIPS.IhiSchemas.Schemas;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrSchemas;
using Nehta.VendorLibrary.CDAPackage;
using Nehta.VendorLibrary.Common;
using Nehta.VendorLibrary.PCEHR;
using Nehta.VendorLibrary.PCEHR.DocumentRepository;

namespace HIPS.CommonBusinessLogic.Pcehr
{
    public class DownloadDocument
    {
        /// <summary>
        /// Returns the Document and Attachments from the passed in request
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="request">Request details.</param>
        /// <returns>DocumentResponse will also pass back success and error messages</returns>
        public DocumentResponse RetrieveDocument(PatientIdentifierBase patientIdentifier, UserDetails user, DocumentRequest request)
        {
            DocumentResponse response = new DocumentResponse();
            PatientAccess patientAccess = new PatientAccess(user);
            Hospital hospital;
            PatientMaster patientMaster;

            //get hospital
            HipsResponse status = patientAccess.GetHospital(patientIdentifier, out hospital);
            if (status.Status != HipsResponseIndicator.OK)
            {
                response.HipsResponse = status;
                return response;
            }

            HospitalPatient hospitalPatient;

            // Get patient ensures that we have an IHI with no outstanding alerts. It is alright if the IHI is invalid for now.
            status = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
            if (status.Status != HipsResponseIndicator.OK && status.Status != HipsResponseIndicator.InvalidIhi)
            {
                response.HipsResponse = status;
                return response;
            }

            // If the IHI was last validated outside the configured period, attempt validation.
            IhiSearchResponse ihiResponse = new PatientIhiValidation().GetValidatedIhi(patientIdentifier, hospital, user, patientMaster);
            if (ihiResponse.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                response.HipsResponse = ihiResponse.HipsResponse;
                return response;
            }

            // If the IHI is still invalid then HIPS must prevent access to the PCEHR.
            patientAccess.ValidateLocalIhiInformation(patientMaster, status);
            if (status.Status != HipsResponseIndicator.OK)
            {
                response.HipsResponse = status;
                return response;
            }

            // Populate user
            if (!User.PopulateAndValidateUser(hospital, user))
            {
                status.Status = HipsResponseIndicator.InvalidUser;
                response.HipsResponse = status;
                return response;
            }

            X509Certificate2 certificate = Helpers.GetConnectionCertificate(hospital);
            Uri uri = Helpers.GetDocumentUrl();

            // Create PCEHR header
            CommonPcehrHeader header = Helpers.GetHeader(patientIdentifier, patientMaster.Ihi, user, hospital);

            // Instantiate the client
            GetDocumentClient getDocumentClient = new GetDocumentClient(uri, certificate, certificate);

            try
            {
                System.Reflection.FieldInfo clientField = getDocumentClient.GetType().GetField("client", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
                object drc = clientField.GetValue(getDocumentClient);
                System.Reflection.FieldInfo repositoryClientField = drc.GetType().GetField("repositoryClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
                DocumentRepository_PortTypeClient ptc = repositoryClientField.GetValue(drc) as DocumentRepository_PortTypeClient;
                WSHttpBinding wsbinding = ptc.Endpoint.Binding as WSHttpBinding;

                //Set the connection timeout for the GetDocument service
                wsbinding.OpenTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentConsumptionTimeoutSeconds);
                wsbinding.ReceiveTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentConsumptionTimeoutSeconds);
                wsbinding.SendTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentConsumptionTimeoutSeconds);

                // Avoid the proxy
                if (Settings.Instance.AvoidProxy)
                {
                    wsbinding.UseDefaultWebProxy = false;
                }

                // Add server certificate validation callback
                ServicePointManager.ServerCertificateValidationCallback += ValidateServiceCertificate;

                // Create a request
                List<RetrieveDocumentSetRequestTypeDocumentRequest> retrieveDocRequest = new List<RetrieveDocumentSetRequestTypeDocumentRequest>();

                // Set the details of the document to retrieve
                retrieveDocRequest.Add(new RetrieveDocumentSetRequestTypeDocumentRequest()
                {
                    DocumentUniqueId = request.DocumentUniqueId,
                    RepositoryUniqueId = request.RepositoryUniqueId
                });

                // Invoke the service
                RetrieveDocumentSetResponseType retrieveDocResponse = getDocumentClient.GetDocument(header, retrieveDocRequest.ToArray());

                //check if there are no errors returned from the service
                if (retrieveDocResponse != null && retrieveDocResponse.RegistryResponse != null)
                {
                    if (retrieveDocResponse.RegistryResponse.RegistryErrorList == null)
                    {
                        if (retrieveDocResponse.DocumentResponse != null && retrieveDocResponse.DocumentResponse.Count() > 0)
                        {
                            // Special workaround for CCA testing of CIS_204_018634_A because
                            // we can't actually download an invalid document, but only inject
                            // it right at the point before the validation happens.
                            /*
                                 retrieveDocResponse.DocumentResponse[0].Document = System.IO.File.ReadAllBytes(
                                   @"..\..\..\Test.CommonCcaNoc\SampleDocuments\PCEHR_CIS_ConformanceTestDataID_51.zip");
                            */

                            //unpack the root document
                            CDAPackage package = CDAPackageUtility.Extract(retrieveDocResponse.DocumentResponse[0].Document, VerifyCertificate);

                            //add the unique document id and repository id
                            response.DocumentUniqueId = retrieveDocResponse.DocumentResponse[0].DocumentUniqueId;
                            response.RepositoryUniqueId = retrieveDocResponse.DocumentResponse[0].RepositoryUniqueId;

                            //add root document details
                            response.Document = package.CDADocumentRoot.FileContent;
                            response.FileName = package.CDADocumentRoot.FileName;
                            response.MimeType = System.Net.Mime.MediaTypeNames.Text.Xml;

                            //add attachments from the package
                            if (package.CDADocumentAttachments != null)
                            {
                                foreach (CDAPackageFile attachment in package.CDADocumentAttachments)
                                {
                                    Attachment responseAttachment = new Attachment();
                                    responseAttachment.FileName = attachment.FileName;
                                    responseAttachment.Contents = attachment.FileContent;
                                    response.Attachments.Add(responseAttachment);
                                }
                            }

                            if (request.SaveDocument)
                            {
                                DownloadedDocumentDl dataAccess = new DownloadedDocumentDl(user);
                                DownloadedDocument document = new DownloadedDocument();
                                document.PatientMasterId = patientMaster.PatientMasterId.Value;
                                document.Package = retrieveDocResponse.DocumentResponse[0].Document;
                                document.DownloadedDate = DateTime.Now;
                                document.SourceSystemDocumentId = request.DocumentUniqueId;
                                document.ClinicalDocumentStatusId = (int)HIPS.PcehrDataStore.Schemas.Enumerators.ClinicalDocumentStatus.Active;
                                document.CurrentDocumentId = null;
                                dataAccess.Save(document);
                            }

                            //all has succeeded so add in the HIPSResponseObject
                            response.HipsResponse = new HipsResponse(HipsResponseIndicator.OK);

                            // Now add warnings to the HipsResponse if the demographics do not match.
                            DemographicCheck demographicCheck = new DemographicCheck(patientMaster, user);
                            demographicCheck.ValidateDemographics(response.Document, response.HipsResponse);
                        }
                        else
                        {
                            //no documents can be found in the registry response
                            HipsResponse noDocumentResponse = new HipsResponse(HipsResponseIndicator.InvalidDocument);
                            noDocumentResponse.ResponseCodeDescription = "No Documents were returned with the registry response";
                            response.HipsResponse = noDocumentResponse;
                            return response;
                        }
                    }
                    else //parse for the errors
                    {
                        //fail has occurred so add in the HIPSResponseObject
                        HipsResponse ErrorResponse = new HipsResponse(HipsResponseIndicator.PcehrServiceError);
                        String errorCodeContext = retrieveDocResponse.RegistryResponse.RegistryErrorList.RegistryError[0].codeContext;
                        if (!errorCodeContext.IsNullOrEmptyWhitespace())
                        {
                            string[] errorCodeDelimiter = new string[] { " - " };
                            string[] errors = errorCodeContext.Split(errorCodeDelimiter, StringSplitOptions.None);
                            ErrorResponse.ResponseCode = errors[0];
                            ErrorResponse.ResponseCodeDescription = errors[1];
                        }
                        else
                        {
                            ErrorResponse.ResponseCode = "Unknown PCEHR Error Code";
                            ErrorResponse.ResponseCodeDescription = retrieveDocResponse.RegistryResponse.RegistryErrorList.RegistryError[0].errorCode;
                        }
                        response.HipsResponse = ErrorResponse;
                    }
                }
                else
                {
                    //no documents can be found in the registry response
                    HipsResponse noDocumentResponse = new HipsResponse(HipsResponseIndicator.InvalidDocument);
                    noDocumentResponse.ResponseCodeDescription = "No Documents were returned with the registry response";
                    response.HipsResponse = noDocumentResponse;
                }
            }
            catch (FaultException fe)
            {
                HipsResponse faultResponse = new HipsResponse(HipsResponseIndicator.PcehrServiceError);
                faultResponse.ResponseCode = fe.Reason.ToString();
                faultResponse.ResponseCodeDescription = fe.Message;
                faultResponse.ResponseCodeDetails = fe.StackTrace;
                response.HipsResponse = faultResponse;
            }
            catch (SignatureVerificationException sve)
            {
                HipsResponse invalidDocumentResponse = new HipsResponse(HipsResponseIndicator.InvalidDocument);
                invalidDocumentResponse.HipsErrorMessage = sve.Message;
                response.HipsResponse = invalidDocumentResponse;
            }
            catch (Exception e)
            {
                HipsResponse exceptionResponse = new HipsResponse(HipsResponseIndicator.SystemError);
                exceptionResponse.HipsErrorMessage = e.Message;

                //grab the inner exception if there is one
                if (e.InnerException != null)
                {
                    exceptionResponse.ResponseCodeDescription = e.InnerException.Message;
                }
                exceptionResponse.ResponseCodeDetails = e.StackTrace;
                response.HipsResponse = exceptionResponse;
            }
            finally
            {
                getDocumentClient.Close();
            }

            Helpers.InsertAudit(patientMaster, user, hospital, AuditOperationNames.RetrieveDocument, response.HipsResponse, getDocumentClient.SoapMessages);

            return response;
        }

        /// <summary>
        /// Validates the service certificate.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="certificate">The certificate.</param>
        /// <param name="chain">The chain.</param>
        /// <param name="sslPolicyErrors">The SSL policy errors.</param>
        /// <returns>True if the service certificate was valid.</returns>
        private bool ValidateServiceCertificate(object sender, X509Certificate certificate, X509Chain chain, SslPolicyErrors sslPolicyErrors)
        {
            return DocumentHelper.ValidateServiceCertificate(sender, certificate, chain, sslPolicyErrors);
        }

        /// <summary>
        /// Verifies that the provided certificate is valid.
        /// </summary>
        /// <param name="certificate">The certificate to verify.</param>
        private void VerifyCertificate(X509Certificate2 certificate)
        {
            // This is an sample certificate check, which does an online revocation check.
            // In the future, there may be CDA packages which are signed with certificates
            // which are valid at signing time, but have since been revoked or expired.
            // In this case, the certificate check should be relaxed. One such way is to
            // change the revocation mode to "no check". Eg:
            // chain.ChainPolicy.RevocationMode = X509RevocationMode.NoCheck;

            // Setup the chain
            var chain = new X509Chain();
            chain.ChainPolicy.RevocationMode = X509RevocationMode.NoCheck;
            chain.ChainPolicy.RevocationFlag = X509RevocationFlag.EndCertificateOnly;

            // Perform the validation
            chain.Build(certificate);

            // Check the results
            if (chain.ChainStatus.Length == 0)
            {
                // No errors found
            }
            else
            {
                // Errors found
            }
        }
    }
}