﻿using System.Collections.Generic;
using System.Security.Cryptography.X509Certificates;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrSchemas;
using Nehta.VendorLibrary.CDAPackage;

using DataSchemas = HIPS.PcehrDataStore.Schemas;

namespace HIPS.CommonBusinessLogic.Pcehr
{
    /// <summary>
    /// Used for Local Clinical Document Business Logic
    /// </summary>
    public class LocalClinicalDocumentLogic
    {
        /// <summary>
        /// Gets a clinical document from the local document store
        /// </summary>
        /// <param name="user">The user who is requesting the information.</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="data">The DataSchemas.PatientInHospital list data</param>
        /// <param name="sourceSystemSetId">The source system set identifier.</param>
        /// <param name="sourceSystemDocumentId">The source system document identifier.</param>
        /// <returns>
        /// A single document uploaded for the patient
        /// </returns>
        public HipsResponse GetLocalUploadedDocument(UserDetails user, PatientIdentifierBase patientIdentifier, out DataSchemas.LocalClinicalDocument data, out byte[] document, out IList<Attachment> attachments, string sourceSystemSetId, string sourceSystemDocumentId = null)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            data = null;
            document = null;
            attachments = new List<Attachment>();

            PatientAccess patientAccess = new PatientAccess(user);
            Hospital hospital;
            HipsResponse status = patientAccess.GetHospital(patientIdentifier, out hospital);
            if (status.Status != HipsResponseIndicator.OK)
            {
                response = status;
                return response;
            }

            X509Certificate2 certificate = Helpers.GetConnectionCertificate(hospital);

            patientAccess = new PatientAccess(user);
            HospitalPatient hospitalPatient;
            PatientMaster patientMaster;
            status = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
            if (status.Status != HipsResponseIndicator.OK && status.Status != HipsResponseIndicator.InvalidIhi)
            {
                response = status;
                return response;
            }

            ClinicalDocumentDl dataAccess = new ClinicalDocumentDl(user);
            response = dataAccess.GetLocalUploadedDocument((int)hospitalPatient.PatientId, out data, sourceSystemSetId, sourceSystemDocumentId);

            if (data != null && data.Package != null)
            {
                //unpack the root document
                CDAPackage package = CDAPackageUtility.Extract(data.Package, VerifyCertificate);
                document = package.CDADocumentRoot.FileContent;

                //add attachments from the package
                if (package.CDADocumentAttachments != null)
                {
                    foreach (CDAPackageFile attachment in package.CDADocumentAttachments)
                    {
                        Attachment responseAttachment = new Attachment();
                        responseAttachment.FileName = attachment.FileName;
                        responseAttachment.Contents = attachment.FileContent;
                        attachments.Add(responseAttachment);
                    }
                }
            }
            else
            {
                response = new HipsResponse(HipsResponseIndicator.InvalidDocument);
                if (sourceSystemDocumentId != null)
                {
                    response.HipsErrorMessage = string.Format(ResponseStrings.LocalDocumentVersionNotFound, sourceSystemSetId, sourceSystemDocumentId);
                }
                else
                {
                    response.HipsErrorMessage = string.Format(ResponseStrings.LocalDocumentSetNotFound, sourceSystemSetId);
                }
            }

            return response;
        }

        /// <summary>
        /// Lists the clinical documents for a patient in the local document store
        /// </summary>
        /// <param name="user">The user who is requesting the information.</param>
        /// <param name="sourceSystemSetId">The source system set identifier.</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="data">The DataSchemas.PatientInHospital list data</param>
        /// <returns>
        /// A list of the documents uploaded for the patient
        /// </returns>
        public HipsResponse ListLocalUploadedDocuments(UserDetails user, PatientIdentifierBase patientIdentifier, out List<DataSchemas.LocalClinicalDocumentMetaData> data, string sourceSystemSetId)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            data = null;

            PatientAccess patientAccess = new PatientAccess(user);
            Hospital hospital;
            HipsResponse status = patientAccess.GetHospital(patientIdentifier, out hospital);
            if (status.Status != HipsResponseIndicator.OK)
            {
                response = status;
                return response;
            }

            patientAccess = new PatientAccess(user);
            HospitalPatient hospitalPatient;
            PatientMaster patientMaster;
            status = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
            // List all the patients and documents even if they have a status of InvalidIhi and UnresolveIhiAlert
            // We just don't action anything with an UnresolveIhiAlert status
            if (status.Status != HipsResponseIndicator.OK && status.Status != HipsResponseIndicator.InvalidIhi && status.Status != HipsResponseIndicator.UnresolvedIhiAlert)
            {
                response = status;
                return response;
            }

            ClinicalDocumentDl dataAccess = new ClinicalDocumentDl(user);
            response = dataAccess.ListLocalUploadedDocuments((int)hospitalPatient.PatientId, out data, sourceSystemSetId);

            if (response.Status == HipsResponseIndicator.OK && status.Status == HipsResponseIndicator.UnresolvedIhiAlert)
            {
                response.Status = status.Status;
            }

            return response;
        }

        private void VerifyCertificate(X509Certificate2 certificate)
        {
            // This is an sample certificate check, which does an online revocation check.
            // In the future, there may be CDA packages which are signed with certificates
            // which are valid at signing time, but have since been revoked or expired.
            // In this case, the certificate check should be relaxed. One such way is to
            // change the revocation mode to "no check". Eg:
            // chain.ChainPolicy.RevocationMode = X509RevocationMode.NoCheck;

            // Setup the chain
            var chain = new X509Chain();
            chain.ChainPolicy.RevocationMode = X509RevocationMode.NoCheck;
            chain.ChainPolicy.RevocationFlag = X509RevocationFlag.EndCertificateOnly;

            // Perform the validation
            chain.Build(certificate);

            // Check the results
            if (chain.ChainStatus.Length == 0)
            {
                // No errors found
            }
            else
            {
                // Errors found
            }
        }
    }
}