﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using HIPS.CommonBusinessLogic.Utility;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.Cda;
using HIPS.CommonSchemas.Exceptions;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrSchemas;
using MDM.Generator;
using MDM.Model.HL7Model;
using Nehta.VendorLibrary.CDAPackage;
using Nehta.VendorLibrary.Common;
using HIPS.CommonBusinessLogic.Singleton;

namespace HIPS.CommonBusinessLogic.Service
{
    /// <summary>
    /// Business service for interacting with CDA documents.
    /// </summary>
    public class CdaDocumentService : BusinessServiceBase
    {
        /// <summary>
        /// Initialises a new instance of the <see cref="CdaDocumentService" /> class.
        /// </summary>
        /// <param name="user">End-user context within which the class is being accessed.</param>
        public CdaDocumentService(UserDetails user)
            : base(user)
        {
        }

        #region Methods

        /// <summary>
        /// Generates a CDA Package, either XDM-ZIP or XDM-ZIP with an HL7-MDM wrapper.
        /// The document will be loaded from the Content into the Document, and both
        /// may be modified slightly to include the integrity checksum for the logo.
        /// </summary>
        /// <param name="metadata">The metadata required for packaging the document.</param>
        /// <returns>The CDA package.</returns>
        public byte[] Package(CdaMetadata metadata)
        {
            LookupAndValidateHospital(metadata);
            if (metadata.Format == CdaPackageFormat.None)
            {
                return metadata.Document.Content;
            }

            metadata.Document.Load();

            X509Certificate2 signingCert = this.GetHpioCertificate(metadata.Hospital);
            Approver approver = metadata.Document.GetAuthor().ToApprover();
            var package = new CDAPackage(approver);
            CdaDocumentUtility.ProcessLogo(metadata);

            // If attachments were supplied by the calling application, add them to the package.
            foreach (HIPS.PcehrSchemas.Attachment a in metadata.Document.Attachments)
            {
                package.AddDocumentAttachment(
                    a.FileName,
                    a.Contents
                );
            }

            // Create the CDA root document and create the CDA package.
            package.CreateRootDocument(metadata.Document.Content);
            byte[] packageContent = CDAPackageUtility.Create(package, signingCert);

            if (metadata.Format == CdaPackageFormat.Hl7Mdm)
            {
                packageContent = CreateHl7MdmWrapper(metadata, packageContent);
            }
            return packageContent;
        }

        /// <summary>
        /// Unpackages a CDA document from either an HL7 MDM message or an XDM ZIP file.
        /// </summary>
        /// <param name="packageContent">Binary content of the package.</param>
        /// <param name="format">Format in which the document is packaged.</param>
        /// <returns>Unpackaged CDA document.</returns>
        public CdaDocument Unpackage(byte[] packageContent, CdaPackageFormat format)
        {
            CDAPackage package;

            if (format == CdaPackageFormat.None)
            {
                return new CdaDocument() { Content = packageContent };
            }
            else if (format == CdaPackageFormat.Hl7Mdm)
            {
                packageContent = MDMGenerator.ExtractZipFile(UTF8Encoding.UTF8.GetString(packageContent));
            }

            // This can throw an exception back to the service layer.
            package = CDAPackageUtility.Extract(packageContent, VerifyCertificate);

            return new CdaDocument()
            {
                Content = package.CDADocumentRoot.FileContent,
                Attachments = package.CDADocumentAttachments.Select(a => new HIPS.PcehrSchemas.Attachment()
                {
                    Contents = a.FileContent,
                    FileName = a.FileName
                }).ToList()
            };
        }

        /// <summary>
        /// Get the content from a provided CDA unpackaged document.
        /// </summary>
        /// <param name="document">The CDA document to extract content from.</param>
        /// <returns>CDA document content.</returns>
        public CdaDocumentContent GetContent(CdaDocument document)
        {
            document.Load();
            return new CdaDocumentContent()
            {
                Author = document.GetAuthorProvider(),
                DocumentType = document.GetDocumentTypeCode(),
                Patient = document.GetPatientConsumer()
            };
        }

        /// <summary>
        /// Attempts to look up the hospital using the hospital code and code system
        /// provided in the request.
        /// </summary>
        /// <param name="metadata">The CDA packaging metadata.</param>
        /// <exception cref="HIPS.CommonSchemas.Exceptions.HospitalNotFoundException">The hospital was not found.</exception>
        private void LookupAndValidateHospital(CdaMetadata metadata)
        {
            metadata.Hospital = HospitalSingleton.Value.Find(metadata.HospitalCode, metadata.HospitalCodeSystem);
            if (metadata.Hospital == null)
            {
                throw new HospitalNotFoundException();
            }
        }

        /// <summary>
        /// Creates an HL7 MDM wrapper around the package.
        /// </summary>
        /// <param name="metadata">The metadata required for packaging the document.</param>
        /// <param name="packageContent">The XDM ZIP package.</param>
        /// <returns>The HL7 MDM package.</returns>
        /// <exception cref="HIPS.CommonSchemas.Exceptions.MdmValidationException">MDM model validation failed.</exception>
        private byte[] CreateHl7MdmWrapper(CdaMetadata metadata, byte[] packageContent)
        {
            try
            {
                HL7Model mdmModel = metadata.ToMdmModel();
                mdmModel.OBX.ZipPackageAsBase64String = Convert.ToBase64String(packageContent);
                var generatedMessage = MDMGenerator.CreateHL7Message(mdmModel);
                return UTF8Encoding.UTF8.GetBytes(generatedMessage);
            }
            catch (ValidationException ex)
            {
                throw new MdmValidationException(ex.GetMessagesString());
            }
        }

        /// <summary>
        /// Gets the NASH certificate for the Hospital.
        /// </summary>
        /// <param name="hospital">The hospital.</param>
        /// <returns>X509Certificate2</returns>
        /// <exception cref="System.Exception">PCEHR certificate Serial has not been found via the hospital</exception>
        private X509Certificate2 GetHpioCertificate(Hospital hospital)
        {
            string serialNumber = hospital.PcehrCertSerial;
            if (string.IsNullOrEmpty(serialNumber))
            {
                throw new Exception("PCEHR certificate Serial has not been found via the hospital");
            }

            // Obtain the certificate for use with TLS
            return X509CertificateUtil.GetCertificate(serialNumber, X509FindType.FindBySerialNumber, StoreName.My, StoreLocation.LocalMachine, false);
        }

        /// <summary>
        /// This is a sample certificate check, which does an online revocation check.
        /// In the future, there may be CDA packages which are signed with certificates
        /// which are valid at signing time, but have since been revoked or expired.
        /// In this case, the certificate check should be relaxed. One such way is to
        /// change the revocation mode to "no check". Eg:
        /// chain.ChainPolicy.RevocationMode = X509RevocationMode.NoCheck;
        /// </summary>
        /// <param name="certificate">The certificate to verify.</param>
        private void VerifyCertificate(X509Certificate2 certificate)
        {
            // Setup the chain
            var chain = new X509Chain();
            chain.ChainPolicy.RevocationMode = X509RevocationMode.Online;
            chain.ChainPolicy.RevocationFlag = X509RevocationFlag.EndCertificateOnly;

            // Perform the validation
            chain.Build(certificate);

            // Check the results
            if (chain.ChainStatus.Length == 0)
            {
                // No errors found
            }
            else
            {
                IEnumerable<string> values = chain.ChainStatus.Select(a => a.StatusInformation);
                string msg = string.Join(", ", values);
                throw new InvalidCertificateException(msg);
            }
        }

        #endregion Methods

    }
}
