﻿using HIPS.CommonBusinessLogic.Cda;
using HIPS.CommonBusinessLogic.HL7;
using HIPS.CommonBusinessLogic.Mapping;
using HIPS.CommonBusinessLogic.Pcehr;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.Exceptions;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Configuration.Tracing;
using HIPS.HL7.Common;
using HIPS.HL7.Common.DataStructure;
using HIPS.HL7.Common.Message;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace HIPS.CommonBusinessLogic.Service
{
    /// <summary>
    /// Business Service to assist upload of Pathology and Diagnostic Imaging Reports
    /// </summary>
    public class PathologyImagingService : BusinessServiceBase
    {
        private UserDetails user;

        /// <summary>
        /// Initialises a new instance of the <see cref="PathologyImagingService" /> class.
        /// </summary>
        /// <param name="user"></param>
        public PathologyImagingService(UserDetails user)
            : base(user)
        {
        }

        #region Public methods

        /// <summary>
        /// Initial validation of the Pathology request and adds the request to the PCEHR Queue for further validation and processing.
        /// Covers the following requirements FR-PDI-4.3, FR-PDI-4.4 and CN-PDI-2.3.
        /// </summary>
        /// <param name="hl7Message">The HL7 Message passed in from the Web Service method</param>
        /// <param name="report">Optional. The attached PDF report represented as bytes</param>
        /// <param name="reportLocation">Optional. The full file path as reportLocation if a reference pointer is used</param>
        /// <returns>HipsResponse object</returns>
        public HipsResponse QueuePathology(string hl7Message, byte[] report, string reportLocation)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            try
            {
                // Validation: The contents of the message form data parses as a valid HL7® message.
                HL7Message message = HL7Message.Parse(hl7Message);
                if (!(message is HL7GenericPathMessage))
                {
                    response = new HipsResponse(HipsResponseIndicator.ValidationError, string.Format("Message rejected, incorrect message type: {0}", message.GetType().Name));
                    throw new HipsResponseException(response);
                }
                HL7GenericPathMessage genericPathMessage = message as HL7GenericPathMessage;

                // Validate the Order - throw an exception if it doesn't validate
                bool isAllResultCancelled = true;
                response = ValidatePathologyOrder(genericPathMessage, out isAllResultCancelled);
                if (response.Status != HipsResponseIndicator.OK)
                    throw new HipsResponseException(response);

                // If all the results are cancelled, remove the document by invoking RemoveDocument and setting the removal reason to withdraw. This method will then exit.
                if (isAllResultCancelled)
                {
                    response = RemovePathologyOrImagingDocument(hl7Message, DocumentTypeCodes.PathologyReport);
                    return response;
                }

                // Get the PDF Report in bytes, if no report is retrieved then throw an exception
                byte[] pdfReport = report;
                if (pdfReport == null)
                { 
                    response = GetPdfReport(genericPathMessage, reportLocation, DocumentTypeCodes.PathologyReport, out pdfReport);
                    if (response.Status != HipsResponseIndicator.OK)
                        throw new HipsResponseException(response);
                }
                
                // Invoke the PopulateHL7MessageLog to create a message log entity.
                HL7MessageLog messageLog = MessageLoader.PopulateHL7MessageLog(message, hl7Message, this.UserContext);

                // Set the queue status of the message log to pending.
                messageLog.QueueStatusId = (int)QueueStatus.Pending;

                // Invoke the Insert method of the HL7MessageLogDl to save the message log.
                HL7MessageLogDl messageLogDataAccess = new HL7MessageLogDl(this.UserContext);
                messageLogDataAccess.Insert(messageLog);

                // Invoke the SendPathologyRequest method in the PcehrQueueLogic class providing the parsed HL7® message and the report.
                HL7ReportUploadBeforeQueue reportUploader = new HL7ReportUploadBeforeQueue(this.UserContext, hl7Message, pdfReport, messageLog.HL7MessageLogId.Value, HIPS.Configuration.Settings.Instance.PathologyReportDocumentFormatCode);
                response = reportUploader.SendHl7ReportRequest();
                
            }
            catch (HipsResponseException ex)
            {
                // This exception is designed to be thrown up to the service layer and mapped by the fault profile.
                throw ex;
            }
            catch (Exception ex)
            {
                response.Status = HipsResponseIndicator.SystemError;
                response.HipsErrorMessage = ex.Message;
                response.ResponseCodeDetails = ex.InnerException != null ? ex.InnerException.Message : null;
                response.ResponseCode = ex.GetType().Name;
                throw new HipsResponseException(response);
            }
            return response;
            
        }

        /// <summary>
        /// Initial validation of the Diagnostic Imaging request and adds the request to the PCEHR Queue for further validation and processing.
        /// Covers the following requirements FR-PDI-4.3, FR-PDI-4.4 and CN-PDI-2.3..
        /// </summary>
        /// <param name="hl7Message">The HL7 Message passed in from the Web Service method</param>
        /// <param name="report">Optional. The attached PDF report represented as bytes</param>
        /// <param name="reportLocation">Optional. The full file path as reportLocation if a reference pointer is used</param>
        /// <returns>HipsResponse object</returns>
        public HipsResponse QueueImaging(string hl7Message, byte[] report, string reportLocation)
        {
            using (HIPSTraceWriter trace = new HIPSTraceWriter())
            {
                trace.WriteTrace(string.Format("IN HIPS.CommonBusinessLogic.Service.PathologyImagingService.QueueImaging:: hl7Message: {0}", hl7Message), System.Diagnostics.TraceEventType.Information);
                HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
                HL7MessageLogDl messageLogDataAccess = new HL7MessageLogDl(user);
                try
                {

                    // Validation: The contents of the message form data parses as a valid HL7® message.
                    HL7Message message = HL7Message.Parse(hl7Message);
                    if (!(message is HL7GenericPathMessage))
                    {
                        response = new HipsResponse(HipsResponseIndicator.ValidationError, string.Format("Message rejected, incorrect message type: {0}", message.GetType().Name));
                        throw new HipsResponseException(response);
                    }
                    HL7GenericPathMessage genericPathMessage = message as HL7GenericPathMessage;

                    // Validate the Order - throw an exception if it doesn't validate
                    bool isAllResultCancelled = true;
                    response = ValidateImagingOrder(genericPathMessage, out isAllResultCancelled);
                    if (response.Status != HipsResponseIndicator.OK)
                        throw new HipsResponseException(response);

                    // If all the results are cancelled, remove the document by invoking RemoveDocument and setting the removal reason to withdraw. This method will then exit.
                    if (isAllResultCancelled)
                    {
                        response = RemovePathologyOrImagingDocument(hl7Message, DocumentTypeCodes.DiagnosticImagingReport);
                        return response;
                    }

                    // Get the PDF Report in bytes, if no report is retrieved then throw an exception
                    byte[] pdfReport = report;
                    if (pdfReport == null)
                    {
                        response = GetPdfReport(genericPathMessage, reportLocation, DocumentTypeCodes.DiagnosticImagingReport, out pdfReport);
                        if (response.Status != HipsResponseIndicator.OK)
                            throw new HipsResponseException(response);
                    }

                    // Invoke the PopulateHL7MessageLog to create a message log entity.
                    HL7MessageLog messageLog = MessageLoader.PopulateHL7MessageLog(message, hl7Message, this.UserContext);
                    
                    // Set the queue status of the message log to pending.
                    messageLog.QueueStatusId = (int)QueueStatus.Pending;

                    // Invoke the Insert method of the HL7MessageLogDl to save the message log.
                    if (!messageLogDataAccess.Insert(messageLog))
                    {
                        throw new Exception(string.Format(ConstantsResource.DatabaseError, messageLogDataAccess.GetType().FullName));
                    }
                    
                    // Invoke the SendPathologyRequest method in the PcehrQueueLogic class providing the parsed HL7® message and the report.
                    HL7ReportUploadBeforeQueue reportUploader = new HL7ReportUploadBeforeQueue(this.UserContext, hl7Message, pdfReport, messageLog.HL7MessageLogId.Value, HIPS.Configuration.Settings.Instance.DiagnosticImagingReportDocumentFormatCode);
                    response = reportUploader.SendHl7ReportRequest();
                    
                }
                catch (HipsResponseException ex)
                {
                    // This exception is designed to be thrown up to the service layer and mapped by the fault profile.
                    trace.WriteTrace(string.Format("ERROR HIPS.CommonBusinessLogic.Service.PathologyImagingService.QueueImaging:: hl7Message: {0}. HipsResponseException Details: {1}", hl7Message, ex.ToString()), System.Diagnostics.TraceEventType.Error);
                    throw ex;
                }
                catch (Exception ex)
                {
                    trace.WriteTrace(string.Format("ERROR HIPS.CommonBusinessLogic.Service.PathologyImagingService.QueueImaging:: hl7Message: {0}. Exception Details: {1}", hl7Message, ex.ToString()), System.Diagnostics.TraceEventType.Error);
                    response.Status = HipsResponseIndicator.SystemError;
                    response.HipsErrorMessage = ex.Message;
                    response.ResponseCodeDetails = ex.InnerException != null ? ex.InnerException.Message : null;
                    response.ResponseCode = ex.GetType().Name;
                    throw new HipsResponseException(response);
                }

                trace.WriteTrace(string.Format("OUT HIPS.CommonBusinessLogic.Service.PathologyImagingService.QueueImaging:: hl7Message: {0}", hl7Message), System.Diagnostics.TraceEventType.Information);

                return response;

            }
        }

        
        #endregion

        #region Private methods

        /// <summary>
        /// Validates the Pathology Order details.
        /// </summary>
        /// <param name="genericPathMessage">The HL7 Pathology Message</param>
        /// <param name="isAllResultCancelled">Out parameter to indicate if this is a removal or not</param>
        /// <returns>HipsResponse object</returns>
        private HipsResponse ValidatePathologyOrder(HL7GenericPathMessage genericPathMessage, out bool isAllResultCancelled)
        {
            HipsResponse validateResponse = new HipsResponse(HipsResponseIndicator.OK);
            // Set the REsults cancelled to true, will return false if any of the OBR's do not contain an 'X'
            isAllResultCancelled = true;

            // Check that the OrderGroup exists (HL7Message has all required segments)
            if (genericPathMessage.Order.Count() <= 0 )
            {
                validateResponse.Status = HipsResponseIndicator.ValidationError;
                validateResponse.HipsErrorMessage = "Message rejected, HL7 Message does not contain the required Order segments.";
                return validateResponse;
            }
            // For each of the OrderGroups
            foreach (var order in genericPathMessage.Order)
            {
                // Check that the Observation segment exists (HL7Message has all required segments)
                if (order.Observation.Count() <= 0)
                {
                    validateResponse.Status = HipsResponseIndicator.ValidationError;
                    validateResponse.HipsErrorMessage = "Message rejected, HL7 Message does not contain the required Order segments.";
                    return validateResponse;
                }
                // For each of the Observation groups
                foreach (var observation in order.Observation)
                {
                    // Check that the OBR segment exists (HL7Message has all required segments)
                    if (observation.ObservationsReportID == null)
                    {
                        validateResponse.Status = HipsResponseIndicator.ValidationError;
                        validateResponse.HipsErrorMessage = "Message rejected, HL7 Message does not contain the required OBR segments.";
                        return validateResponse;
                    }
                    
                    // Get the OBR record
                    HIPS.HL7.Common.Segment.OBR obr = observation.ObservationsReportID;

                    // Check that the DiagnosticServSectID from each OBR record matches a code in the DiagnosticServicesEnum.
                    if (!Enum.IsDefined(typeof(PathologyDiscipline), obr.DiagnosticServSectID.ToUpper()))
                    {
                        validateResponse.Status = HipsResponseIndicator.ValidationError;
                        validateResponse.HipsErrorMessage = string.Format("Message rejected, DiagnosticServSectID {0} does not exist.", obr.DiagnosticServSectID);
                        return validateResponse;
                    }

                    // Check the status of each OBR record and if all results are cancelled 
                    if (obr.ResultStatus.ToUpper() != "X")
                        isAllResultCancelled = false;
                }

                // Check that there is only 1 ordering provider
                validateResponse = CheckOrderingProvider(order);
                
            }
            return validateResponse;
        }

        /// <summary>
        /// Validates a Diagnostic Imaging Order details
        /// </summary>
        /// <param name="genericPathMessage">HL7 Message</param>
        /// <param name="isAllResultCancelled">Out. Returns true if all of the Results are cancelled, false otherwise</param>
        /// <returns>HipsResponse object</returns>
        private HipsResponse ValidateImagingOrder(HL7GenericPathMessage genericPathMessage, out bool isAllResultCancelled)
        {
            HipsResponse validateResponse = new HipsResponse(HipsResponseIndicator.OK);

            // Set the Results cancelled to true, will return false if any of the OBR's do not contain an 'X'
            isAllResultCancelled = true;

            // Check that the OrderGroup exists (HL7Message has all required segments)
            if (genericPathMessage.Order.Count() <= 0)
            {
                validateResponse.Status = HipsResponseIndicator.ValidationError;
                validateResponse.HipsErrorMessage = "Message rejected, HL7 Message does not contain the required Order segments.";
                return validateResponse;
            }
            // For each of the OrderGroups
            foreach (var order in genericPathMessage.Order)
            {
                // Check that the Observation segment exists (HL7Message has all required segments)
                if (order.Observation.Count() <= 0)
                {
                    validateResponse.Status = HipsResponseIndicator.ValidationError;
                    validateResponse.HipsErrorMessage = "Message rejected, HL7 Message does not contain the required Order segments.";
                    return validateResponse;
                }
                // For each of the Observation groups
                foreach (var observation in order.Observation)
                {
                    // Check that the Observation segment exists (HL7Message has all required segments)
                    if (observation.ObservationsReportID == null)
                    {
                        validateResponse.Status = HipsResponseIndicator.ValidationError;
                        validateResponse.HipsErrorMessage = "Message rejected, HL7 Message does not contain the required Order segments.";
                        return validateResponse;
                    }
                    
                    // Get the OBR record
                    HIPS.HL7.Common.Segment.OBR obr = observation.ObservationsReportID;

                    // Check the status of each OBR record and if all results are cancelled 
                    if (obr.ResultStatus.ToUpper() != "X")
                        isAllResultCancelled = false;
                }

                // Check that there is only 1 ordering provider
                validateResponse = CheckOrderingProvider(order);
                if (validateResponse.Status != HipsResponseIndicator.OK)
                    return validateResponse;

                // Check that there is only 1 Filler Order Number to populate Acession Number with
                validateResponse = CheckFillerOrderNumber(order);
                if (validateResponse.Status != HipsResponseIndicator.OK)
                    return validateResponse;

            }
            return validateResponse;
        }

        /// <summary>
        /// Check that the Order only contains one Ordering Provider.
        /// FR-PDI-15.1	The eHISC service will reject messages that contain multiple OBR results with different individuals in OBR-16 Requester (ordering provider).
        /// </summary>
        /// <param name="order"></param>
        /// <returns>HipsResponse object</returns>
        private HipsResponse CheckOrderingProvider(HIPS.HL7.Common.SegmentGroup.OrderGroup order)
        {
            HipsResponse orderingProviderResponse = new HipsResponse(HipsResponseIndicator.OK);
            
            // Check that the HL7 message contains ordering providers
            if (order.Observation.Any(i => (i.ObservationsReportID.OrderingProvider == null || string.IsNullOrEmpty(i.ObservationsReportID.OrderingProvider.IDnumberST) || i.ObservationsReportID.OrderingProvider.familylastname == null)))
            {
                orderingProviderResponse.Status = HipsResponseIndicator.ValidationError;
                orderingProviderResponse.HipsErrorMessage = "The HL7 message must contain a valid Ordering Provider. Report cannot be uploaded to the PCEHR.";
                return orderingProviderResponse;
            }
            // Check that the ordering provider of the all OBR records match.
            var distinctOrderingProviders = order.Observation.GroupBy(
                                                        i => string.Format("{0} {1} {2} {3}", i.ObservationsReportID.OrderingProvider.IDnumberST, i.ObservationsReportID.OrderingProvider.familylastname.familyname, i.ObservationsReportID.OrderingProvider.givenname, i.ObservationsReportID.OrderingProvider.prefix),
                                                        (key, group) => group.First()
                                                        ).ToArray();
            if (distinctOrderingProviders.Count() > 1)
            {
                orderingProviderResponse.Status = HipsResponseIndicator.ValidationError;
                orderingProviderResponse.HipsErrorMessage = "There is more than one Ordering Provider. Report cannot be uploaded to the PCEHR.";
            }

            return orderingProviderResponse;
        }
        /// <summary>
        /// Check that the Order only contains one FillerOrderNumber (for Diagnostic Imaging Orders)
        /// </summary>
        /// <param name="order">The Order Group section from the HL7 Message</param>
        /// <returns>HipsResponse object</returns>
        private HipsResponse CheckFillerOrderNumber(HIPS.HL7.Common.SegmentGroup.OrderGroup order)
        {
            HipsResponse fonResponse = new HipsResponse(HipsResponseIndicator.OK);

            // Check that the ordering provider of the all OBR records match.
            var distinctFillerOrderNumbers = order.Observation.GroupBy(
                                                        i => string.Format("{0}", i.ObservationsReportID.FillerOrderNumber.entityidentifier),
                                                        (key, group) => group.First()
                                                        ).ToArray();
            if (distinctFillerOrderNumbers.Count() > 1)
            {
                fonResponse.Status = HipsResponseIndicator.ValidationError;
                fonResponse.HipsErrorMessage = "There is more than one Filler Order Number. Report cannot be uploaded to the PCEHR.";
            }

            return fonResponse;
        }

        /// <summary>
        /// Gets the PDF report either from the report passed in, Report location passed in or the OBX segment of the HL7Message
        /// </summary>
        /// <param name="genericPathMessage">HL7 Message</param>
        /// <param name="reportLocation">The report location (full file name)</param>
        /// <param name="pdfReport">Out. The returned pdfReport in bytes</param>
        /// <returns>HipsResponse object</returns>
        private HipsResponse GetPdfReport(HL7GenericPathMessage genericPathMessage, string reportLocation, string docTypeCode, out byte[] pdfReport)
        {
            HipsResponse reportResponse = new HipsResponse(HipsResponseIndicator.OK);
            pdfReport = null;

            // If a report location is provided then retrieve the report from the specified location.
            reportResponse = GetPdfReportFromLocation(reportLocation, out pdfReport);
            if (reportResponse.Status != HipsResponseIndicator.OK)
                return reportResponse;

            // If report and report location are not provided then extract the report or report location from the OBX segment. 
            // If it is a report location then retrieve the report from the specified location.
            if (pdfReport == null && string.IsNullOrEmpty(reportLocation))
            {
                // For each of the OrderGroups
                foreach (var order in genericPathMessage.Order)
                {
                    // For each of the Observation groups
                    foreach (var observation in order.Observation)
                    {
                        // Check that the OBX segment exists (HL7Message has all required segments)
                        if (observation.Result.Count() > 0)
                        {
                            // If the ValueType of the OBX is ‘ED’ then the PDF is ‘Embedded Data’ and is within the ObservationValue[3] field encoded as Base64, so you will need to decode and store as a PDF.
                            if (observation.Result.Any(i => i.ValueType.ToUpper() == "ED" && i.ObservationIdentifier.identifier.ToUpper() == "PDF"))
                            {
                                if (string.IsNullOrEmpty(observation.Result.Where(i => i.ValueType.ToUpper() == "ED" && i.ObservationIdentifier.identifier.ToUpper() == "PDF").First().ObservationValue[4]))
                                {
                                    reportResponse.Status = HipsResponseIndicator.ValidationError;
                                    reportResponse.HipsErrorMessage = "Message rejected, embedded report field is empty.";
                                }
                                pdfReport = Convert.FromBase64String(observation.Result.Where(i => i.ValueType.ToUpper() == "ED" && i.ObservationIdentifier.identifier.ToUpper() == "PDF").First().ObservationValue[4]);
                            }

                            // If the ValueType of the OBX is ‘RP’ then the PDF is ‘Reference Pointer’ and 
                            // is stored on a file directory and the filename of the file is within the ObservationValue[0] field. 
                            // A new configurable setting needs to be added that is the Directory Path for the PDI files.
                            if (observation.Result.Any(i => i.ValueType.ToUpper() == "RP" && i.ObservationIdentifier.identifier.ToUpper() == "PDF"))
                            {
                                if (string.IsNullOrEmpty(observation.Result.Where(i => i.ValueType.ToUpper() == "RP" && i.ObservationIdentifier.identifier.ToUpper() == "PDF").First().ObservationValue[0]))
                                {
                                    reportResponse.Status = HipsResponseIndicator.ValidationError;
                                    reportResponse.HipsErrorMessage = "Message rejected, report location field is empty.";
                                }
                                string reportPath = (docTypeCode == DocumentTypeCodes.PathologyReport.ToString() ? Helpers.GetPathologyReportsUploadPath() : Helpers.GetImagingReportUploadLocation());
                                reportLocation = reportPath + observation.Result.Where(i => i.ValueType.ToUpper() == "RP" && i.ObservationIdentifier.identifier.ToUpper() == "PDF").First().ObservationValue[0].ToString();
                                reportResponse = GetPdfReportFromLocation(reportLocation, out pdfReport);
                                if (reportResponse.Status != HipsResponseIndicator.OK)
                                    return reportResponse;
                            }
                        }
                    }
                }
            }
            // If the report cannot be retrieved throw an exception
            if (pdfReport == null)
            {
                reportResponse.Status = HipsResponseIndicator.ValidationError;
                reportResponse.HipsErrorMessage = "Message rejected, report cannot be retrieved.";
            }
            return reportResponse;
        }

        /// <summary>
        /// Reads the PDF from the reportLocation and loads the byte array into the pdfReport output parameter
        /// </summary>
        /// <param name="reportLocation">The full file path of the report</param>
        /// <param name="pdfReport">Out. The PDF Report in bytes</param>
        /// <returns>HipsReponse object</returns>
        private HipsResponse GetPdfReportFromLocation(string reportLocation, out byte[] pdfReport)
        {
            HipsResponse reportResponse = new HipsResponse(HipsResponseIndicator.OK);
            pdfReport = null;

            // Check that we have a report Location
            if (!string.IsNullOrEmpty(reportLocation))
            {
                // Check that a filed exists at the given location
                if (!File.Exists(reportLocation))
                {
                    reportResponse.Status = HipsResponseIndicator.ValidationError;
                    reportResponse.HipsErrorMessage = string.Format("Report cannot be found using the provided report location: {0}", reportLocation);
                    return reportResponse;
                }
                // try and read the report from the report Location
                try
                {
                    pdfReport = File.ReadAllBytes(reportLocation);
                }
                catch (Exception ex)
                {
                    reportResponse.Status = HipsResponseIndicator.SystemError;
                    reportResponse.HipsErrorMessage = string.Format("Report cannot be read from the provided report location: {0}", reportLocation);
                    reportResponse.ResponseCodeDetails = ex.InnerException != null ? ex.InnerException.Message : null;
                    reportResponse.ResponseCode = ex.GetType().Name;
                    return reportResponse;
                }
            }
            return reportResponse;
        }

        /// <summary>
        /// Retrieves and validates the remove document information from the HL7 message.
        /// Invokes the Remove document logic to place a Remove Operation on the PCEHR MSMQ.
        /// </summary>
        /// <param name="hl7Message">The HL7Message passed in from the web service</param>
        /// <param name="docType">The document type, either Pathology or Diagnostic Imaging</param>
        /// <returns>HipsResponse object</returns>
        private HipsResponse RemovePathologyOrImagingDocument(string hl7Message, string docType)
        {
            HipsResponse removeResponse = new HipsResponse(HipsResponseIndicator.OK);

            // Parse the HL7 message and Populate the HL7MessageLog    
            HL7MessageLog messageLog;
            HL7Message message = ParseAndLogHL7Message(hl7Message, out messageLog);

            if (!(message is HL7GenericPathMessage))
            {
                removeResponse = new HipsResponse(HipsResponseIndicator.ValidationError, string.Format("Message rejected, incorrect message type: {0}", message.GetType().Name));
                throw new HipsResponseException(removeResponse);
            }
            HL7GenericPathMessage genericPathMessage = message as HL7GenericPathMessage;

            // Extract Patient details
            HL7GenericPasMessage genericPasMessage = new HL7GenericPasMessage();
            genericPasMessage.PatientIdentification = genericPathMessage.PatientIdentification;
            CX mrn, statePatientId;
            PasLoader.ExtractPatientIdentifiers(genericPasMessage, out statePatientId, out mrn);
            Mrn patientMrn = new Mrn(mrn.ID, message.MessageHeader.SendingFacility.universalID, HL7.ConstantsResource.PatientAdministrationSystemHospitalCodeSystem);
            var patientIdentifier = ObjectMapper.Map<HIPS.CommonSchemas.PatientIdentifier.PatientIdentifierBase>(patientMrn);

            // Lookup Hospital
            CodeSystemDl codeSystemDl = new CodeSystemDl();
            CodeSystem codeSystem = codeSystemDl.GetAll().Where(i => i.Code == HL7.ConstantsResource.PatientAdministrationSystemHospitalCodeSystem).FirstOrDefault();
            HospitalDl hospitalDl = new HospitalDl();
            Hospital hospital = hospitalDl.Get(codeSystem.CodeSystemId.Value, message.MessageHeader.SendingFacility.universalID);

            // Extract FillerOrderNumber
            string fillerOrderNumber = string.Empty;
            foreach (var order in genericPathMessage.Order)
            {
                foreach (var observation in order.Observation)
                {
                    var obr = observation.ObservationsReportID;
                    // Populate the Filler Order Number array
                    fillerOrderNumber = obr.FillerOrderNumber.entityidentifier;
                }
            }
            if (string.IsNullOrEmpty(fillerOrderNumber))
            {
                removeResponse.Status = HipsResponseIndicator.ValidationError;
                removeResponse.HipsErrorMessage = "Filler Order Number could not be found. The document cannot be removed from the PCEHR.";
                return removeResponse;
            }
            
            
            // Get Episode By Filler Order Number
            EpisodeDl episodeDataAccess = new EpisodeDl(this.UserContext);
            List<Episode> episodeList = episodeDataAccess.GetAllEpisodeByFillerOrderNumber(fillerOrderNumber);
            if (episodeList.Count <= 0)
            {
                removeResponse.Status = HipsResponseIndicator.InvalidEpisode;
                removeResponse.HipsErrorMessage = "Patient Episode could not be found. The document cannot be removed from the PCEHR.";
                return removeResponse;
            }
            Episode episode = episodeList.First();
            DateTime admissionDate = episode.AdmissionDate;

            // Get CDASetId by Filler OrderNumber
            CdaSetNumberDl cdaSetNumberDl = new CdaSetNumberDl(this.UserContext);
            CdaSetNumber cdaSetNumber = cdaSetNumberDl.GetByEpisodeAndDocumentType(episode.EpisodeId.Value, (ListSingleton.Instance.AllDocumentTypes.Single(a => a.Code == docType).DocumentTypeId.Value));
            string sourceSystemSetId = string.Format("{0}.{1}^{2}", HIPS.CommonSchemas.Cda.Constants.Oid.DOCUMENT_SET_ID, hospital.HpiO, cdaSetNumber.CdaSetNumberId.Value);

            // Call Remove Document 
            DocumentRemovalBeforeQueue businessLogic = new DocumentRemovalBeforeQueue(patientIdentifier, admissionDate, this.UserContext, sourceSystemSetId);
            businessLogic.SetHL7MessageLogId(messageLog.HL7MessageLogId.Value);
            HipsResponse response = businessLogic.LoadPatientEpisode(); // Loads all of the details required for the Remove, Validates IHI and checks for outstanding alerts
            if (response.Status != HipsResponseIndicator.OK)
            {
                removeResponse.Status = HipsResponseIndicator.InvalidEpisode;
                removeResponse.HipsErrorMessage = "Patient Episode could not be found. The document cannot be removed from the PCEHR.";
                return removeResponse;
            }

            removeResponse = businessLogic.Remove(RemovalReason.Withdrawn, null);
            return removeResponse;

        }

        /// <summary>
        /// Parses the HL7Message and adds it to the MessageLog
        /// </summary>
        /// <param name="hl7Message"></param>
        /// <returns></returns>
        private HL7Message ParseAndLogHL7Message(string hl7Message, out HL7MessageLog messageLog)
        {
            HL7Message message = HL7Message.Parse(hl7Message);
            messageLog = MessageLoader.PopulateHL7MessageLog(message, hl7Message, this.UserContext);
            return message;
        }

        #endregion Private methods
    }
}
