﻿using System;
using System.Text.RegularExpressions;

namespace HIPS.CommonBusinessLogic.Utility
{

    /// <summary>
    /// Regular expression utility methods.
    /// </summary>
    internal class RegExUtility
    {

        /// <summary>
        /// Extracts the value of a specified regular expression match group from the provided string.
        /// </summary>
        /// <param name="input">Input string to apply regular expression to.</param>
        /// <param name="pattern">Regular expression to apply to the input string.</param>
        /// <param name="groupName">Name of the regular expression match group.</param>
        /// <returns>Value of the regular expression match group if a match was found, empty string otherwise.</returns>
        internal static string ExtractRegExGroupValue(string input, string pattern, string groupName)
        {
            return RegExUtility.ExtractRegExGroupValue(input, pattern, groupName, RegexOptions.None);
        }

        /// <summary>
        /// Extracts the value of a specified regular expression match group from the provided string.
        /// </summary>
        /// <param name="input">Input string to apply regular expression to.</param>
        /// <param name="pattern">Regular expression to apply to the input string.</param>
        /// <param name="groupName">Name of the regular expression match group.</param>
        /// <param name="regExOptions">RegexOptions to be applied to the regular expression.</param>
        /// <returns>Value of the regular expression match group if a match was found, empty string otherwise.</returns>
        internal static string ExtractRegExGroupValue(string input, string pattern, string groupName, RegexOptions regExOptions)
        {
            string result = string.Empty;

            Match m = Regex.Match(input, pattern, regExOptions);
            if (m.Success)
            {
                if (m.Groups[groupName] != null)
                {
                    result = m.Groups[groupName].Value;
                }
            }

            return result;
        }

        /// <summary>
        /// Returns a Boolean value indicating whether the provided input matches the provided regular expression pattern.
        /// </summary>
        /// <param name="input">Input string to apply regular expression to.</param>
        /// <param name="pattern">Regular expression to apply to the input string.</param>
        /// <returns>A Boolean value indicating whether the provided input matches the provided regular expression pattern.</returns>
        internal static bool IsMatch(string input, string pattern)
        {
            return Regex.IsMatch(input, pattern);
        }

    }
}
