﻿using System.Collections.Generic;
using System.Linq;
using HIPS.CommonBusinessLogic;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;

using DataSchemas = HIPS.PcehrDataStore.Schemas;

namespace HIPS.PatientBusinessLogic
{
    public class PatientListLogic
    {
        /// <summary>
        /// Finds out whether HIPS has recorded a withdrawal of consent to upload
        /// clinical documents for a specified episode - defined from a retrieved episode id
        /// </summary>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="sourceSystemEpisodeId">The source system's episode identifier to return the consent information</param>
        /// <param name="episode">Out. The episode details matching the provider episode id.</param>
        /// <returns>
        /// Response containing the consent status for this episode and an error indicator
        /// </returns>
        public HipsResponse GetEpisodeDetails(UserDetails user, PatientIdentifierBase patientIdentifier, string sourceSystemEpisodeId, out EpisodePatientExtendedDetails episode)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            episode = null;
            PatientAccess patientAccess = new PatientAccess(user);
            Hospital hospital;
            HipsResponse status = patientAccess.GetHospital(patientIdentifier, out hospital);
            if (status.Status != HipsResponseIndicator.OK)
            {
                response = status;
                return response;
            }

            HospitalPatient hospitalPatient;
            PatientMaster patientMaster;
            // When checking consent, it's OK if the patient has an invalid IHI or an unresolved IHI alert.
            status = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
            if (status.Status != HipsResponseIndicator.OK && status.Status != HipsResponseIndicator.InvalidIhi && status.Status != HipsResponseIndicator.UnresolvedIhiAlert)
            {
                response = status;
                return response;
            }

            //A Specific Episode
            episode = patientAccess.GetEpisodePatientExtendedDetails(sourceSystemEpisodeId, hospitalPatient, patientIdentifier);
            if (episode == null || episode.Mrn == null)
            {
                episode = null;
                response = new HipsResponse(HipsResponseIndicator.InvalidEpisode);
                response.HipsErrorMessage = string.Format(ResponseStrings.InvalidEpisodeforSourceSystemEpisodeId, sourceSystemEpisodeId, hospitalPatient.PatientId, hospital.Description);
            }
            else
            {
                response = new HipsResponse(HipsResponseIndicator.OK);
            }

            return response;
        }

        /// <summary>
        /// Finds out whether HIPS has recorded a withdrawal of consent to upload
        /// clinical documents for a specified episode - defined from a retrieved episode id
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="sourceSystemEpisodeId">The source system's episode identifier to return the consent information</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <returns>
        /// Response containing the consent status for this episode and an error indicator
        /// </returns>
        public HipsResponse GetPatientDisclosureDetails(UserDetails user, PatientIdentifierBase patientIdentifier, out PatientDisclosureDetails patientDisclosure)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            patientDisclosure = null;
            PatientAccess patientAccess = new PatientAccess(user);
            Hospital hospital;
            HipsResponse status = patientAccess.GetHospital(patientIdentifier, out hospital);
            if (status.Status != HipsResponseIndicator.OK)
            {
                response = status;
                return response;
            }

            HospitalPatient hospitalPatient;
            PatientMaster patientMaster;
            // When checking consent, it's OK if the patient has an invalid IHI or an unresolved IHI alert.
            status = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
            if (status.Status != HipsResponseIndicator.OK && status.Status != HipsResponseIndicator.InvalidIhi && status.Status != HipsResponseIndicator.UnresolvedIhiAlert)
            {
                response = status;
                return response;
            }

            //A Specific Patient Details
            HospitalPatientDl patient = new HospitalPatientDl(user);
            patientDisclosure = patient.SinglePatientInHospital(hospitalPatient, patientIdentifier.HospitalCodeSystem, hospital.HealthProviderOrganisationNetworkId);
            if (patientDisclosure == null)
            {
                response = new HipsResponse(HipsResponseIndicator.InvalidPatient);
            }
            else
            {
                response = new HipsResponse(HipsResponseIndicator.OK);
            }

            return response;
        }

        /// <summary>
        /// Lists admitted or recently discharged patients in hospital.
        /// </summary>
        /// <param name="hospitalCodeSystem">The hospital code system.</param>
        /// <param name="hospitalCode">The hospital code.</param>
        /// <param name="withIhi">Optional filter on having an active verified IHI, or no IHI.</param>
        /// <param name="withPcehr">Optional filter on having a PCEHR found, or no PCEHR found</param>
        /// <param name="excludeMedicareExclusions">Whether to exclude patients with special Medicare numbers.</param>
        /// <param name="daysDischarged">Number of days after discharge that patients are included in the list. Set to 0 to list only currently admitted patients.</param>
        /// <param name="daysAfterService">Number of days after service date that non-inpatients are included in the list. Set to 0 to exclude non-inpatients.</param>
        /// <param name="user">The user who is requesting the information.</param>
        /// <param name="data">Out. List of patients in hospital.</returns>
        /// <returns>Success (OK) or reason for failure.</returns>
        public HipsResponse ListAdmittedPatients(string hospitalCodeSystem, string hospitalCode, bool? withIhi, bool? withPcehr, bool excludeMedicareExclusions, int daysDischarged, int daysAfterService, UserDetails user, out List<DataSchemas.PatientInHospital> data)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            data = null;

            PatientListDl dataAccess = new PatientListDl(user);
            int? hospitalId = GetHospitalId(hospitalCodeSystem, hospitalCode, response);
            if (response.Status == HipsResponseIndicator.InvalidHospital)
            {
                return response;
            }

            response = dataAccess.ListPatientsInHospital(hospitalCodeSystem, hospitalId, withIhi, withPcehr, excludeMedicareExclusions, daysDischarged, daysAfterService, out data);
            return response;
        }

        /// <summary>
        /// Lists Episodes for patients in hospital
        /// </summary>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="daysDischarged">Number of days after discharge that patients are included in the list. Set to 0 to list only currently admitted patients.</param>
        /// <param name="data">Out. List of episodes for patient in hospital.</returns>
        /// <param name="includeDocuments">if set to <c>true</c> [include documents].</param>
        /// <param name="documentTypeCode">The document type code.</param>
        /// <returns>Success (OK) or reason for failure.</returns>
        public HipsResponse ListPatientEpisodesInHospital(UserDetails user, PatientIdentifierBase patientIdentifier, int daysDischarged, out List<PatientInHospitalEpisode> data, bool includeDocuments = false, string documentTypeCode = null)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            data = null;

            PatientAccess patientAccess = new PatientAccess(user);
            Hospital hospital;
            HipsResponse status = patientAccess.GetHospital(patientIdentifier, out hospital);
            if (status.Status != HipsResponseIndicator.OK)
            {
                response = status;
                return response;
            }

            patientAccess = new PatientAccess(user);
            HospitalPatient hospitalPatient;
            PatientMaster patientMaster;
            status = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
            if (status.Status != HipsResponseIndicator.OK && status.Status != HipsResponseIndicator.InvalidIhi)
            {
                response = status;
                return response;
            }

            EpisodeListDl dataAccess = new EpisodeListDl(user);
            response = dataAccess.ListPatientInHospitalEpisodes((int)hospitalPatient.PatientId, daysDischarged, out data);

            if (includeDocuments && !string.IsNullOrEmpty(documentTypeCode))
            {
                List<LocalClinicalDocumentMetaData> metadata;
                var documentType = ListSingleton.Instance.AllDocumentTypes.Single(dt => dt.Code == documentTypeCode);
                var documentAccess = new ClinicalDocumentDl(user);

                status = documentAccess.ListLocalUploadedDocuments((int)hospitalPatient.PatientId, out metadata, documentTypeCode: documentType.Code);
                if (status.Status != HipsResponseIndicator.OK)
                {
                    response = status;
                    return response;
                }

                foreach (var episode in data)
                {
                    episode.LocalClinicalDocumentMetaData.AddRange(metadata.Where(m => m.EpisodeId == episode.EpisodeId));
                }
            }

            return response;
        }

        /// <summary>
        /// Lists admitted or recently discharged patients in hospital with an active IHI, with or without a PCEHR.
        /// </summary>
        /// <param name="user">The user who is requesting the information.</param>
        /// <param name="hospitalCodeSystem">The hospital code system.</param>
        /// <param name="hospitalCode">The hospital code.</param>
        /// <param name="withPcehr">True to list patients with a PCEHR, false to list patients without a PCEHR. Null to list all patients.</param>
        /// <param name="daysDischarged">Number of days after discharge that patients are included in the list. Set to 0 to list only currently admitted patients.</param>
        /// <param name="data">Out. The list of patients.</param>
        /// <returns>Success (OK) or reason for failure.</returns>
        public HipsResponse ListPatientsInHospital(UserDetails user, string hospitalCodeSystem, string hospitalCode, bool? withPcehr, int daysDischarged, out List<DataSchemas.PatientInHospital> data)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            data = null;

            PatientListDl dataAccess = new PatientListDl(user);
            int? hospitalId = GetHospitalId(hospitalCodeSystem, hospitalCode, response);
            if (response.Status == HipsResponseIndicator.InvalidHospital)
            {
                return response;
            }

            response = dataAccess.ListPatientsInHospital(hospitalCodeSystem, hospitalId, true, withPcehr, false, daysDischarged, 0, out data);

            return response;
        }

        /// <summary>
        /// Lists admitted or recently discharged patients in hospital without an IHI. Patients whose Medicare number is in the list of exclusions will be excluded.
        /// </summary>
        /// <param name="user">The user who is requesting the information.</param>
        /// <param name="hospitalCodeSystem">The hospital code system.</param>
        /// <param name="hospitalCode">The hospital code.</param>
        /// <param name="daysDischarged">Number of days after discharge that patients are included in the list. Set to 0 to list only currently admitted patients.</param>
        /// <param name="data">Out. The list of patients.</param>
        /// <returns>Success (OK) or reason for failure.</returns>
        public HipsResponse ListPatientsWithoutIhi(UserDetails user, string hospitalCodeSystem, string hospitalCode, int daysDischarged, out List<DataSchemas.PatientInHospital> data)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            data = null;

            int? hospitalId = GetHospitalId(hospitalCodeSystem, hospitalCode, response);
            if (response.Status == HipsResponseIndicator.InvalidHospital)
            {
                return response;
            }

            PatientListDl dataAccess = new PatientListDl(user);
            response = dataAccess.ListPatientsInHospital(hospitalCodeSystem, hospitalId, false, null, true, daysDischarged, 0, out data);
            return response;
        }

        /// <summary>
        /// Resolves the hospital code (if supplied) to a hospital ID. If no hospital code is supplied, returns null.
        /// If an invalid hospital code is supplied, returns null and sets the response status to InvalidHospital.
        /// </summary>
        /// <param name="hospitalCodeSystem">The hospital code system.</param>
        /// <param name="hospitalCode">The hospital code.</param>
        /// <param name="response">The HIPS response.</param>
        /// <returns>The hospital ID.</returns>
        private static int? GetHospitalId(string hospitalCodeSystem, string hospitalCode, HipsResponse response)
        {
            int? hospitalId;
            if (string.IsNullOrEmpty(hospitalCode))
            {
                hospitalId = null;
            }
            else
            {
                DataSchemas.Hospital hospital = HospitalSingleton.Value.Find(hospitalCode, hospitalCodeSystem);
                if (hospital == null)
                {
                    hospitalId = null;
                    response.Status = HipsResponseIndicator.InvalidHospital;
                    response.HipsErrorMessage = ResponseStrings.HospitalNotFound;
                }
                else
                {
                    hospitalId = hospital.HospitalId.Value;
                }
            }
            return hospitalId;
        }
    }
}