﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.DataAccess
{
    /// <summary>
    /// This class allows access to the PatientMasterIhi table
    /// </summary>
    public class PatientMasterIhiDl : BaseDl
    {
        #region Constructors

        public PatientMasterIhiDl(UserDetails user)
            : base(user)
        {
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Gets the specified patient master IHI record by patient master ID.
        /// </summary>
        /// <param name="patientMasterID">The patient master ID.</param>
        /// <param name="newItem">The newItem.</param>
        /// <returns>Whether or not the get was successful</returns>
        public bool Get(int patientMasterID, int healthProviderOrganisationNetworkId, out PatientMasterIhi item)
        {
            item = new PatientMasterIhi();
            bool result = false;
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.PatientMasterIhiGet"))
                {
                    command.Parameters.Add(new SqlParameter("@PatientMasterId", patientMasterID));
                    command.Parameters.Add(new SqlParameter("@HealthProviderOrganisationNetworkId", healthProviderOrganisationNetworkId));
                    result = PopulateBusinessObject<PatientMasterIhi>(command.ExecuteReader(), item);
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessagePatientMasterIhiGet, ex, User, LogMessage.HIPS_MESSAGE_046);
            }
            return result;
        }

        /// <summary>
        /// Gets a list of patient master IHI records by IHI status.
        /// </summary>
        /// <param name="ihiStatus">The IHI status.</param>
        /// <param name="transaction">An existing SQL transaction context in which to execute this query (optional).</param>
        /// <returns>List of patient master IHI records</returns>
        public List<PatientMasterIhi> GetAll(IhiStatus ihiStatus, int healthProviderOrganisationNetworkId, SqlTransaction transaction = null)
        {
            List<PatientMasterIhi> result = new List<PatientMasterIhi>();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.PatientMasterIhiGet", transaction))
                {
                    AddIntParameter("@IhiStatusId", (int)ihiStatus, command);
                    AddIntParameter("@HealthProviderOrganisationNetworkId", healthProviderOrganisationNetworkId, command);
                    result = GetPopulatedBusinessList<PatientMasterIhi>(command.ExecuteReader());
                    if (transaction == null)
                    {
                        command.Connection.Close();
                    }
                }
            }
            catch (Exception ex)
            {
                string error = string.Format(ConstantsResource.ErrorMessagePatientMasterIhiGetAll, ihiStatus);
                EventLogger.WriteLog(error, ex, User, LogMessage.HIPS_MESSAGE_047);
            }
            return result;
        }

        /// <summary>
        /// Inserts or updates the specified patient master Ihi link.
        /// </summary>
        /// <param name="newItem">The new IHI to be assigned to the patient</param>
        /// <param name="transaction">The existing SQL Transaction in which to perform the insert or update</param>
        /// <returns></returns>
        public bool Update(PatientMasterIhi newItem, SqlTransaction transaction = null)
        {
            bool updateResult;
            PatientMasterIhi oldItem;

            if (!Get(newItem.PatientMasterId.Value, newItem.HealthProviderOrganisationNetworkId.Value, out oldItem))
            {
                oldItem = null;
            }

            RemoveUnusedIhi(newItem, oldItem, transaction);
            bool updateHistory = (oldItem != null && (oldItem.Ihi != newItem.Ihi || oldItem.IhiRecordStatusId != newItem.IhiRecordStatusId || oldItem.IhiStatusId != newItem.IhiStatusId));

            if (oldItem == null)
            {
                using (SqlCommand command = GetSqlCommand("[hips].[PatientMasterIhiInsert]", transaction))
                {
                    updateResult = base.Insert<PatientMasterIhi>(newItem, command);
                }
            }
            else
            {
                bool useExistingTransaction = transaction != null;

                using (SqlCommand command = GetSqlCommand("[hips].[PatientMasterIhiUpdate]", transaction))
                {
                    try
                    {
                        if (!useExistingTransaction)
                        {
                            transaction = command.Connection.BeginTransaction();
                            command.Transaction = transaction;
                        }
                        newItem.DateModified = oldItem.DateModified;
                        updateResult = base.Update<PatientMasterIhi>(newItem, command);
                        if (updateResult)
                        {
                            updateResult = InsertHistory(newItem.PatientMasterId.Value, oldItem.Ihi, oldItem.IhiRecordStatusId, oldItem.IhiStatusId, updateResult, updateHistory, newItem.HealthProviderOrganisationNetworkId.GetValueOrDefault(), transaction);
                        }
                        if (updateResult)
                        {
                            if (!useExistingTransaction)
                            {
                                transaction.Commit();
                                command.Connection.Close();
                            }
                        }
                        else
                        {
                            transaction.Rollback();
                            command.Connection.Close();
                        }
                        return updateResult;
                    }
                    catch (Exception ex)
                    {
                        if (transaction != null)
                        {
                            transaction.Rollback();
                        }
                        command.Connection.Close();
                        EventLogger.WriteLog(ConstantsResource.ErrorPatientMasterInsert, ex, User, LogMessage.HIPS_MESSAGE_122);
                        updateResult = false;
                    }
                }
            }
            return updateResult;
        }

        /// <summary>
        /// Gets the number of other patients at the same hospital whose demographic information matches the information used to obtain this IHI.
        /// If a hospital ID is given, includes only patients at the given hospital.
        /// </summary>
        /// <param name="newItem">The new item.</param>
        /// <param name="transaction">The current SQL transaction.</param>
        /// <returns>The number of matching patients</returns>
        private int GetPossibleDuplicatePatients(PatientMasterIhi newItem, SqlTransaction transaction)
        {
            if (!string.IsNullOrEmpty(newItem.Ihi))
            {
                try
                {
                    using (SqlCommand command = GetSqlCommand("hips.IhiDuplicatePatientCount", transaction))
                    {
                        AddIntParameter(@"PatientMasterId", newItem.PatientMasterId, command);
                        AddStringParameter("@LastName", newItem.RegisteredFamilyName, command);
                        AddStringParameter("@FirstNames", newItem.RegisteredGivenName, command);
                        AddStringParameter("@MedicareNumber", newItem.MedicareNumber, command);
                        AddStringParameter("@MedicareIrn", newItem.MedicareNumberSequence, command);
                        AddStringParameter("@DvaNumber", newItem.DvaNumber, command);
                        AddDateParameter("@DateOfBirth", newItem.DateOfBirth, command);
                        return int.Parse(command.ExecuteScalar().ToString());
                    }
                }
                catch (Exception ex)
                {
                    EventLogger.WriteLog(string.Format(ConstantsResource.ErrorDuplicatePatientCheck, newItem.Ihi), ex, User, LogMessage.HIPS_MESSAGE_125);
                    throw new Exception(string.Format(ConstantsResource.ErrorDuplicatePatientCheck, newItem.Ihi), ex);
                }
            }
            return 0;
        }

        /// <summary>
        /// Gets the number of other patients at the same hospital that already have been assigned the given IHI.
        /// </summary>
        /// <param name="newItem">The new item.</param>
        /// <param name="transaction">The current SQL transaction.</param>
        /// <returns>The number of matching patients</returns>
        private int GetTimesIhiUsed(PatientMasterIhi newItem, SqlTransaction transaction)
        {
            if (!string.IsNullOrEmpty(newItem.Ihi))
            {
                try
                {
                    using (SqlCommand command = GetSqlCommand("hips.IhiUsageCount", transaction))
                    {
                        AddIntParameter("@PatientMasterId", newItem.PatientMasterId, command);
                        AddStringParameter("@Ihi", newItem.Ihi, command);
                        AddIntParameter("@HealthProviderOrganisationNetworkId", newItem.HealthProviderOrganisationNetworkId, command);
                        int result = int.Parse(command.ExecuteScalar().ToString());

                        //EventLogger.WriteLog("Found " + result.ToString(),null);
                        return result;
                    }
                }
                catch (Exception ex)
                {
                    EventLogger.WriteLog(string.Format(ConstantsResource.ErrorPatientMasterIhiCount, newItem.Ihi), ex, User, LogMessage.HIPS_MESSAGE_123);
                    throw new Exception(string.Format(ConstantsResource.ErrorPatientMasterIhiCount, newItem.Ihi), ex);
                }
            }
            EventLogger.WriteLog(ConstantsResource.InfoDuplicateIhiCheckNotPerformed, null, User, LogMessage.HIPS_MESSAGE_124);
            return 0;
        }

        /// <summary>
        /// Inserts the historic IHI record.
        /// </summary>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <param name="ihi">The ihi.</param>
        private bool InsertHistory(int patientMasterId, string ihi, int ihiRecordStatusId, int ihiStatusId, bool updateResult, bool updateHistory, int healthProviderOrganisationNetworkId, SqlTransaction transaction)
        {
            if (updateResult && updateHistory)
            {
                PatientMasterIhiHistory result = new PatientMasterIhiHistory();
                using (SqlCommand command = GetSqlCommand("hips.PatientMasterIhiHistoryInsert", transaction))
                {
                    result.PatientMasterID = patientMasterId;
                    result.Ihi = ihi;
                    result.IhiRecordStatusId = ihiRecordStatusId;
                    result.IhiStatusId = ihiStatusId;
                    result.HealthProviderOrganisationNetworkId = healthProviderOrganisationNetworkId;
                    updateResult = base.Insert<PatientMasterIhiHistory>(result, command);
                }
            }
            return updateResult;
        }

        /// <summary>
        /// Removes the unused ihi and Log any errors out to the error logger.
        /// </summary>
        /// <param name="newItem">The new item.</param>
        /// <param name="oldItem">The old item (or null if no IHI has been stored for this patient).</param>
        /// <param name="transaction">The current SQL transaction.</param>
        private void RemoveUnusedIhi(PatientMasterIhi newItem, PatientMasterIhi oldItem, SqlTransaction transaction)
        {
            if (string.IsNullOrEmpty(newItem.Ihi))
            {
                // No IHI was found.
                return;
            }

            // Setting the status to Resolved now happens outside this method,
            // only when the message from HI Service indicates that it was
            // resolved.
            // If the IHI has changed and the status isn't Resolved, this is a
            // demographic mismatch condition. We want to keep the old IHI and
            // mark it as DemographicMismatch, rather than treating it as a
            // resolution from the old to the new.
            if (oldItem != null && newItem.Ihi != oldItem.Ihi && !string.IsNullOrEmpty(oldItem.Ihi))
            {
                if (newItem.IhiStatusId != (int)IhiStatus.Resolved)
                {
                    // Keep the old IHI and set its status to demographic mismatch.
                    newItem.Ihi = oldItem.Ihi;
                    newItem.IhiStatusId = (int)IhiStatus.DemographicMismatch;
                }
            }

            IhiLookupAlertDl dataAccess = new IhiLookupAlertDl();
            int ihiUseCounter = GetTimesIhiUsed(newItem, transaction);
            if (ihiUseCounter > 0)
            {
                //EventLogger.WriteLog("OK Setting duplicate flag", null);
                newItem.IhiStatusId = (int)IhiStatus.DuplicateIhi;
            }

            int possibleDuplicatePatients = GetPossibleDuplicatePatients(newItem, transaction);
            if (possibleDuplicatePatients > 0)
            {
                newItem.IhiStatusId = (int)IhiStatus.DuplicatePatient;
            }

            string message = string.Empty;

            // IHI Status's where we cannot use the IHI, so we need to return it as a null
            // However Some IHI's we cannot use need to still be saved in the database (i.e. duplicates)
            switch (newItem.IhiStatusId)
            {
                // Duplicate Patient Alert - we can store the IHI but it cannot be used.
                case (int)IhiStatus.DuplicatePatient:
                    dataAccess.Insert(new IhiLookupAlert(newItem.PatientMasterId.Value, ConstantsResource.AlertPossibleDuplicatePatient, newItem.Request, newItem.Response), transaction);
                    newItem.UsableIhi = null;
                    break;

                // IHIs we can still store in the database
                case (int)IhiStatus.DuplicateIhi:
                case (int)IhiStatus.Expired:
                case (int)IhiStatus.Retired:
                    message = string.Format(ConstantsResource.AlertIhiStatusChanged, ((IhiStatus)newItem.IhiStatusId).ToString());
                    dataAccess.Insert(new IhiLookupAlert(newItem.PatientMasterId.Value, message, newItem.Request, newItem.Response), transaction);
                    newItem.UsableIhi = newItem.Ihi;
                    break;

                // IHI's we cannot use or store in the database
                case (int)IhiStatus.Unknown:
                    message = string.Format(ConstantsResource.AlertIhiStatusChanged, ((IhiStatus)newItem.IhiStatusId).ToString());
                    dataAccess.Insert(new IhiLookupAlert(newItem.PatientMasterId.Value, message, newItem.Request, newItem.Response), transaction);
                    newItem.UsableIhi = null;
                    break;

                // Where a IHI Status is resolved the corrected IHI is returned, this needs to now be marked
                // active and the previous status set to resolved for the history record
                case (int)IhiStatus.Resolved:
                    dataAccess.Insert(new IhiLookupAlert(newItem.PatientMasterId.Value, ConstantsResource.AlertIhiStatusNowResolved, newItem.Request, newItem.Response), transaction);
                    if (oldItem != null)
                    {
                        oldItem.IhiStatusId = (int)IhiStatus.Resolved;
                    }

                    if (newItem.IhiRecordStatusId == (int)IhiRecordStatus.Unknown)
                    {
                        // If the new IHI cannot be validated with the existing patient demographics,
                        // the software shall not store that new IHI.
                        // Restore the old IHI, or if there was none, store NULL.
                        newItem.IhiStatusId = (int)IhiStatus.DemographicMismatch;
                        newItem.Ihi = oldItem == null ? null : oldItem.Ihi;
                    }
                    else
                    {
                        // If the IHI number can be validated using local patient demographic data,
                        // the new IHI number, IHI status and IHI record status shall be stored in
                        // the patient record.
                        newItem.IhiStatusId = (int)IhiStatus.Active;
                    }

                    newItem.UsableIhi = newItem.Ihi;
                    break;

                // All other IHI Status's have an IHI that can be used, so it is returned
                case (int)IhiStatus.Active:
                    newItem.UsableIhi = newItem.Ihi;
                    break;

                case (int)IhiStatus.Deceased:
                    dataAccess.Insert(new IhiLookupAlert(newItem.PatientMasterId.Value, ConstantsResource.AlertIhiStatusNowDeceased, newItem.Request, newItem.Response), transaction);
                    newItem.UsableIhi = newItem.Ihi;
                    break;
            }

            switch (newItem.IhiRecordStatusId)
            {
                // We are not catering for Provisional or unverified IHI's
                // so cannot use or store them
                case (int)IhiRecordStatus.Provisional:
                case (int)IhiRecordStatus.Unverified:
                    message = string.Format(ConstantsResource.AlertIhiRecordStatusChanged, ((IhiRecordStatus)newItem.IhiRecordStatusId).ToString());
                    dataAccess.Insert(new IhiLookupAlert(newItem.PatientMasterId.Value, message, newItem.Request, newItem.Response), transaction);
                    newItem.Ihi = null;
                    newItem.UsableIhi = null;
                    break;

                case (int)IhiRecordStatus.Unknown:
                    message = string.Format(ConstantsResource.AlertIhiRecordStatusChanged, ((IhiRecordStatus)newItem.IhiRecordStatusId).ToString());
                    dataAccess.Insert(new IhiLookupAlert(newItem.PatientMasterId.Value, message, newItem.Request, newItem.Response), transaction);
                    newItem.UsableIhi = null;
                    break;

                // Only using Verified IHI's
                case (int)IhiRecordStatus.Verified:
                    newItem.UsableIhi = newItem.Ihi;
                    break;
            }
        }

        #endregion Methods
    }
}