﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Xml;
using HIPS.CommonBusinessLogic;
using HIPS.PcehrDataStore.Schemas;

namespace HIPS.PcehrHiBusinessLogic.Cda
{
    public class CdaValidation
    {
        private const string HI_OID_PREFIX = "1.2.36.1.2001.1003.0.";
        private const string SIXTEEN_DIGIT_REGEX = "^\\d{16}$";
        private readonly object syncRoot = new object();
        private readonly HiValidationParameters Ihi = new HiValidationParameters() { FirstSixDigits = "800360", Description = "IHI", Type = HiType.Ihi };
        private readonly HiValidationParameters HpiI = new HiValidationParameters() { FirstSixDigits = "800361", Description = "HPI-I", Type = HiType.HpiI };
        private readonly HiValidationParameters HpiO = new HiValidationParameters() { FirstSixDigits = "800362", Description = "HPI-O", Type = HiType.HpiO };
        private XmlDocument xmlDoc;
        private XmlNamespaceManager xnm;
        private DocumentType documentType;
        private Hospital hospital;
        private PatientMaster patient;

        /// <summary>
        /// Validates the Facility, Custodian and Author's HPI-O, the Author's HPI-I,
        /// the Patient's IHI and Date of Birth, and the Creation Time in the CDA document.
        /// </summary>
        /// <param name="xmlDoc">The CDA document</param>
        /// <param name="xnm">An XML namespace manager with "x" (HL7v3) and "ext" (Australian CDA extensions) entries.</param>
        /// <param name="documentType">The document's type.</param>
        /// <param name="hospital">The hospital whose HPI-O should match the HPI-O(s) in the document</param>
        /// <param name="patient">The patient whose IHI should match the IHI in the document</param>
        public void Validate(XmlDocument xmlDoc, XmlNamespaceManager xnm, DocumentType documentType, Hospital hospital, PatientMaster patient)
        {
            lock (syncRoot)
            {
                this.xmlDoc = xmlDoc;
                this.xnm = xnm;
                this.documentType = documentType;
                this.hospital = hospital;
                this.patient = patient;
                ValidateHpiOs();
                ValidateHpiIs();
                ValidateIhi();
                ValidateDateOfBirth();
                ValidateCreationTime();
            }
        }

        /// <summary>
        /// Validates the creation time in the CDA document.
        /// </summary>
        private void ValidateCreationTime()
        {
            string xpath = "/x:ClinicalDocument/x:effectiveTime";
            DateTime date = ValidateTimestamp(xpath, "Creation Time");
        }

        /// <summary>
        /// Validates the patient's date of birth in the CDA document.
        /// </summary>
        private void ValidateDateOfBirth()
        {
            string xpath = "/x:ClinicalDocument/x:recordTarget/x:patientRole/x:patient/x:birthTime";

            DateTime date = ValidateTimestamp(xpath, "Patient DOB");
            if (date.Date != patient.DateOfBirth.Date)
            {
                throw new Exception(string.Format("The patient DOB {0} in the document does not match the DOB {1} in HIPS demographics.",
                    date.ToShortDateString(), patient.DateOfBirth.ToShortDateString()));
            }
        }
        
        /// <summary>
        /// Validates a timestamp in the CDA document.
        /// </summary>
        /// <param name="xpath"></param>
        /// <param name="itemName"></param>
        /// <returns></returns>
        private DateTime ValidateTimestamp(string xpath, string itemName)
        {
            XmlNode node = xmlDoc.SelectSingleNode(xpath, xnm);
            if (node == null || node.Attributes["value"] == null)
            {
                throw new Exception(string.Format("{0} was not found in the document at the path {1}.", itemName, xpath));
            }
            string value = node.Attributes["value"].Value;
            DateTime? date = null;
            
            // Attempt to parse the timestamp.
            try
            {
                date = HL7DateTime.Parse(value);
            }
            catch (Exception)
            {
                throw new Exception(string.Format("The {0} {1} in the document does not have the expected format.", itemName, value));
            }

            if (!date.HasValue)
            {
                throw new Exception(string.Format("The {0} in the document was empty", itemName));
            }
            return date.Value;
        }

        /// <summary>
        /// Validates the entity identifier for the patient's IHI.
        /// </summary>
        private void ValidateIhi()
        {
            string xpath = "/x:ClinicalDocument/x:recordTarget/x:patientRole/x:patient/ext:asEntityIdentifier/ext:id[@assigningAuthorityName='IHI']";
            ValidateHealthcareIdentifierNode(Ihi, "Patient", xpath);            
        }

        /// <summary>
        /// Performs additional validation of HPI-O in a CDA document to meet NPDR NOC requirements.
        /// All types of document should contain the HPI-O in the Custodian section (custodian/assignedCustodian/representedCustodianOrganization).
        /// Some types of document also contain the HPI-O in the Facility section (location/healthCareFacility/serviceProviderOrganisation)
        /// while other types contain the HPI-O in the Author section (author/assignedAuthor/assignedPerson/asEmployment/employerOrganisation).
        /// The assumption made here is that the HPI-O in all these locations should be the same and should also be the same as the HPI-O configured
        /// for the hospital in the HIPS database.
        /// </summary>
        /// <param name="xmlDoc">XML Document</param>
        /// <param name="xnm">XML Namespace Manager</param>
        private void ValidateHpiOs()
        {
            string[] documentTypesWithFacilityHpio = 
            { 
                DocumentTypeCodes.DischargeSummary, DocumentTypeCodes.PcehrPrescriptionRecord, DocumentTypeCodes.PcehrDispenseRecord 
            };
            string[] documentTypesWithAuthorHpio = 
            {
                DocumentTypeCodes.DischargeSummary, DocumentTypeCodes.EventSummary, DocumentTypeCodes.SpecialistLetter, DocumentTypeCodes.SharedHealthSummary 
            };
            string documentTypeCode = documentType.Code;

            ValidateHealthcareIdentifierNode(HpiO, "Custodian", 
                "/x:ClinicalDocument/x:custodian/x:assignedCustodian/x:representedCustodianOrganization/ext:asEntityIdentifier/ext:id[@assigningAuthorityName='HPI-O']");

            if (documentTypesWithFacilityHpio.Contains(documentTypeCode))
            {
                ValidateHealthcareIdentifierNode(HpiO, "Facility", 
                    "/x:ClinicalDocument/x:componentOf/x:encompassingEncounter/x:location/x:healthCareFacility/x:serviceProviderOrganization/x:asOrganizationPartOf/x:wholeOrganization/ext:asEntityIdentifier/ext:id[@assigningAuthorityName='HPI-O']");
            }

            if (documentTypesWithAuthorHpio.Contains(documentTypeCode))
            {
                ValidateHealthcareIdentifierNode(HpiO, "Author", 
                    "/x:ClinicalDocument/x:author/x:assignedAuthor/x:assignedPerson/ext:asEmployment/ext:employerOrganization/x:asOrganizationPartOf/x:wholeOrganization/ext:asEntityIdentifier/ext:id[@assigningAuthorityName='HPI-O']");
            }
        }

        /// <summary>
        /// Performs validation of HPI-I in a CDA document to meet NPDR NOC requirements.
        /// All types of document should contain the HPI-I in the Author section (author/assignedAuthor/assignedPerson),
        /// but it is not treated as mandatory here because of the HPI-I relaxation.
        /// </summary>
        /// <param name="xmlDoc">XML Document</param>
        /// <param name="xnm">XML Namespace Manager</param>
        private void ValidateHpiIs()
        {
            ValidateHealthcareIdentifierNode(HpiI, "Author",
                "/x:ClinicalDocument/x:author/x:assignedAuthor/x:assignedPerson/ext:asEntityIdentifier/ext:id[@assigningAuthorityName='HPI-I']",
                isMandatory: false);
        }

        /// <summary>
        /// Validates an "ext:id" node that should contain a national healthcare identifier (IHI, HPI-I or HPI-O).
        /// </summary>
        /// <param name="type">The parameters for validation of the particular type of healthcare identifier</param>
        /// <param name="subject">A user-friendly description of which item within the CDA document is identified by this identifier, e.g. "Facility" or "Custodian" or "Author".</param>
        /// <param name="xpath">An XPath expression that matches the "ext:id" node for the healthcare identifier</param>
        /// <param name="xmlDoc">The CDA document</param>
        /// <param name="xnm">The XML namespace manager</param>
        /// <param name="isMandatory">Whether this healthcare identifier is mandatory</param>
        /// <returns>The value of the healthcare identifier, stripped of the OID prefix</returns>
        private string ValidateHealthcareIdentifierNode(HiValidationParameters type, string subject, string xpath, bool isMandatory = true)
        {
            XmlNode node = xmlDoc.SelectSingleNode(xpath, xnm);
            if (node == null && !isMandatory)
            {
                // It's not found, but it's not mandatory so ignore that.
                return null;
            }
            if (node == null || node.Attributes["root"] == null)
            {
                throw new Exception(string.Format("{0} {1} was not found in the CDA document at the path {2}.", subject, type.Description, xpath));
            }
            string value = node.Attributes["root"].Value;
            if (!value.StartsWith(HI_OID_PREFIX))
            {
                throw new Exception(string.Format("{0} {1} does not start with {2} and so is not a valid {1}.", subject, type.Description, HI_OID_PREFIX));
            }
            value = value.Substring(HI_OID_PREFIX.Length);
            if (!Regex.IsMatch(value, SIXTEEN_DIGIT_REGEX))
            {
                throw new Exception(string.Format("{0} {1} {2} does not consist of 16 digits and so is not a valid {1}.", subject, type.Description, value));
            }
            if (!value.StartsWith(type.FirstSixDigits))
            {
                throw new Exception(string.Format("{0} {1} {2} does not start with {3} and so is not a valid {1}.", subject, type.Description, value, type.FirstSixDigits));
            }
            if (type.Type == HiType.HpiO && value != hospital.HpiO)
            {
                throw new Exception(string.Format("{0} {1} {2} in document does not match {1} {3} in HIPS configuration for hospital {4}",
                    subject, type.Description, value, hospital.HpiO, hospital.Description));
            }
            return value;
        }
    }
}
