﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Security.Cryptography;

using AutoMapper;
using DatabaseSchemas = HIPS.PcehrDataStore.Schemas;
using ModelSchemas = HIPS.CommonSchemas;
using NehtaSchemas = Nehta.VendorLibrary.CDA;

namespace HIPS.PcehrHiBusinessLogic.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for the Attachment entity.
    /// </summary>
    internal class AttachmentProfile : Profile
    {
        #region Methods

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            // Model --> NEHTA
            this.CreateMap<ModelSchemas.Cda.CdaAttachment, NehtaSchemas.SCSModel.Common.Entities.ExternalData>()
                .ForMember(dest => dest.Caption, opt => opt.MapFrom(src => src.Caption))
                .ForMember(dest => dest.ExternalDataMediaType, opt => opt.ResolveUsing((ModelSchemas.Cda.CdaAttachment src) =>
                {
                    switch (src.AttachmentType)
                    {
                        case ModelSchemas.Cda.AttachmentType.AttachedDocumentPdf:
                        case ModelSchemas.Cda.AttachmentType.DocumentBodyPdf:
                            return NehtaSchemas.Generator.Enums.MediaType.PDF;

                        case ModelSchemas.Cda.AttachmentType.AttachedImageGif:
                            return NehtaSchemas.Generator.Enums.MediaType.GIF;

                        case ModelSchemas.Cda.AttachmentType.AttachedImageJpeg:
                            return NehtaSchemas.Generator.Enums.MediaType.JPEG;

                        case ModelSchemas.Cda.AttachmentType.AttachedImagePng:
                        case ModelSchemas.Cda.AttachmentType.OrganisationalLogoPng:
                            return NehtaSchemas.Generator.Enums.MediaType.PNG;

                        case ModelSchemas.Cda.AttachmentType.AttachedImageTiff:
                            return NehtaSchemas.Generator.Enums.MediaType.TIFF;
                    }
                    return NehtaSchemas.Generator.Enums.MediaType.Undefined;
                }))
                .ForMember(dest => dest.DigestValue, opt => opt.MapFrom(src => SHA1.Create().ComputeHash(src.Content)))
                .ForMember(dest => dest.DigestCheckAlgorithm, opt => opt.UseValue(NehtaSchemas.Common.Enums.DigestCheckAlgorithm.SHA1))
                .ForMember(dest => dest.Path, opt => opt.MapFrom(src => src.FileName));

            // CommonSchemas.Cda --> PcehrSchemas
            this.CreateMap<ModelSchemas.Cda.CdaAttachment, PcehrSchemas.Attachment>()
                .ForMember(dest => dest.Contents, opt => opt.MapFrom(src => src.Content));

        }

        #endregion
    }
}
