﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.Serialization;
using System.Text;
using System.Threading.Tasks;

using HIPS.ServiceContracts.Common.DTO.PatientIdentifier;
using EntLibVal = Microsoft.Practices.EnterpriseLibrary.Validation;

namespace HIPS.ServiceContracts.Common.DTO.PatientIdentifier
{
    /// <summary>
    /// The Demographic identifier is used for Assisted Registration of a person using manually-entered demographic information.
    /// Services other than Assisted Registration will reject this Patient Identifier.
    /// </summary>
    [KnownType(typeof(PatientIdentifierBase))]
    [DataContract(Namespace = HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1)]
    [EntLibVal.Validators.HasSelfValidation]
    public class Demographic : PatientIdentifierBase
    {
        #region Properties

        /// <summary>
        /// Gets or sets the person's date of birth.
        /// </summary>        
        [DataMember(IsRequired = true)]        
        [EntLibVal.Validators.DateTimeRangeValidator(Constants.SupportedDates.MINIMUM, Constants.SupportedDates.MAXIMUM)]
        public DateTime DateOfBirth { get; set; }

        /// <summary>
        /// Gets or sets the DVA file number on the person's repatriation health card.
        /// </summary>    
        [DataMember(IsRequired = false)]
        public string DvaNumber { get; set; }

        /// <summary>
        /// Gets or sets the family name that appears on the person's Medicare or DVA card.
        /// </summary>        
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.NotNullValidator]
        public string FamilyName { get; set; }

        /// <summary>
        /// Gets or sets the given name that appears on the person's Medicare or DVA card.
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.NotNullValidator]
        public string GivenName { get; set; }

        /// <summary>
        /// Gets or sets the 1-digit IRN for this person on the Medicare card.
        /// </summary>        
        [DataMember(IsRequired = false)]
        public string MedicareIrn { get; set; }

        /// <summary>
        /// Gets or sets the 10-digit card number on the person's Medicare card.
        /// </summary>        
        [DataMember(IsRequired = false)]
        public string MedicareNumber { get; set; }

        /// <summary>
        /// Gets or sets the person's sex.
        /// </summary>        
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.NotNullValidator]
        public Sex Sex { get; set; }
        
        #endregion Properties

        #region Methods

        /// <summary>
        /// Validate the instance is valid.
        /// </summary>
        /// <param name="results">Validation results to add results of validation to.</param>
        [EntLibVal.Validators.SelfValidation]
        protected void Validate(EntLibVal.ValidationResults results)
        {
            if (string.IsNullOrEmpty(this.DvaNumber) && string.IsNullOrEmpty(this.MedicareNumber))
            {
                results.AddResult(new EntLibVal.ValidationResult("At least one of the DvaNumber and MedicareNumber must be specified.", this, "Demographic", string.Empty, null));
            }
        }

        #endregion Methods
    }
}
