using System;
using System.Collections.Concurrent;
using System.Web;

namespace HIPS.Web.Components.Cache
{
    /// <summary>
    /// Provides a memory-based cache leveraging System.Web caching.
    /// </summary>
    public class WebMemoryCacheProvider : CacheProvider
    {
        /// <summary>
        /// Lock reference for WebMemoryCacheProvider used for double-check locking.
        /// This is static as all instances refer to HttpContext.Current
        /// </summary>
        private static readonly ConcurrentDictionary<string, object> WebCacheLock = new ConcurrentDictionary<string, object>();

        /// <summary>
        /// Initialises a new instance of the <see cref="WebMemoryCacheProvider"/> class.
        /// </summary>
        /// <param name="defaultAbsoluteExpirationOffset"></param>
        /// <param name="defaultSlidingExpiration"></param>
        public WebMemoryCacheProvider(TimeSpan? defaultAbsoluteExpirationOffset, TimeSpan? defaultSlidingExpiration = null)
            : base(defaultAbsoluteExpirationOffset, defaultSlidingExpiration)
        {
        }

        /// <summary>
        /// Gets the value identified by the cache key from the cache.
        /// Returns null if there is no cached value for the specified key.
        /// </summary>
        /// <typeparam name="T">Type of the item to be retrieved from the cache.</typeparam>
        /// <param name="cacheKey">A unique identifier for the cache entry.</param>
        /// <returns>
        /// The cached value for the specified key, or null if no value found in the cache.
        /// </returns>
        public override T Get<T>(string cacheKey)
        {
            return HttpRuntime.Cache.Get(cacheKey) as T;
        }

        /// <summary>
        /// Stores the provided value against the identified cache key.
        /// If the provided value is null the identified cache key is cleared.
        /// Dependencies are not supported.
        /// </summary>
        /// <typeparam name="T">Type of the item to be stored in the cache.</typeparam>
        /// <param name="cacheKey">A unique identifier for the cache entry.</param>
        /// <param name="item">The item value to set. If null is provided the key is cleared.</param>
        /// <param name="slidingExpiration">A value indicating when a cache entry should be evicted if it has not been accessed in a given span of time.</param>
        /// <param name="absoluteExpiration">A value indicating whether a cache entry should be evicted at a specified date and time.</param>
        /// <param name="dependencyKeys">If supported, an optional set of keys that this entry is dependent on. If these keys do not exist in the cache, or are changed or removed, this entry is removed from the cache.</param>
        public override void Set<T>(string cacheKey, T item, TimeSpan slidingExpiration, DateTimeOffset absoluteExpiration, params string[] dependencyKeys)
        {
            // Null values cannot be cached and are interpreted as clearing the cache.
            if (item == null)
            {
                // Remove does not throw if the key doesn't exist.
                HttpRuntime.Cache.Remove(cacheKey);
                return;
            }

            /* Set cache key to equal item */

            // Insert does not throw if key already exists.
            HttpRuntime.Cache.Insert(cacheKey, item, null, absoluteExpiration.LocalDateTime, slidingExpiration);
        }

        /// <summary>
        /// Returns a reference to an object that can be used as an applicable thread lock for this cache for the specified key.
        /// This is useful for functionality such as double-check locking.
        ///  </summary>
        /// <param name="cacheKey">The cache key.</param>
        /// <returns>A cache-specific key-specific thread lock.</returns>
        public override object GetLock(string cacheKey)
        {
            return WebCacheLock.GetOrAdd(cacheKey, new object());
        }
    }
}