using System;
using System.Collections.Generic;
using System.Text;
using HIPS.Client.Proxy;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using HIPS.Web.Components.ServiceModel;
using HIPS.Web.Model.AssistedRegistration;
using HIPS.Web.Model.Common;
using HIPS.Web.ModelInterface.AssistedRegistration;

namespace HIPS.Web.Data.Hips
{
    /// <summary>
    /// Assisted Registration Service
    /// </summary>
    public class AssistedRegistrationService : IAssistedRegistrationService
    {
        /// <summary>
        /// The client proxy
        /// </summary>
        private AssistedRegistrationProxy clientProxy = new AssistedRegistrationProxy("AssistedRegistrationEndPoint");

        public RegisterResponse RegisterApplicant(PersonDemographic applicant, Hospital applicantHospital, bool applicantConsent, Model.AssistedRegistration.IndigenousStatus applicantIndigenousStatus, Dictionary<MedicareConsent, bool> medicareConsents, IdentityVerificationMethod identityVerificationMethod, IvcDeliveryMethodDecision ivcDeliveryMethod, UserDetails operatingUser, int? applicantPatientMasterId)
        {
            return Register(applicant, null, applicantHospital, applicantConsent, applicantIndigenousStatus, null, null, medicareConsents, identityVerificationMethod, ivcDeliveryMethod, operatingUser, applicantPatientMasterId);
        }

        public RegisterResponse RegisterApplicant(ValidatedPatient applicant, bool applicantConsent, Model.AssistedRegistration.IndigenousStatus applicantIndigenousStatus, Dictionary<MedicareConsent, bool> medicareConsents, IdentityVerificationMethod identityVerificationMethod, IvcDeliveryMethodDecision ivcDeliveryMethod, UserDetails operatingUser, int? applicantPatientMasterId)
        {
            return Register(applicant.RegisteredDemographics, applicant.VerifiedIhi, applicant.CurrentLocation.Hospital, applicantConsent, applicantIndigenousStatus, null, null, medicareConsents, identityVerificationMethod, ivcDeliveryMethod, operatingUser, applicantPatientMasterId);
        }

        public RegisterResponse RegisterDependant(PersonDemographic applicant, Hospital applicantHospital, bool applicantConsent, Model.AssistedRegistration.IndigenousStatus applicantIndigenousStatus, PersonDemographic representative, bool representativeConsent, Dictionary<MedicareConsent, bool> medicareConsents, IdentityVerificationMethod identityVerificationMethod, IvcDeliveryMethodDecision ivcDeliveryMethod, UserDetails operatingUser, int? applicantPatientMasterId)
        {
            return Register(applicant, null, applicantHospital, applicantConsent, applicantIndigenousStatus, representative, representativeConsent, medicareConsents, identityVerificationMethod, ivcDeliveryMethod, operatingUser, applicantPatientMasterId);
        }

        public RegisterResponse RegisterDependant(ValidatedPatient applicant, bool applicantConsent, Model.AssistedRegistration.IndigenousStatus applicantIndigenousStatus, PersonDemographic representative, bool representativeConsent, Dictionary<MedicareConsent, bool> medicareConsents, IdentityVerificationMethod identityVerificationMethod, IvcDeliveryMethodDecision ivcDeliveryMethod, UserDetails operatingUser, int? applicantPatientMasterId)
        {
            return Register(applicant.RegisteredDemographics, applicant.VerifiedIhi, applicant.CurrentLocation.Hospital, applicantConsent, applicantIndigenousStatus, representative, representativeConsent, medicareConsents, identityVerificationMethod, ivcDeliveryMethod, operatingUser, applicantPatientMasterId);
        }

        private RegisterResponse Register(PersonDemographic applicant, VerifiedIhi verifiedIhi, Hospital applicantHospital, bool applicantConsent, Model.AssistedRegistration.IndigenousStatus applicantIndigenousStatus, PersonDemographic representative, bool? representativeConsent, Dictionary<MedicareConsent, bool> medicareConsents, IdentityVerificationMethod identityVerificationMethod, IvcDeliveryMethodDecision ivcDeliveryMethod, UserDetails operatingUser, int? applicantPatientMasterId)
        {
            // Load Medicare Consent
            List<RegistrationDocumentConsent> documentConsentList = ModelConversions.LoadMedicareConsent(medicareConsents);

            // Load IVC Delivery Method
            RegistrationCorrespondenceChannel correspondenceChannel = ModelConversions.LoadIvcChannel(ivcDeliveryMethod);

            // Load Identity Verification Method
            EvidenceOfIdentityType evidenceOfIndentityType = ModelConversions.LoadEvidenceOfIdentityType(identityVerificationMethod);

            // Load Indigenous Status
            HIPS.PcehrDataStore.Schemas.Enumerators.IndigenousStatus indigenousStatus = ModelConversions.LoadIndigenousStatus(applicantIndigenousStatus);

            // Load Applicant
            PatientIdentifierBase applicantIdentifier;
            if (verifiedIhi != null)
            {
                applicantIdentifier = ModelConversions.LoadValidatedIhi(applicant, verifiedIhi, applicantHospital);
            }
            else
            {
                applicantIdentifier = ModelConversions.LoadDemographics(applicant, applicantHospital);
            }

            PcehrRegistrationResponse response;

            // Is dependant / has representative
            if (representative != null)
            {
                Demographic representativeIdentifier = ModelConversions.LoadDemographics(representative, applicantHospital);

                if (!representativeConsent.HasValue)
                {
                    throw new InvalidOperationException("Unexpected missing consent for representative.");
                }

                // Re-registering
                if (applicantPatientMasterId != null)
                {
                    response = clientProxy.ReregisterDependant(applicantPatientMasterId.Value, applicantIdentifier, representativeIdentifier, evidenceOfIndentityType, indigenousStatus, documentConsentList, correspondenceChannel, applicantConsent, representativeConsent.Value, operatingUser);
                }
                else // First register attempt
                {
                    response = clientProxy.RegisterDependant(applicantIdentifier, representativeIdentifier, evidenceOfIndentityType, indigenousStatus, documentConsentList, correspondenceChannel, applicantConsent, representativeConsent.Value, operatingUser);
                }
            }
            else // Standard applicant / not dependant
            {
                // Re-registering
                if (applicantPatientMasterId != null)
                {
                    response = clientProxy.ReregisterPatient(applicantPatientMasterId.Value, applicantIdentifier, evidenceOfIndentityType, indigenousStatus, documentConsentList, correspondenceChannel, applicantConsent, operatingUser);
                }
                else // First register attempt
                {
                    response = clientProxy.RegisterPatient(applicantIdentifier, evidenceOfIndentityType, indigenousStatus, documentConsentList, correspondenceChannel, applicantConsent, operatingUser);
                }
            }

            // Process Response

            if (response.HipsResponse.Status == HipsResponseIndicator.OK)
            {
                return new RegisterResponse { Ivc = response.IvcCode, IvcExpiry = response.IvcExpiryDate };
            }
            else
            {
                RegisterResponse resp = new RegisterResponse() { PatientMasterId = response.PatientMasterId == 0 ? (int?)null : response.PatientMasterId };

                if (response.HipsResponse.Status == HipsResponseIndicator.PcehrServiceError)
                {
                    StringBuilder message = new StringBuilder();
                    if (!string.IsNullOrEmpty(response.HipsResponse.ResponseCode))
                    {
                        message.Append(response.HipsResponse.ResponseCode);
                    }
                    if (!string.IsNullOrEmpty(response.HipsResponse.ResponseCodeDescription))
                    {
                        if (message.Length > 0)
                        {
                            message.Append(", ");
                        }
                        message.Append(response.HipsResponse.ResponseCodeDescription);
                    }
                    if (!string.IsNullOrEmpty(response.HipsResponse.HipsErrorMessage))
                    {
                        if (message.Length > 0)
                        {
                            message.Append(", ");
                        }
                        message.Append(response.HipsResponse.HipsErrorMessage);
                    }

                    resp.Errors.Add(string.Format("Service Error: {0} ({1})", response.HipsResponse.Status.ToString(), message.ToString()));
                    return resp;
                }
                else
                {
                    resp.Errors.Add(string.Format("Register Error: {0} ({1})", response.HipsResponse.Status.ToString(), string.Join(", ", response.HipsResponse.HipsErrorMessage)));
                    return resp;
                }
            }
        }

        public void Dispose()
        {
            if (clientProxy != null)
            {
                clientProxy.DisposeSafely();
                clientProxy = null;
            }
        }
    }
}