﻿using System;
using System.Collections.Generic;

using HIPS.P2P.Service.Contract.Common;
using HIPS.P2P.Service.Contract.Messaging.Delivery.Message;
using HIPS.Web.Components.ServiceModel;
using HIPS.Web.Data.P2P.Mapping;
using HIPS.Web.ModelInterface.Messaging;

using CommonSchema = HIPS.Web.Model.Messaging;
using ServiceSchema = HIPS.P2P.Service.Contract.Common.DTO;

namespace HIPS.Web.Data.P2P.Service
{
    /// <summary>
    /// Defines the implementation of the Message Delivery interface.
    /// </summary>
    public class MessageDeliveryService : ServiceBase<HIPS.P2P.Client.Proxy.MessageDeliveryProxy>, IMessageDeliveryService
    {
        #region Constructor

        /// <summary>
        /// Initialises a new instance of the <see cref="MessageDeliveryService"/> class.
        /// </summary>
        public MessageDeliveryService()
        {
            this.ClientProxy = new HIPS.P2P.Client.Proxy.MessageDeliveryProxy("MessageDeliveryEndPoint");
        }

        #endregion

        #region Methods

        /// <summary>
        /// Returns a list of outbound sealed messages.
        /// </summary>
        /// <param name="sender">Sender of the outbound sealed message to be return.</param>
        /// <param name="receiver">Receiver of the outbound sealed message to be return.</param>
        /// <param name="deliveryPeriod">Start/End of the outbound sealed message to be return.</param>
        /// <param name="messageStatusId">Message status Id of outbound sealed message to be return.</param>        
        /// <param name="documentType">Document type of the outbound sealed message to be return.</param>
        /// <param name="userContext">User context within which the service is accessed.</param>
        /// <returns>A ServiceResponse object.</returns>
        public ServiceResponse<List<CommonSchema.OutboundSealedMessage>> QueryMessages(
            CommonSchema.MessageAddressee sender,
            CommonSchema.MessageAddressee receiver,
            CommonSchema.DateTimeRange deliveryPeriod,
            int? messageStatusId,            
            string documentType,          
            Model.Identity.LocalUser userContext, 
            int recordLimit,
            int recordSkipFrom)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var request = new QueryMessagesRequest();                
                request.Senders.Add(ObjectMapper.Map<ServiceSchema.MessageAddressee>(sender));
                request.DeliveryPeriod = ObjectMapper.Map<ServiceSchema.DateTimeRange>(deliveryPeriod);
                request.User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext);
                request.RecordLimit = recordLimit;
                request.RecordSkipFrom = recordSkipFrom;

                if (receiver != null)
                {
                    request.Receivers.Add(ObjectMapper.Map<ServiceSchema.MessageAddressee>(receiver));
                }

                if (messageStatusId.HasValue)
                {
                    request.MessageStatuses.Add(ObjectMapper.Map<OutboundMessageStatus>(messageStatusId.Value));
                }

                if (!string.IsNullOrEmpty(documentType))
                {
                    request.DocumentTypes.Add(documentType);
                }

                var result = this.ClientProxy.QueryMessages(request);

                // HIPS Status should be OK
                isSuccessful = result.Status == ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<List<CommonSchema.OutboundSealedMessage>>(ObjectMapper.Map<List<CommonSchema.OutboundSealedMessage>>(result.Data), isSuccessful, messages, DateTime.Now);
            }
            catch (System.Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<List<CommonSchema.OutboundSealedMessage>>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Aborts a specified outbound message.
        /// </summary>
        /// <param name="invocationIdentifier">Identifier of the outbound message.</param>
        /// <param name="userContext">User context within which the service is accessed.</param>
        /// <returns>A ServiceResponse object.</returns>
        public ServiceResponse<object> AbortMessage(string invocationIdentifier, Model.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;
            
            try
            {
                var request = new AbortMessageRequest()
                {
                    InvocationIdentifier = invocationIdentifier,
                    User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext)
                };

                var result = this.ClientProxy.AbortMessage(request);
                // HIPS Status should be OK
                isSuccessful = result.Status == ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<object>(null, isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<object>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Retries a specified outbound message.
        /// </summary>
        /// <param name="invocationIdentifier">Identifier of the outbound message.</param>
        /// <param name="expiryDate">Sets a new expiry date for the outbound message.</param>
        /// <param name="userContext">User context within which the service is accessed.</param>
        /// <returns>A ServiceResponse object.</returns>
        public ServiceResponse<object> RetryMessage(string invocationIdentifier, DateTime expiryDate, Model.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var request = new RetryMessageRequest()
                {
                    InvocationIdentifier = invocationIdentifier,
                    Expiry = expiryDate,
                    User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext)
                };

                var result = this.ClientProxy.RetryMessage(request);
                // HIPS Status should be OK
                isSuccessful = result.Status == ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<object>(null, isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<object>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Downloads the message content of the specified outbound message.
        /// </summary>
        /// <param name="invocationIdentifier">Identifier of the outbound message.</param>
        /// <param name="userContext">User context within which the service is accessed.</param>
        /// <returns>A ServiceResponse object.</returns>
        public ServiceResponse<CommonSchema.BinaryMessageContent> GetMessage(string invocationIdentifier, Model.Identity.LocalUser userContext)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            try
            {
                var request = new GetMessageRequest()
                {
                    InvocationIdentifier = invocationIdentifier,
                    User = ObjectMapper.Map<ServiceSchema.UserIdentity.LocalUser>(userContext)
                };

                var result = this.ClientProxy.GetMessage(request);
                isSuccessful = result.Status == ResponseStatus.OK;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(result.Messages));

                return new ServiceResponse<CommonSchema.BinaryMessageContent>(ObjectMapper.Map<CommonSchema.BinaryMessageContent>(result.Data), isSuccessful, messages, DateTime.Now);
            }
            catch (Exception ex)
            {
                isSuccessful = false;

                // Add HIPS messages
                messages.AddRange(ObjectMapper.Map<List<ResponseMessage>>(ex));

                return new ServiceResponse<CommonSchema.BinaryMessageContent>(null, isSuccessful, messages, DateTime.Now);
            }
        }

        /// <summary>
        /// Downloads the CDA message content of the specified outbound message.
        /// </summary>
        /// <param name="invocationIdentifier">Identifier of the outbound message.</param>
        /// <param name="userContext">User context within which the service is accessed.</param>
        /// <returns>A ServiceResponse object.</returns>
        public ServiceResponse<CommonSchema.CdaMessageContent> GetCdaMessage(string invocationIdentifier, Model.Identity.LocalUser userContext)
        {
            throw new NotImplementedException();
        }

        #endregion

        
    }
}
