﻿using System.Collections.Generic;

using HIPS.Web.Components.Cache;
using HIPS.Web.Model.Common;

namespace HIPS.Web.ModelInterface.DischargeSummary
{
    /// <summary>
    /// Implements a repository for interacting with discharge summary references.
    /// </summary>
    public class CachedDischargeSummaryReferenceRepository : IDischargeSummaryReferenceRepository
    {
        #region Fields

        /// <summary>
        /// Cache key format.
        /// </summary>
        private readonly string cacheKeyFormat;

        /// <summary>
        /// Cache provider.
        /// </summary>
        private readonly ICacheProvider cacheProvider;

        /// <summary>
        /// Reference repository.
        /// </summary>
        private IDischargeSummaryReferenceRepository referenceRepository;

        #endregion Fields

        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="CachedDischargeSummaryReferenceRepository"/> class.
        /// </summary>
        /// <param name="referenceRepository">Reference repository.</param>
        /// <param name="cacheProvider">Cache provider.</param>
        /// <param name="cacheKeyPrefix">Cache key format.</param>
        public CachedDischargeSummaryReferenceRepository(
            IDischargeSummaryReferenceRepository referenceRepository,
            ICacheProvider cacheProvider,
            string cacheKeyPrefix = "")
        {
            this.referenceRepository = referenceRepository;
            this.cacheProvider = cacheProvider;
            this.cacheKeyFormat = cacheKeyPrefix + ".CachedDischargeSummaryReferenceRepository.{0}";
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Gets all clinical specialties.
        /// </summary>
        /// <returns>List of Clinical Specialties.</returns>
        public List<ClinicalSpecialty> GetClinicalSpecialties()
        {
            return this.cacheProvider.GetOrSet(string.Format(this.cacheKeyFormat, "GetClinicalSpecialties()"), () => this.referenceRepository.GetClinicalSpecialties());
        }

        /// <summary>
        /// Dispose.
        /// </summary>
        public void Dispose()
        {
            if (this.referenceRepository != null)
            {
                this.referenceRepository.Dispose();
                this.referenceRepository = null;
            }
        }

        #endregion Methods
    }
}