﻿#region Using Statements

using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;

using HIPS.Web.Components.Common;
using HIPS.Web.Components.Web;
using HIPS.Web.Data.P2P.Service;
using HIPS.Web.Model.Common;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.ModelInterface.Directory;
using HIPS.Web.UI.Areas.Directory.ViewModels;
using HIPS.Web.UI.Conversion.Custom;
using HIPS.Web.UI.Filters;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.Helpers.Mapping;
using HIPS.Web.UI.ViewModels.Shared;

#endregion

namespace HIPS.Web.UI.Areas.Directory.Controllers
{
    /// <summary>
    /// Controller for the maintaining delegates as part of the "Directory Maintenance" feature.
    /// </summary>
    [NoCache]
    [HpoRequired]
    public class DelegatesController : HIPS.Web.UI.Controllers.ControllerBase
    {
        #region Fields

        /// <summary>
        /// Gets the directory configuration service to be used by this controller.
        /// </summary>
        private readonly IDirectoryConfigurationService directoryConfigurationService;

        #endregion

        #region Constructors

        /// <summary>
        /// Initialises a new instance of the DelegatesController class
        /// </summary>
        /// <param name="directoryConfigurationService">Directory configuration service to be used by this controller.</param>
        /// <param name="settingsRepository">Settings repository to be used by this controller.</param>
        /// <param name="sessionConfiguration">Session configuration to be used by this controller.</param>
        public DelegatesController(IDirectoryConfigurationService directoryConfigurationService, ISettingsRepository settingsRepository, ISessionConfiguration sessionConfiguration) : base(settingsRepository, sessionConfiguration)
        {
            this.directoryConfigurationService = directoryConfigurationService;
        }

        #endregion

        #region Methods

        #region Actions

        /// <summary>
        /// Display delegates for maintenance.
        /// </summary>
        /// <returns>Returns the index view</returns>
        [HttpGet]
        public ActionResult Index()
        {
            return this.View();
        }

        /// <summary>
        /// List delegates for maintenance.
        /// </summary>
        /// <returns>Returns the list view</returns>
        [HttpGet]
        public ActionResult List()
        {
            var m = new ListDelegatesViewModel();
            var result = this.directoryConfigurationService.ListDelegates(this.GetLocalUser());
            
            if (result.IsSuccessful)
            {
                m.Delegates = result.Data.OrderByDescending(i => i.PriorityNumber).ThenBy(i => i.Name).ToList();

                // Setup the intermediary select list
                var intermediaries = this.directoryConfigurationService.ListIntermediaries(this.GetLocalUser());
                m.Intermediaries = intermediaries.Data.OrderBy(o => o.Name).ToList();
                m.Intermediaries.Insert(0, new Model.Directory.Intermediary { Name = "-", ServiceEndpointUri = string.Empty });
            }

            // Copy over response messages.
            m.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(result.Messages));

            return this.View(m);
        }

        /// <summary>
        /// Activate a specified delegate.
        /// </summary>
        /// <param name="m">View model identifying the delegate to be activated or deactivated.</param>
        /// <returns>Partial view result containing any messages generated by the action.</returns>
        [HttpPost]
        public ActionResult Activate(ActivateDelegateViewModel m)
        {
            if (!this.ModelState.IsValid)
            {
                var modelErrorMessages = this.ModelState.Values.SelectMany(i => i.Errors).Select(i => new ViewMessage(i.ErrorMessage, MessageLevel.Error));
                m.Messages.AddRange(modelErrorMessages);
            }
            else
            {
                var result = this.directoryConfigurationService.ActivateDelegate(
                    m.QualifiedIdentifier, m.IsActive, this.GetLocalUser());
                
                // Copy over response messages.
                m.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(result.Messages));
            }

            return this.PartialView("ViewMessageList", m.Messages);
        }

        /// <summary>
        /// Sets the intermediary of the specified delegate.
        /// </summary>
        /// <param name="m">View model identifying the intermediary information.</param>
        /// <returns>Partial view result containing any messages generated by the action.</returns>
        [HttpPost]
        public ActionResult SetIntermediary(MediateDelegateViewModel m)
        {
            if (!this.ModelState.IsValid)
            {
                var modelErrorMessages = this.ModelState.Values.SelectMany(i => i.Errors).Select(i => new ViewMessage(i.ErrorMessage, MessageLevel.Error));
                m.Messages.AddRange(modelErrorMessages);
            }
            else
            {
                var result = this.directoryConfigurationService.MediateDelegate(
                    m.QualifiedIdentifier, m.ServiceEndpointUri, this.GetLocalUser());

                // Copy over response messages.
                m.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(result.Messages));
            }

            return this.PartialView("ViewMessageList", m.Messages);
        }

        /// <summary>
        /// Reprioritises the order of delegates.
        /// </summary>
        /// <param name="m">View model identifying the prioritisation information.</param>
        /// <returns>Partial view result containing any messages generated by the action.</returns>
        [HttpPost]
        public ActionResult Reprioritise(ReprioritiseDelegateViewModel m)
        {
            if (!this.ModelState.IsValid)
            {
                var modelErrorMessages = this.ModelState.Values.SelectMany(i => i.Errors).Select(i => new ViewMessage(i.ErrorMessage, MessageLevel.Error));
                m.Messages.AddRange(modelErrorMessages);
            }
            else
            {
                // reconstruct qualified identifier string
                string qualifiedIdentifier = JtmlSafeId.Decode(m.Id);

                // get relative delegate's qualified identifier
                var delegates = this.directoryConfigurationService.ListDelegates(this.GetLocalUser());
                if (delegates.IsSuccessful)
                {
                    // extract relative delegate from list
                    var relative = delegates
                                    .Data
                                    .OrderByDescending(o => o.PriorityNumber)
                                    .Skip(m.ToPosition + (m.FromPosition > m.ToPosition ? 0 : 1))
                                    .Take(1)
                                    .SingleOrDefault();

                    // invoke prioritisation
                    var result = this.directoryConfigurationService.PrioritiseDelegate(
                        qualifiedIdentifier, relative == null ? null : relative.QualifiedIdentifier, this.GetLocalUser());

                    if (!result.IsSuccessful)
                    {
                        // Copy over response messages.
                        m.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(result.Messages));
                    }
                }
                else
                {
                    // Copy over response messages.
                    m.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(delegates.Messages));
                }
            }

            return this.PartialView("ViewMessageList", m.Messages);
        }

        #endregion

        #endregion
    }
}
