﻿using System.Collections.Generic;
using System.Linq;
using System.Web.Mvc;

using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Web.Components.Collections;
using HIPS.Web.Components.Common;
using HIPS.Web.Components.ServiceModel;
using HIPS.Web.Components.Web;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.ModelInterface.ConsentManagement;
using HIPS.Web.UI.Filters;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.Helpers.Mapping;
using HIPS.Web.UI.ViewModels.DisclosureManagement;
using HIPS.Web.UI.ViewModels.Shared;
using HIPS.Web.ModelInterface.AssistedRegistration;
using HIPS.ServiceContracts.Patient.Message;

namespace HIPS.Web.UI.Controllers
{
    /// <summary>
    /// Controller for the management of disclosure of hidden PCEHRs.
    /// </summary>
    [NoCache]
    [HpoRequired]
    public class DisclosureManagementController : ControllerBase
    {
        #region Fields

        /// <summary>
        /// Gets the hospital repository to be used by this controller.
        /// </summary>
        private readonly IHospitalRepository hospitalRepository;

        /// <summary>
        /// Gets the patient repository to be used by this controller.
        /// </summary>
        private readonly IPatientRepository patientRepository;

        /// <summary>
        /// Gets the consent management service to be used by this controller.
        /// </summary>
        private readonly IConsentManagementService consentManagementService;

        /// <summary>
        /// Assisted Registration Reference repository to be used by this controller.
        /// </summary>
        private readonly IAssistedRegistrationReferenceRepository assistedRegistrationReferenceRepository;

        #endregion Fields

        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="DisclosureManagementController" /> class.
        /// </summary>
        /// <param name="hospitalRepository">Hospital repository to be used by this controller.</param>
        /// <param name="patientRepository">Patient repository to be used by this controller.</param>
        /// <param name="settingsRepository">Settings repository to be used by this controller.</param>
        /// <param name="consentManagementService">Consent management service to be used by this controller.</param>
        /// <param name="sessionConfiguration">Session configuration to be used by this controller.</param>
        public DisclosureManagementController(
            IHospitalRepository hospitalRepository,
            IPatientRepository patientRepository,
            ISettingsRepository settingsRepository,
            IConsentManagementService consentManagementService,
            IAssistedRegistrationReferenceRepository assistedRegistrationReferenceRepository,
            ISessionConfiguration sessionConfiguration)
            : base(settingsRepository, sessionConfiguration)
        {
            this.hospitalRepository = hospitalRepository;
            this.patientRepository = patientRepository;
            this.consentManagementService = consentManagementService;
            this.assistedRegistrationReferenceRepository = assistedRegistrationReferenceRepository;
        }

        #endregion Constructors

        #region Methods

        #region Actions

        /// <summary>
        /// Display a list of patients for a selected hospital on the Disclose Hidden PCEHR screen.
        /// </summary>
        /// <param name="lookupMessages">Additional messages to be displayed coming from the lookup MRN action.</param>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult Patients(ViewMessageList lookupMessages = null)
        {
            string hospitalId = SessionConfiguration.RepresentingHospital.HospitalFacilityCode;
            // Create ViewModel:
            var m = new PatientsViewModel() { HospitalId = hospitalId };
            this.LoadCurrentContext(m);

            // Load reference data:
            var hospitals = ObjectMapper.Map<IEnumerable<HospitalViewModel>>(this.hospitalRepository.GetHospitals(this.DefaultHospitalCodeSystem), new Helpers.Mapping.Context.HospitalMappingContext(this.DefaultHospitalCodeSystem));

            if ((hospitals != null) && (hospitals.Count() > 0))
            {
                // Update ViewModel with reference data:
                m.Hospitals = hospitals.ToSelectListItems(h => h.Code, h => h.Name);
            }
            else
            {
                m.Messages.Add("No hospitals available for selection.", MessageLevel.Error);
            }

            // Don't load patients if the hospital has not been selected.
            if (hospitalId != null)
            {
                // Load patients for selected hospital.
                var response = this.patientRepository.ListAdmittedPatients(
                    this.GetCurrentUserDetails(), this.DefaultHospitalCodeSystem, hospitalId, withIhi: null, withPcehr: null, excludeMedicareExclusions: false);

                // Ensure loading was successful.
                if (response.IsSuccessful)
                {
                    if (response.Data.AdmittedPatientList.Count() > 0)
                    {
                        // Update ViewModel with patients.
                        m.Patients.AddRange(ObjectMapper.Map<IEnumerable<PatientViewModel>>(response.Data.AdmittedPatientList));
                    }
                    else
                    {
                        m.Messages.Add("There are no patients at the selected hospital.", MessageLevel.Information);
                    }
                }
                else
                {
                    string errorMessage = "Failed to retrieve patients for the selected hospital.";
                    // Log details:
                    Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, response.Messages.AsString())));
                    // Display error message.
                    this.SetAjaxErrorResponseCode();
                    m.Messages.Add(errorMessage, MessageLevel.Error);
                }
                if (lookupMessages != null)
                {
                    m.Messages.AddRange(lookupMessages);
                }
            }
            return this.View("Patients", m);
        }

        /// <summary>
        /// Look up a patient by MRN. If found then display the appropriate
        /// form for disclosing or removing the disclosure of the PCEHR.
        /// Otherwise redisplay the list of patients with a message indicating
        /// that the patient was not found.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="lookupMrn">User-entered identifier of the patient.</param>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult LookupMrn(string hospitalId, string lookupMrn)
        {
            var m = new DisclosePcehrViewModel(hospitalId, lookupMrn);
            this.LoadCurrentContext(m);
            if (m.CurrentPatient != null)
            {
                return m.CurrentPatient.PcehrDisclosed ? this.RemoveDisclosure(hospitalId, lookupMrn) : this.View("DisclosePcehr", m);
            }
            else
            {
                ViewMessageList messages = new ViewMessageList();
                messages.Add("There is no registered patient with this MRN.", MessageLevel.Error);
                return this.Patients(messages);
            }
        }

        /// <summary>
        /// Display a form to disclose the existence of a PCEHR for a selected patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the patient.</param>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult DisclosePcehr(string hospitalId, string patientId)
        {
            var m = new DisclosePcehrViewModel(hospitalId, patientId);
            this.LoadCurrentContext(m);
            return this.View("DisclosePcehr", m);
        }

        /// <summary>
        /// Display a form to remove the disclosure of the existence of a PCEHR for a selected patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the patient.</param>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult RemoveDisclosure(string hospitalId, string patientId)
        {
            var m = new DisclosePcehrViewModel(hospitalId, patientId);
            this.LoadCurrentContext(m);
            Mrn patientIdentifier = new Mrn(m.PatientId, m.HospitalId, m.CurrentHospital.CodeSystemCode);
            var response = this.patientRepository.GetPatientDisclosureDetails(this.GetCurrentUserDetails(), patientIdentifier);
            if (response.IsSuccessful)
            {
                m.DisclosureAuditInformation = response.Data.DisclosureAuditInformation;
            }
            else
            {
                string errorMessage = "Failed to retrieve disclosure details for the selected patient.";
                // Log details:
                Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, response.Messages.AsString())));
                // Display error message.
                this.SetAjaxErrorResponseCode();
                m.Messages.Add(errorMessage, MessageLevel.Error);
            }
            return this.View("RemoveDisclosure", m);
        }

        /// <summary>
        /// Process the AJAX request to disclose the existence of a PCEHR for a selected patient.
        /// </summary>
        /// <param name="disclosure">Contents of the form submission.</param>
        /// <returns>The JSON result.</returns>
        [HttpPost]
        public ActionResult DisclosePcehr(DisclosePcehrViewModel disclosure)
        {
            return this.DiscloseOrRemove(disclosure, true);
        }

        /// <summary>
        /// Process the AJAX request to remove the disclosure of the existence of a PCEHR for a selected patient.
        /// </summary>
        /// <param name="removeDisclosureManagement">Contents of the form submission.</param>
        /// <returns>The JSON result.</returns>
        [HttpPost]
        public ActionResult RemoveDisclosure(DisclosePcehrViewModel removeDisclosureManagement)
        {
            return this.DiscloseOrRemove(removeDisclosureManagement, false);
        }

        /// <summary>
        /// Process the AJAX request to disclose or remove disclosure of the existence of a PCEHR for a selected patient.
        /// </summary>
        /// <param name="model">Contents of the form submission.</param>
        /// <param name="pcehrDisclosed">Whether PCEHR existence is being disclosed (true) or the disclosure is being removed (false).</param>
        /// <returns>The JSON result.</returns>
        private ActionResult DiscloseOrRemove(DisclosePcehrViewModel model, bool pcehrDisclosed)
        {
            // As returning JSON ensure the request was AJAX.
            if (!Request.IsAjaxRequest())
            {
                return this.Content("JavaScript is required for this functionality.");
            }

            // Check binding validation errors
            if (!ModelState.IsValid)
            {
                return this.Json(new { Echo = model, Errors = ModelState.ToErrorDictionary() });
            }

            this.LoadCurrentContext(model);

            Mrn patientIdentifier = new Mrn(model.PatientId, model.HospitalId, model.CurrentHospital.CodeSystemCode);

            ServiceResponse<bool> response = this.consentManagementService.RecordDisclosure(pcehrDisclosed, model.Notes, patientIdentifier, this.GetCurrentUserDetails());

            return this.Json(new { Response = response, Echo = model, Errors = ModelState.ToErrorDictionary() });
        }

        /// <summary>
        /// Displays the Modal for the Register Patient service
        /// </summary>
        /// <param name="mrn">Mrn.</param>
        /// <returns>The Partial view for the Register Patient Modal.</returns>
        public ActionResult Register(string mrn)
        {
            var model = new RegisterPatientViewModel();
            var sexes = this.assistedRegistrationReferenceRepository.GetSexes();

            List<SelectListItem> sexList = new List<SelectListItem>();
            foreach (var sex in sexes)
            {
                sexList.Add(new SelectListItem()
                {
                    Text = sex.Description,
                    Value = sex.Code
                });
            }

            model.Sexes = sexList;
            model.Mrn = mrn;

            string hospitalId = SessionConfiguration.RepresentingHospital.HospitalFacilityCode;

            var hospital = this.hospitalRepository.GetHospitals(this.DefaultHospitalCodeSystem).Where(i => i.Description == hospitalId).FirstOrDefault();
            ViewBag.HospitalName = hospital.Name;

            return this.PartialView("_Register", model);
        }

        /// <summary>
        /// Saves the Patient using the filled up model
        /// </summary>
        /// <param name="model">RegisterPatientViewModel.</param>
        /// <returns>The Result of the Register Patient Service</returns>
        public ActionResult Save(RegisterPatientViewModel model)
        {
            // var model = new RegisterPatientViewModel();
            model.Messages = new ViewMessageList();

            var validationResults = ValidateRegisterPatient(model);

            if (validationResults.Count > 0)
            {
                model.Messages.AddRange(validationResults);
                return this.PartialView("ViewMessageList", model.Messages);
            }

            var response = this.patientRepository.RegisterPatient(this.GetLocalUser(), model.FamilyName, model.GivenName, model.SexCode, model.DateOfBirth, model.MedicareNumber, model.MedicareNumberIrn, model.DvaFileNumber, model.Mrn, SessionConfiguration.RepresentingHospital.HospitalFacilityCode, this.DefaultHospitalCodeSystem);

            if (response.IsSuccessful)
            {
                model.Messages.Add("Patient successfully added.", MessageLevel.Information);
            }
            else
            {
                model.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(response.Messages));
            }

            return this.PartialView("ViewMessageList", model.Messages);
        }

        #endregion Actions

        #region Helpers

        private ViewMessageList ValidateRegisterPatient(RegisterPatientViewModel model)
        {
            var result = new ViewMessageList();

            if (string.IsNullOrEmpty(model.FamilyName))
            {
                result.Add("Family name is required.", MessageLevel.Error);
            }

            if (string.IsNullOrEmpty(model.GivenName))
            {
                result.Add("Given name is required.", MessageLevel.Error);
            }

            if (string.IsNullOrEmpty(model.SexCode))
            {
                result.Add("Sex is required.", MessageLevel.Error);
            }

            if (model.DateOfBirth == null || model.DateOfBirth == System.DateTime.MinValue)
            {
                result.Add("Date of Birth is required.", MessageLevel.Error);
            }

            if (string.IsNullOrEmpty(model.Mrn))
            {
                result.Add("Mrn is required.", MessageLevel.Error);
            }
            else
            {
                // Check if a patient with the specified MRN already exists.
                // Note: In practice this is unlikely to occur because the register dialog window is only shown when the MRN search returned 0 results.
                // This can occur if a new patient with the specified MRN is created on another client while the dialog is active OR if the user 
                // manipulates the html to change the readonly MRN field.
                var mrn = new Mrn(model.Mrn, SessionConfiguration.RepresentingHospital.HospitalFacilityCode, this.DefaultHospitalCodeSystem);
                var response = this.patientRepository.GetPatientDisclosureDetails(this.GetCurrentUserDetails(), mrn);
                if (response.Data.AdmittedPatient != null) {
                    result.Add("There is already a registered patient with this MRN.", MessageLevel.Error);
                }
            }

            return result;
        }

        /// <summary>
        /// Loads required context into the provided view model.
        /// </summary>
        /// <param name="model">View model to load context into.</param>
        private void LoadCurrentContext(DisclosureManagementViewModelBase model)
        {
            var mrn = new CommonSchemas.PatientIdentifier.Mrn(model.PatientId, model.HospitalId, this.DefaultHospitalCodeSystem);
            var user = this.GetCurrentUserDetails();

            // Load current hospital.
            if (!string.IsNullOrEmpty(model.HospitalId))
            {
                var hospitals = ObjectMapper.Map<IEnumerable<HospitalViewModel>>(this.hospitalRepository.GetHospitals(this.DefaultHospitalCodeSystem), new Helpers.Mapping.Context.HospitalMappingContext(this.DefaultHospitalCodeSystem));

                if ((hospitals != null) && (hospitals.Count() > 0))
                {
                    model.CurrentHospital = hospitals.FirstOrDefault(h => h.Code == model.HospitalId);
                }
            }

            // Load current patient.
            if (!string.IsNullOrEmpty(model.PatientId))
            {
                var response = this.patientRepository.GetPatientDisclosureDetails(this.GetCurrentUserDetails(), mrn);
                model.CurrentPatient = ObjectMapper.Map<PatientViewModel>(response.Data.AdmittedPatient);
            }
        }

        #endregion Helpers

        #endregion Methods
    }
}