using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using System.Web.Mvc;

using HIPS.CommonSchemas.Hi;
using HIPS.HpiiSchemas;
using HIPS.ServiceContracts.Hpii.Message;
using HIPS.Web.Components.Collections;
using HIPS.Web.Components.Common;
using HIPS.Web.Components.ServiceModel;
using HIPS.Web.Components.Web;
using HIPS.Web.Model.Hpii;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.ModelInterface.Hi;
using HIPS.Web.ModelInterface.HpiiSearch;
using HIPS.Web.UI.Conversion.HpiiSearch;
using HIPS.Web.UI.Filters;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.Helpers.Mapping;
using HIPS.Web.UI.Helpers.Mapping.Context;
using HIPS.Web.UI.ViewModels.HpiiSearch;
using HIPS.Web.UI.ViewModels.Shared;

namespace HIPS.Web.UI.Controllers
{
    /// <summary>
    /// Controller for HPI-I search.
    /// </summary>
    [HpoRequired]
    public class HpiiSearchController : ControllerBase
    {
        #region Fields

        /// <summary>
        /// The HPI-I search service to be used by this controller.
        /// </summary>
        private readonly IHpiiSearchService hpiiSearchService;

        /// <summary>
        /// The HPI-I search service (v2) to be used by this controller.
        /// </summary>
        private readonly IHpiiSearchServiceV2 hpiiSearchServiceV2;

        /// <summary>
        /// The HI reference repository to be used by this controller.
        /// </summary>
        private readonly IHiReferenceRepository hiReferenceRepository;

        /// <summary>
        /// The HpiiSearch Reference repository to be used by this controller.
        /// </summary>
        private readonly IHpiiSearchReferenceRepository hpiiSearchReferenceRepository;

        #endregion

        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="HpiiSearchController" /> class.
        /// </summary>
        /// <param name="hpiiSearchService">The HPI-I search service to be used by this controller.</param>
        /// <param name="hpiiSearchServiceV2">The HPI-I search service (v2) to be used by this controller.</param>
        /// <param name="healthcareIdentifierReferenceRepository">The HI reference repository to be used by this controller.</param>
        /// <param name="hpiiSearchReferenceRepository">The HpiiSearch Reference repository to be used by this controller.</param>
        /// <param name="settingsRepository">The settings repository to be used by this controller.</param>
        /// <param name="sessionConfiguration">Session configuration to be used by this controller.</param>
        public HpiiSearchController(
            IHpiiSearchService hpiiSearchService,
            IHpiiSearchServiceV2 hpiiSearchServiceV2,
            IHiReferenceRepository healthcareIdentifierReferenceRepository,
            IHpiiSearchReferenceRepository hpiiSearchReferenceRepository,
            ISettingsRepository settingsRepository,
            ISessionConfiguration sessionConfiguration)
            : base(settingsRepository, sessionConfiguration)
        {
            this.hpiiSearchService = hpiiSearchService;
            this.hpiiSearchServiceV2 = hpiiSearchServiceV2;
            this.hiReferenceRepository = healthcareIdentifierReferenceRepository;
            this.hpiiSearchReferenceRepository = hpiiSearchReferenceRepository;
        }

        #endregion Constructors

        #region Methods      

        /// <summary>
        /// Displays the HPI-I identifier search form.
        /// Note: Output caching needs to be disabled to properly return partial views to ajax requests.
        /// </summary>
        /// <returns>View result.</returns>
        [HttpGet]
        [OutputCache(NoStore = true, Duration = 0, VaryByParam = "*")]
        public ActionResult SearchById()
        {
            // Create ViewModel
            SearchByIdViewModel viewModel = new SearchByIdViewModel();

            // Load reference data
            this.LoadAndAddReferenceData(viewModel);

            if (this.Request.IsAjaxRequest())
            {
                return this.PartialView("SearchById", viewModel);
            }

            // Render view
            return this.View("SearchById", viewModel);
        }

        /// <summary>
        /// Processes the submission of the HPI-I identifier search form.
        /// </summary>
        /// <param name="searchById">View model.</param>
        /// <returns>View result.</returns>
        [HttpPost]
        public ActionResult SearchById(SearchByIdViewModel searchById)
        {
            searchById.Hpio = this.SessionConfiguration.RepresentingHospital.Hpio;

            // Load reference data
            this.LoadAndAddReferenceData(searchById);

            // Check basic validation errors
            if (!this.ModelState.IsValid)
            {
                // Return view (with original ViewModel)
                return this.PartialView(searchById);
            }

            // Invoke service
            HpiiQueryResponse queryResponse;
            try
            {
                queryResponse = this.hpiiSearchService.SearchByIdentifier(searchById.ToIdentifierQuery(), this.GetCurrentUserDetails());
            }
            catch (EndpointNotFoundException endPoint)
            {
                // Add Model State error
                this.ModelState.AddModelError(string.Empty, "Unable to Connect to HIPS so the HPI-I cannot be validated.");
                Elmah.ErrorSignal.FromCurrentContext().Raise(endPoint);
                // Render view (with original ViewModel)
                return this.PartialView(searchById);
            }
            catch (Exception e)
            {
                // Add Model State error (TODO: Remove this when HIPS provides improved validation support)
                this.ModelState.AddModelError(string.Empty, "Unable to locate HPI-I.");
                Elmah.ErrorSignal.FromCurrentContext().Raise(e);
                // Render view (with original ViewModel)
                return this.PartialView(searchById);
            }

            // Convert to response ViewModel
            SearchResultViewModel responseVm = new SearchResultViewModel().LoadFromResponse(queryResponse);

            // Set View Logic Properties
            responseVm.ShowAustralianAddress = false;
            responseVm.ShowInternationalAddress = false;

            // Load reference data
            this.LoadAndAddReferenceData(responseVm, responseVm.ShowAustralianAddress, responseVm.ShowInternationalAddress);

            // If service errors, or "Not Found", return
            if (responseVm.ResponseMessages.Errors.Any() || queryResponse.HipsResponse.ResponseCode == "WSE0035")
            {
                // Set the Response Messages on the View Model being returned
                searchById.ResponseMessages = responseVm.ResponseMessages;

                // Render view (with original ViewModel)
                return this.PartialView(searchById);
            }

            // Add success to view model
            responseVm.ResponseMessages.Successes.Add("HPI-I Search result found.");
            // Check if the HPI-I Status is not Active then display a warning
            if (responseVm.HpiiStatus != Web.Model.HpiiStatus.Active.ToString())
                responseVm.ResponseMessages.Warnings.Add("Warning: The HPI-I found has a current status of " + responseVm.HpiiStatus);

            // Render response success view
            return this.PartialView("SearchResult", responseVm);
        }

        /// <summary>
        /// Displays the HPI-I demographic search form.
        /// Note: Output caching needs to be disabled to properly return partial views to ajax requests.
        /// </summary>
        /// <returns>View result.</returns>
        [HttpGet]
        [OutputCache(NoStore = true, Duration = 0, VaryByParam = "*")]
        public ActionResult SearchByDemographics()
        {
            // Create ViewModel
            SearchByDemographicsViewModel viewModel = new SearchByDemographicsViewModel();

            // Load reference data
            this.LoadAndAddReferenceData(viewModel);

            if (this.Request.IsAjaxRequest())
            {
                return this.PartialView(viewModel);
            }

            // Render view
            return this.View(viewModel);
        }

        /// <summary>
        /// Processes the submission of the HPI-I demographic search form.
        /// </summary>
        /// <param name="searchByDemographics">View model.</param>
        /// <returns>View result.</returns>
        [HttpPost]
        public ActionResult SearchByDemographics(SearchByDemographicsViewModel searchByDemographics)
        {
            searchByDemographics.Hpio = this.SessionConfiguration.RepresentingHospital.Hpio;

            // Load reference data
            this.LoadAndAddReferenceData(searchByDemographics);

            // Check basic validation errors
            if (!this.ModelState.IsValid)
            {
                // Return view (with original ViewModel)
                return this.PartialView(searchByDemographics);
            }

            // Invoke service
            HpiiQueryResponse queryResponse;
            try
            {
                queryResponse = this.hpiiSearchService.SearchByDemographic(searchByDemographics.ToDemographicQuery(), this.GetCurrentUserDetails());
            }
            catch (EndpointNotFoundException endPoint)
            {
                // Add Model State error
                this.ModelState.AddModelError(string.Empty, "Unable to Connect to HIPS so the HPI-I cannot be validated.");
                Elmah.ErrorSignal.FromCurrentContext().Raise(endPoint);
                // Render view (with original ViewModel)
                return this.PartialView(searchByDemographics);
            }
            catch (Exception e)
            {
                // Add Model State error (TODO: Remove this when HIPS provides improved validation support)
                this.ModelState.AddModelError(string.Empty, "Unable to locate HPI-I.");
                Elmah.ErrorSignal.FromCurrentContext().Raise(e);
                // Render view (with original ViewModel)
                return this.PartialView(searchByDemographics);
            }

            // Convert to response ViewModel
            SearchResultViewModel responseVm = new SearchResultViewModel().LoadFromResponse(queryResponse);

            // Set View Logic Properties
            responseVm.ShowInternationalAddress = responseVm.InternationalAddress.HasAnyValue();
            responseVm.ShowAustralianAddress = !responseVm.ShowInternationalAddress;

            // Load reference data
            this.LoadAndAddReferenceData(responseVm, responseVm.ShowAustralianAddress, responseVm.ShowInternationalAddress);

            // If service errors, or "Not Found", return
            if (responseVm.ResponseMessages.Errors.Any() || queryResponse.HipsResponse.ResponseCode == "WSE0035")
            {
                // Set the Response Messages on the View Model being returned
                searchByDemographics.ResponseMessages = responseVm.ResponseMessages;

                // Render view (with original ViewModel)
                return this.PartialView(searchByDemographics);
            }

            // Add success to view model
            responseVm.ResponseMessages.Successes.Add("HPI-I Search result found.");
            // Check if the HPI-I Status is not Active then display a warning
            if (responseVm.HpiiStatus != Web.Model.HpiiStatus.Active.ToString())
                responseVm.ResponseMessages.Warnings.Add("Warning: The HPI-I found has a current status of " + responseVm.HpiiStatus);

            // Render response success view
            return this.PartialView("SearchResult", responseVm);
        }

        /// <summary>
        /// Displays a list of Providers for the provided hospital.
        /// </summary>
        /// <returns>View result.</returns>
        [HttpGet]
        public ActionResult Providers()
        {
            var model = new ListLocalProvidersViewModel();

            string hospitalCode = this.SessionConfiguration.RepresentingHospital.HospitalFacilityCode;
            string hospitalCodeSystem = this.DefaultHospitalCodeSystem;
            var response = this.hpiiSearchServiceV2.ListLocalProviders(hospitalCode, hospitalCodeSystem, this.GetLocalUser());

            if (response.IsSuccessful)
            {
                if (response.Data.Count > 0)
                {
                    var context = new LocalProviderMappingContext(this.SessionConfiguration.RepresentingHospital.HospitalFacilityCode, this.DefaultHospitalCodeSystem);
                    model.LocalProviders = ObjectMapper.Map<List<LocalProviderViewModel>>(response.Data, context);
                }
                else
                {
                    model.Messages.Add("There are no local providers found.", MessageLevel.Information);
                }
            }
            else
            {
                string errorMessage = "Failed to retrieve local providers.";

                // Log details:
                Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception(string.Format("{0} {1}", errorMessage, response.Messages.AsString())));

                // Display error message.
                this.SetAjaxErrorResponseCode();
                model.Messages.Add(errorMessage, MessageLevel.Error);
            }

            return this.View("Providers", model);
        }

        /// <summary>
        /// Displays the Add Provider form.
        /// </summary>
        /// <returns>View result.</returns>
        [HttpGet]
        public ActionResult AddProvider()
        {
            var model = new AddProviderViewModel();
            this.LoadAndAddReferenceData(model);

            return View("AddProvider", model);
        }

        /// <summary>
        /// Process the result of Add Local Provider.
        /// </summary>
        /// <param name="model">View model.</param>
        /// <returns>View result.</returns>
        [HttpPost]
        [ValidateAntiForgeryToken]
        public ActionResult AddProvider(AddProviderViewModel model)
        {
            this.LoadAndAddReferenceData(model);

            if (this.ModelState.IsValid)
            {
                var user = this.GetLocalUser();

                string hospitalCode = this.SessionConfiguration.RepresentingHospital.HospitalFacilityCode;
                string hospitalCodeSystem = this.DefaultHospitalCodeSystem;
                string title = string.Empty;
                string suffix = string.Empty;

                if (!string.IsNullOrEmpty(model.SelectedTitleId))
                {
                    title = model.Titles.Where(t => t.Value == model.SelectedTitleId).FirstOrDefault().Value;
                }

                if (!string.IsNullOrEmpty(model.SelectedSuffixId))
                {
                    suffix = model.Suffixes.Where(s => s.Value == model.SelectedSuffixId).FirstOrDefault().Value;
                }

                var response = this.hpiiSearchServiceV2.AddEditLocalProvider(hospitalCode, hospitalCodeSystem, model.Hpii, title, model.FamilyName, model.GivenName, suffix, user);
                this.LoadAddEditErrorMessages(model, response);

                if (!model.Messages.Any(m => m.Level == MessageLevel.Error))
                {
                    AddEditResultViewModel responseVm = new AddEditResultViewModel()
                    {
                        FamilyName = model.FamilyName,
                        GivenName = model.GivenName,
                        Hpii = model.Hpii,
                        Title = (this.hpiiSearchReferenceRepository.GetTitles().Find(t => t.Code == model.SelectedTitleId) != null ? this.hpiiSearchReferenceRepository.GetTitles().Find(t => t.Code == model.SelectedTitleId).Description : null),
                        Suffix = (this.hpiiSearchReferenceRepository.GetSuffixes().Find(s => s.Code == model.SelectedSuffixId) != null ? this.hpiiSearchReferenceRepository.GetSuffixes().Find(s => s.Code == model.SelectedSuffixId).Description : null),
                        CurrentHospital = model.CurrentHospital
                    };
                    responseVm.Messages = model.Messages;
                    return this.View("AddEditResult", responseVm);
                }
                else
                {
                    string errorMessage = "Failed to add new local provider.";

                    // Log details:
                    Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception(string.Format("{0} {1}", errorMessage, response.Messages.AsString())));

                    // Display error message.
                    this.SetAjaxErrorResponseCode();
                    model.Messages.Add(errorMessage, MessageLevel.Error);
                }
            }

            return View("AddProvider", model);
        }

        /// <summary>
        /// Displays the Edit Provider form.
        /// </summary>
        /// <param name="localProviderCode">Local Provider Code.</param>
        /// <returns>View result.</returns>
        [HttpGet]
        public ActionResult EditProvider(string localProviderCode)
        {
            string hospitalCode = this.SessionConfiguration.RepresentingHospital.HospitalFacilityCode;
            string hospitalCodeSystem = this.DefaultHospitalCodeSystem;

            var response = this.hpiiSearchServiceV2.ListLocalProviders(hospitalCode, hospitalCodeSystem, this.GetLocalUser());
            var localProvider = response.Data.Find(lp => lp.LocalProviderCode == localProviderCode);

            var model = new EditProviderViewModel();
            this.LoadAndAddReferenceData(model);

            model.FamilyName = localProvider.FamilyName;
            model.GivenName = localProvider.GivenNames;
            model.Hpii = localProvider.Hpii;
            model.SelectedTitleId = model.Titles.Where(t => t.Text == localProvider.Title).FirstOrDefault() == null ? null : model.Titles.Where(t => t.Text == localProvider.Title).FirstOrDefault().Value;
            model.SelectedSuffixId = model.Suffixes.Where(t => t.Text == localProvider.Suffix).FirstOrDefault() == null ? null : model.Suffixes.Where(t => t.Text == localProvider.Suffix).FirstOrDefault().Value;

            model.LocalProviderIdentifier = localProvider.LocalProviderIdentifiers.Where(lpi => lpi.HospitalCode == hospitalCode && lpi.HospitalCodeSystem == hospitalCodeSystem).FirstOrDefault().PasProviderIdentifier;

            return View("EditProvider", model);
        }

        /// <summary>
        /// Process the result of the Edit Local Provider.
        /// </summary>
        /// <param name="model">View model.</param>
        /// <returns>View result.</returns>
        [HttpPost]
        [ValidateAntiForgeryToken]
        public ActionResult EditProvider(EditProviderViewModel model)
        {
            this.LoadAndAddReferenceData(model);

            if (this.ModelState.IsValid)
            {
                var user = this.GetLocalUser();

                string hospitalCode = this.SessionConfiguration.RepresentingHospital.HospitalFacilityCode;
                string hospitalCodeSystem = this.DefaultHospitalCodeSystem;
                string title = string.Empty;
                string suffix = string.Empty;

                if (!string.IsNullOrEmpty(model.SelectedTitleId))
                {
                    title = model.Titles.Where(t => t.Value == model.SelectedTitleId).FirstOrDefault().Value;
                }

                if (!string.IsNullOrEmpty(model.SelectedSuffixId))
                {
                    suffix = model.Suffixes.Where(s => s.Value == model.SelectedSuffixId).FirstOrDefault().Value;
                }

                var editResponse = this.hpiiSearchServiceV2.AddEditLocalProvider(hospitalCode, hospitalCodeSystem, model.Hpii, title, model.FamilyName, model.GivenName, suffix, user, localProviderCode: model.LocalProviderCode);
                this.LoadAddEditErrorMessages(model, editResponse);

                if (!model.Messages.Any(m => m.Level == MessageLevel.Error))
                {
                    var localProviderIdentifier = new LocalProviderIdentifier();
                    localProviderIdentifier.HospitalCode = this.SessionConfiguration.RepresentingHospital.HospitalFacilityCode;
                    localProviderIdentifier.HospitalCodeSystem = this.DefaultHospitalCodeSystem;
                    localProviderIdentifier.LocalProviderCode = model.LocalProviderCode;
                    localProviderIdentifier.PasProviderIdentifier = model.LocalProviderIdentifier;

                    // Look up the local provider to fill any existing Cis provider identifier.
                    // Note: This code should be removed when the UI gains support for editing the Cis provider identifier.
                    var lookupResponse = this.hpiiSearchServiceV2.ListLocalProviders(hospitalCode, hospitalCodeSystem, this.GetLocalUser());
                    var localProvider = lookupResponse.Data.Find(lp => lp.LocalProviderCode == model.LocalProviderCode);
                    localProviderIdentifier.CisProviderIdentifier = localProvider.LocalProviderIdentifiers.Where(lpi => lpi.HospitalCode == hospitalCode && lpi.HospitalCodeSystem == hospitalCodeSystem).FirstOrDefault().CisProviderIdentifier;

                    var linkResponse = this.hpiiSearchServiceV2.AddReplaceLocalProviderIdentifier(localProviderIdentifier, true, this.GetLocalUser());

                    if (linkResponse.IsSuccessful && lookupResponse.IsSuccessful)
                    {
                        AddEditResultViewModel responseVm = new AddEditResultViewModel()
                        {
                            FamilyName = model.FamilyName,
                            GivenName = model.GivenName,
                            Hpii = model.Hpii,
                            Title = (this.hpiiSearchReferenceRepository.GetTitles().Find(t => t.Code == model.SelectedTitleId) != null ? this.hpiiSearchReferenceRepository.GetTitles().Find(t => t.Code == model.SelectedTitleId).Description : null),
                            Suffix = (this.hpiiSearchReferenceRepository.GetSuffixes().Find(s => s.Code == model.SelectedSuffixId) != null ? this.hpiiSearchReferenceRepository.GetSuffixes().Find(s => s.Code == model.SelectedSuffixId).Description : null),
                            LocalProviderIdentifier = model.LocalProviderIdentifier,
                            CurrentHospital = model.CurrentHospital
                        };
                        responseVm.Messages = model.Messages;
                        return this.View("AddEditResult", responseVm);
                    }
                    else
                    {
                        string errorMessage = "Failed to link local provider.";

                        // Log details:
                        Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception(string.Format("{0} {1} {2}", errorMessage, linkResponse.Messages.AsString(), lookupResponse.Messages.AsString())));

                        // Display error message.
                        this.SetAjaxErrorResponseCode();
                        model.Messages.Add(errorMessage, MessageLevel.Error);
                    }
                }
                else
                {
                    string errorMessage = "Failed to edit local provider.";

                    // Log details:
                    Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception(string.Format("{0} {1}", errorMessage, editResponse.Messages.AsString())));

                    // Display error message.
                    this.SetAjaxErrorResponseCode();
                    model.Messages.Add(errorMessage, MessageLevel.Error);
                }
            }

            return View("EditProvider", model);
        }

        /// <summary>
        /// Process the result of Activate Local Provider.
        /// </summary>
        /// <param name="localProviderCode">Local Provider Code.</param>
        /// <returns>View result.</returns>
        [HttpPost]
        public ActionResult ActivateProvider(string localProviderCode)
        {
            string hospitalCode = this.SessionConfiguration.RepresentingHospital.HospitalFacilityCode;
            string hospitalCodeSystem = this.DefaultHospitalCodeSystem;
            var response = this.hpiiSearchServiceV2.DeactivateLocalProvider(hospitalCode, hospitalCodeSystem, localProviderCode, null, this.GetLocalUser());
            var messages = new ViewMessageList();
            messages.AddRange(ObjectMapper.Map<List<ViewMessage>>(response.Messages));

            return this.PartialView("ViewMessageList", messages);
        }

        /// <summary>
        /// Displays the Deactivate Local Provider modal form.
        /// </summary>
        /// <param name="localProviderCode">Local Provider Code.</param>
        /// <returns>View result.</returns>
        [HttpGet]
        public ActionResult DeactivateProvider(string localProviderCode)
        {
            var model = new DeactivateProviderViewModel();
            model.LocalProviderCode = localProviderCode;

            return this.PartialView("DeactivateProvider", model);
        }

        /// <summary>
        /// Process the result of Deactivate Local Provider.
        /// </summary>
        /// <param name="model">View model.</param>
        /// <returns>View result.</returns>
        [HttpPost]
        public ActionResult DeactivateProvider(DeactivateProviderViewModel model)
        {
            ResponseMessageList messages = null;

            if (this.ModelState.IsValid)
            {
                string hospitalCode = this.SessionConfiguration.RepresentingHospital.HospitalFacilityCode;
                string hospitalCodeSystem = this.DefaultHospitalCodeSystem;

                var response = this.hpiiSearchServiceV2.DeactivateLocalProvider(hospitalCode, hospitalCodeSystem, model.LocalProviderCode, model.DeactivatedDate, this.GetLocalUser());
                messages = response.Messages;

                if (response.IsSuccessful)
                {
                    return RedirectToAction("Providers");
                }
                else
                {
                    // Log error messages to database
                    string errorMessage = "Failed to deactivate local provider.";
                    Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception(string.Format("{0} {1}", errorMessage, response.Messages.AsString())));
                }
            }

            return this.Json(new { Messages = messages, Errors = this.ModelState.ToErrorDictionary() });
        }

        /// <summary>
        /// Displays the Link Provider modal form.
        /// </summary>
        /// <param name="localProviderCode">Local Provider Code.</param>
        /// <returns>View result.</returns>
        [HttpGet]
        public ActionResult LinkProvider(string localProviderCode)
        {
            var model = new LinkProviderViewModel();
            model.LocalProviderCode = localProviderCode;

            return this.PartialView("LinkProvider", model);
        }

        /// <summary>
        /// Process the result of Link Provider.
        /// </summary>
        /// <param name="model">View model.</param>
        /// <returns>View result.</returns>
        [HttpPost]
        public ActionResult LinkProvider(LinkProviderViewModel model)
        {
            ResponseMessageList messages = null;

            if (this.ModelState.IsValid)
            {
                var localProviderIdentifier = new LocalProviderIdentifier();
                localProviderIdentifier.HospitalCode = this.SessionConfiguration.RepresentingHospital.HospitalFacilityCode;
                localProviderIdentifier.HospitalCodeSystem = this.DefaultHospitalCodeSystem;
                localProviderIdentifier.LocalProviderCode = model.LocalProviderCode;
                localProviderIdentifier.PasProviderIdentifier = model.LocalProviderIdentifier;

                var response = this.hpiiSearchServiceV2.AddReplaceLocalProviderIdentifier(localProviderIdentifier, false, this.GetLocalUser());
                messages = response.Messages;

                if (response.IsSuccessful)
                {
                    return RedirectToAction("Providers");
                }
                else
                {
                    // Log error messages to database
                    string errorMessage = "Failed to link local provider.";
                    Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception(string.Format("{0} {1}", errorMessage, response.Messages.AsString())));
                }
            }

            return this.Json(new { Messages = messages, Errors = this.ModelState.ToErrorDictionary() });
        }

        #region Helper Methods

        /// <summary>
        /// Loads reference data into the view model.
        /// </summary>
        /// <param name="searchByIdViewModel">View model.</param>
        /// <returns>The given view model, for fluency.</returns>
        private SearchByIdViewModel LoadAndAddReferenceData(SearchByIdViewModel searchByIdViewModel)
        {
            // Load reference data
            List<HiSex> sexes = this.hiReferenceRepository.GetHiSexes();
            List<HiState> states = this.hiReferenceRepository.GetHiStates();

            // Update ViewModel with reference data
            searchByIdViewModel.Sexes = sexes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByIdViewModel.States = states.ToSelectListItems(s => s.Code, s => s.Description);

            // Return for fluency
            return searchByIdViewModel;
        }

        /// <summary>
        /// Loads reference data into the view model.
        /// </summary>
        /// <param name="searchByDemographicsViewModel">View model.</param>
        /// <returns>The given view model, for fluency.</returns>
        private SearchByDemographicsViewModel LoadAndAddReferenceData(SearchByDemographicsViewModel searchByDemographicsViewModel)
        {
            // Load reference data
            List<HiSex> sexes = this.hiReferenceRepository.GetHiSexes();
            List<HiState> states = this.hiReferenceRepository.GetHiStates();
            List<HiUnitType> unitTypes = this.hiReferenceRepository.GetHiUnitTypes();
            List<HiLevelType> levelTypes = this.hiReferenceRepository.GetHiLevelTypes();
            List<HiStreetType> streetTypes = this.hiReferenceRepository.GetHiStreetTypes();
            List<HiStreetSuffixType> streetSuffixes = this.hiReferenceRepository.GetHiStreetSuffixTypes();
            List<HiPostalDeliveryType> postalDeliveryTypes = this.hiReferenceRepository.GetHiPostalDeliveryTypes();
            List<HiCountry> countries = this.hiReferenceRepository.GetHiCountries();

            // Update ViewModel with reference data
            searchByDemographicsViewModel.Sexes = sexes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.States = states.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.UnitTypes = unitTypes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.LevelTypes = levelTypes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.StreetTypes = streetTypes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.StreetSuffixes = streetSuffixes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.PostalDeliveryTypes = postalDeliveryTypes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.Countries = countries.ToSelectListItems(s => s.Code, s => s.Description);

            return searchByDemographicsViewModel;
        }

        /// <summary>
        /// Loads reference data into the view model.
        /// </summary>
        /// <param name="searchResultViewModel">View model.</param>
        /// <param name="usingAustralianAddress">Whether australian addresses are used.</param>
        /// <param name="usingInternationalAddress">Whether international addresses are used.</param>
        /// <returns>The given view model, for fluency.</returns>
        private SearchResultViewModel LoadAndAddReferenceData(SearchResultViewModel searchResultViewModel, bool usingAustralianAddress, bool usingInternationalAddress)
        {
            // Load reference data
            List<HiSex> sexes = this.hiReferenceRepository.GetHiSexes();
            List<HiState> states = this.hiReferenceRepository.GetHiStates();

            // Update ViewModel with reference data
            searchResultViewModel.Sexes = sexes.ToSelectListItems(s => s.Code, s => s.Description);
            searchResultViewModel.States = states.ToSelectListItems(s => s.Code, s => s.Description);

            if (usingAustralianAddress)
            {
                // Load reference data
                List<HiUnitType> unitTypes = this.hiReferenceRepository.GetHiUnitTypes();
                List<HiLevelType> levelTypes = this.hiReferenceRepository.GetHiLevelTypes();
                List<HiStreetType> streetTypes = this.hiReferenceRepository.GetHiStreetTypes();
                List<HiStreetSuffixType> streetSuffixes = this.hiReferenceRepository.GetHiStreetSuffixTypes();
                List<HiPostalDeliveryType> postalDeliveryTypes = this.hiReferenceRepository.GetHiPostalDeliveryTypes();

                // Update ViewModel with reference data
                searchResultViewModel.UnitTypes = unitTypes.ToSelectListItems(s => s.Code, s => s.Description);
                searchResultViewModel.LevelTypes = levelTypes.ToSelectListItems(s => s.Code, s => s.Description);
                searchResultViewModel.StreetTypes = streetTypes.ToSelectListItems(s => s.Code, s => s.Description);
                searchResultViewModel.StreetSuffixes = streetSuffixes.ToSelectListItems(s => s.Code, s => s.Description);
                searchResultViewModel.PostalDeliveryTypes = postalDeliveryTypes.ToSelectListItems(s => s.Code, s => s.Description);
            }
            if (usingInternationalAddress)
            {
                // Load reference data
                List<HiCountry> countries = this.hiReferenceRepository.GetHiCountries();

                // Update ViewModel with reference data
                searchResultViewModel.Countries = countries.ToSelectListItems(s => s.Code, s => s.Description);
            }

            // Return for fluency
            return searchResultViewModel;
        }

        /// <summary>
        /// Loads reference data into the view model.
        /// </summary>
        /// <param name="model">View model.</param>
        /// <returns>The given view model, for fluency.</returns>
        private AddProviderViewModel LoadAndAddReferenceData(AddProviderViewModel model)
        {
            // Load current hospital.
            model.CurrentHospital = ObjectMapper.Map<HospitalViewModel>(this.SessionConfiguration.RepresentingHospital);

            // Load Titles.
            model.Titles = this.hpiiSearchReferenceRepository.GetTitles().ToSelectListItems(t => t.Code, t => t.Description);

            // Load Suffixes.
            model.Suffixes = this.hpiiSearchReferenceRepository.GetSuffixes().ToSelectListItems(t => t.Code, t => t.Description);

            return model;
        }

        /// <summary>
        /// Loads reference data into the view model.
        /// </summary>
        /// <param name="model">View model.</param>
        /// <returns>The given view model, for fluency.</returns>
        private EditProviderViewModel LoadAndAddReferenceData(EditProviderViewModel model)
        {
            // Load current hospital.
            model.CurrentHospital = ObjectMapper.Map<HospitalViewModel>(this.SessionConfiguration.RepresentingHospital);

            // Load Titles.
            model.Titles = this.hpiiSearchReferenceRepository.GetTitles().ToSelectListItems(t => t.Code, t => t.Description);

            // Load Suffixes.
            model.Suffixes = this.hpiiSearchReferenceRepository.GetSuffixes().ToSelectListItems(t => t.Code, t => t.Description);

            return model;
        }

        /// <summary>
        /// Loads error messages from AddEditLocalProvider response into the view model.
        /// </summary>
        /// <param name="model">The view model.</param>
        /// <param name="response">The response to get error messages from.</param>
        private void LoadAddEditErrorMessages(LocalProviderViewModelBase model, ServiceResponse<AddEditLocalProviderResponse> response)
        {
            // Add service response errors
            response.Messages.ForEach(m => model.Messages.Add(m.Description, m.Level));
            
            // Add AddEditLocalProvider response messages (if they exist)
            if (response.Data != null)
            {
                response.Data.Messages.ForEach(m => model.Messages.Add(m.Description, ObjectMapper.Map<MessageLevel>(m.Level)));
            }
        }

        #endregion Helper Methods

        #endregion Methods
    }
}