﻿using System;
using System.Text.RegularExpressions;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.Web.UI.ViewModels.DataIntegrity
{
    /// <summary>
    /// View model for a particular patient.
    /// </summary>
    public class PatientViewModel
    {
        #region Properties

        /// <summary>
        /// Gets or sets the name of the patient.
        /// </summary>
        public string Name { get; set; }

        /// <summary>
        /// Gets or sets the sex code of the patient.
        /// </summary>
        public SexEnumerator SexCode { get; set; }

        /// <summary>
        /// Gets the sex of the patient, translated from SexCode to a string.
        /// </summary>
        public string Sex
        {
            get
            {
                switch (this.SexCode)
                {
                    case SexEnumerator.IntersexOrIndeterminate: return "Intersex or Indeterminate";
                    case SexEnumerator.Male: return "Male";
                    case SexEnumerator.Female: return "Female";
                    case SexEnumerator.NotStatedOrInadequatelyDescribed: return "Not Stated or Inadequately Described";
                    default: return string.Empty;
                }
            }
        }

        /// <summary>
        /// Gets or sets the date of birth of the patient.
        /// </summary>
        public DateTime DateOfBirth { get; set; }

        /// <summary>
        /// Gets or sets the name of the hospital.
        /// </summary>
        public string HospitalName { get; set; }

        /// <summary>
        /// Gets or sets the ward of the patient's most recent episode.
        /// </summary>
        public string Ward { get; set; }

        /// <summary>
        /// Gets or sets the room of the patient's most recent episode.
        /// </summary>
        public string Room { get; set; }

        /// <summary>
        /// Gets or sets the bed of the patient's most recent episode.
        /// </summary>
        public string Bed { get; set; }

        /// <summary>
        /// Gets the location (ward:room:bed) of the patient's most recent episode.
        /// </summary>
        public string Location
        {
            get
            {
                return string.Format("{0}:{1}:{2}", this.Ward, this.Room, this.Bed);
            }
        }

        /// <summary>
        /// Gets or sets the MRN of the patient.
        /// </summary>
        public string MRN { get; set; }

        /// <summary>
        /// Gets or sets the Medicare IRN.
        /// </summary>
        public string MedicareIrn { get; set; }

        /// <summary>
        /// Gets or sets the Medicare Card Number.
        /// </summary>
        public string MedicareCardNumber { get; set; }

        /// <summary>
        /// Gets or sets the DVA File Number.
        /// </summary>
        public string DvaFileNumber { get; set; }

        /// <summary>
        /// Gets the Medicare Card Number, formatted as the style on the card.
        /// </summary>
        public string FormattedMedicareCardNumber
        {
            get
            {
                if (this.MedicareCardNumber != null)
                {
                    string firstPart = Regex.Replace(this.MedicareCardNumber, @"(\d{4})(\d{5})(\d{1})", @"$1 $2 $3");
                    string secondPart = this.MedicareIrn == null ? string.Empty : string.Format("({0})", this.MedicareIrn);
                    return string.Format("{0} {1}", firstPart, secondPart);
                }
                else
                {
                    return null;
                }
            }
        }

        #endregion Properties
    }
}