using System;
using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Web.Mvc;
using HIPS.Web.UI.ViewModels.Shared;

namespace HIPS.Web.UI.ViewModels.HpiiSearch
{
    /// <summary>
    ///     View Model for the HpiiSearchController.SearchById action.
    /// </summary>
    public class SearchByIdViewModel : IValidatableObject
    {
        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="SearchByIdViewModel"/> class.
        /// </summary>
        /// <remarks>
        ///     Called automatically by MVC Model Binding.
        ///     Should initialise collections and complex objects to default/empty values.
        /// </remarks>
        public SearchByIdViewModel()
        {
            this.ResponseMessages = new ResponseMessages();
            this.Sexes = Enumerable.Empty<SelectListItem>();
            this.States = Enumerable.Empty<SelectListItem>();
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Gets or sets HPI-O
        /// </summary>
        public string Hpio { get; set; }

        /// <summary>
        /// Gets or sets HPI-I
        /// </summary>
        [StringLength(16, MinimumLength = 16, ErrorMessage = "HPI-I must be 16 characters long.")]
        [Display(Name = "HPI-I")]
        public string Hpii { get; set; }

        /// <summary>
        /// Gets or sets Registration Id.
        /// </summary>
        [StringLength(20, ErrorMessage = "Registration ID must be less than 20 characters long.")]
        [Display(Name = "Registration ID (AHPRA)")]
        public string RegistrationId { get; set; }

        /// <summary>
        /// Gets or sets Family name of the patient.
        /// </summary>
        [Required]
        [StringLength(40, ErrorMessage = "Family Name must be less than 40 characters long.")]
        [Display(Name = "Family Name")]
        public string FamilyName { get; set; }

        /// <summary>
        /// Gets or sets Given name of the patient.
        /// </summary>
        [StringLength(100, ErrorMessage = "Given Name(s) must be less than 100 characters long.")]
        [Display(Name = "Given Name(s)")]
        public string GivenNames { get; set; }

        /// <summary>
        /// Gets or sets Sex code.
        /// </summary>
        [Display(Name = "Sex")]
        public string SexCode { get; set; }

        /// <summary>
        /// Gets or sets Date of birth of the patient.
        /// </summary>
        [Display(Name = "Date of Birth")]
        [DisplayFormat(DataFormatString = "dd/MM/yyyy")]
        public DateTime? DateOfBirth { get; set; }

        /// <summary>
        /// Gets or sets State code.
        /// </summary>
        [Display(Name = "State")]
        public string StateCode { get; set; }

        /// <summary>
        /// Gets or sets Post code.
        /// </summary>
        [Display(Name = "Postcode")]
        [StringLength(4, ErrorMessage = "Postcode must be less than 4 characters long.")]
        public string Postcode { get; set; }

        /// <summary>
        /// Gets response messages.
        /// </summary>
        public ResponseMessages ResponseMessages { get; internal set; }

        #endregion Properties

        #region Data Source Properties

        /// <summary>
        /// Gets Sex drop down.
        /// </summary>
        public IEnumerable<SelectListItem> Sexes { get; internal set; }

        /// <summary>
        /// Gets States drop down.
        /// </summary>
        public IEnumerable<SelectListItem> States { get; internal set; }

        #endregion Data Source Properties

        /// <summary>
        ///     Determines whether the specified object is valid.
        /// </summary>
        /// <returns>
        ///     A collection that holds failed-validation information.
        /// </returns>
        /// <param name="validationContext">The validation context.</param>
        /// <remarks>
        ///     Executed server-side after all annotation validators are successful.
        /// </remarks>
        public IEnumerable<ValidationResult> Validate(ValidationContext validationContext)
        {
            var errors = new List<ValidationResult>();

            if (string.IsNullOrWhiteSpace(this.Hpii) && string.IsNullOrWhiteSpace(this.RegistrationId))
            {
                errors.Add(new ValidationResult("Either the HPI-I Number or the Registration ID are required to complete an identifier search."));
            }

            return errors;
        }
    }
}