﻿using System;
using System.Configuration;
using System.Data.SqlClient;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.Common.PcehrDataStore.DataAccess
{
    public class BaseDl : DataAccessBase
    {
        #region Private Members

        private SqlConnection connection;
        private SqlCommand command;

        #endregion Private Members

        public BaseDl(UserDetails user)
            : base(user)
        {
        }

        public BaseDl()
            : base()
        {
        }

        #region Properties

        /// <summary>
        /// Gets the SQL connection for the HIPS database (PCEHR Data Store).
        /// </summary>
        protected SqlConnection Connection
        {
            get
            {
                if (connection == null)
                {
                    try
                    {
                        string connectionString = ConfigurationManager.ConnectionStrings["PcehrDataStoreConnectionString"].ToString();
                        connection = new SqlConnection(connectionString);
                    }
                    catch (Exception ex)
                    {
                        EventLogger.WriteLog(ConstantsResource.ErrorMessageSqlConnectionError, ex, User, LogMessage.HIPS_MESSAGE_001);
                    }
                }
                return connection;
            }
        }

        /// <summary>
        /// Gets the command.
        /// </summary>
        protected new SqlCommand Command
        {
            get
            {
                if (command == null)
                {
                    try
                    {
                        command = new SqlCommand();
                        command.Connection = Connection;
                    }
                    catch (Exception ex)
                    {
                        EventLogger.WriteLog(ConstantsResource.ErrorMessageSqlConnectionError, ex, User, LogMessage.HIPS_MESSAGE_002);
                    }
                }
                if (command.Connection.State == System.Data.ConnectionState.Closed)
                {
                    command.Connection.Open();
                }
                return command;
            }
        }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Gets the SQL command.
        /// </summary>
        /// <param name="storedProcedure">The stored procedure.</param>
        /// <returns></returns>
        protected SqlCommand GetSqlCommand(string storedProcedure)
        {
            SqlCommand result = Command;
            result.CommandText = storedProcedure;
            result.CommandType = System.Data.CommandType.StoredProcedure;
            result.Parameters.Clear();
            return result;
        }

        /// <summary>
        /// Gets the SQL command.
        /// </summary>
        /// <param name="storedProcedure">The stored procedure.</param>
        /// <param name="transaction">The transaction.</param>
        /// <returns></returns>
        protected SqlCommand GetSqlCommand(string storedProcedure, SqlTransaction transaction)
        {
            if (transaction != null)
            {
                SqlCommand result = new SqlCommand(storedProcedure, transaction.Connection, transaction);
                result.CommandType = System.Data.CommandType.StoredProcedure;
                result.Parameters.Clear();
                return result;
            }
            else
            {
                return GetSqlCommand(storedProcedure);
            }
        }

        /// <summary>
        /// Gets the SQL text command.
        /// </summary>
        /// <param name="sql">The SQL.</param>
        /// <returns></returns>
        protected SqlCommand GetSqlTextCommand(string sql)
        {
            SqlCommand result = Command;
            result.CommandText = sql;
            result.CommandType = System.Data.CommandType.Text;
            result.Parameters.Clear();
            return result;
        }

        /// <summary>
        /// Gets the minimum date startTime so SQL access.
        /// </summary>
        protected DateTime MinimumDateTime
        {
            get
            {
                return new DateTime(1900, 1, 1);
            }
        }

        public bool IsDatabaseAvailable
        {
            get
            {
                SqlConnection connection = Connection;
                if (connection.State == System.Data.ConnectionState.Open)
                {
                    return true;
                }
                if (connection.State == System.Data.ConnectionState.Closed)
                {
                    try
                    {
                        connection.Open();
                        connection.Close();
                        return true;
                    }
                    catch
                    {
                        return false;
                    }
                }
                return false;
            }
        }

        /// <summary>
        /// Adds the string parameter.
        /// </summary>
        /// <param name="parameterName">Name of the parameter.</param>
        /// <param name="value">The value.</param>
        /// <param name="command">The command.</param>
        protected static void AddStringParameter(string parameterName, string value, SqlCommand command)
        {
            if (!string.IsNullOrEmpty(value))
            {
                command.Parameters.Add(new SqlParameter(parameterName, value));
            }
        }

        /// <summary>
        /// Adds the int parameter.
        /// </summary>
        /// <param name="parameterName">Name of the parameter.</param>
        /// <param name="value">The value.</param>
        /// <param name="command">The command.</param>
        protected static void AddIntParameter(string parameterName, int? value, SqlCommand command)
        {
            if (value != null && value.HasValue)
            {
                command.Parameters.Add(new SqlParameter(parameterName, value.Value));
            }
        }

        /// <summary>
        /// Adds the bool parameter.
        /// </summary>
        /// <param name="parameterName">Name of the parameter.</param>
        /// <param name="value">The value.</param>
        /// <param name="command">The command.</param>
        protected static void AddBoolParameter(string parameterName, bool? value, SqlCommand command)
        {
            if (value != null && value.HasValue)
            {
                command.Parameters.Add(new SqlParameter(parameterName, value.Value));
            }
        }

        /// <summary>
        /// Adds the date parameter.
        /// </summary>
        /// <param name="parameterName">Name of the parameter.</param>
        /// <param name="value">The value.</param>
        /// <param name="command">The command.</param>
        protected static void AddDateParameter(string parameterName, DateTime? value, SqlCommand command)
        {
            if (value != null && value.HasValue)
            {
                command.Parameters.Add(new SqlParameter(parameterName, value.Value));
            }
        }

        #endregion Methods
    }
}