﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Data;
using System.Data.Common;
using System.Data.SqlClient;
using System.Diagnostics;
using System.Globalization;
using System.Linq;
using System.Reflection;
using HIPS.Base.DataAccess;
using HIPS.Base.Schemas;
using HIPS.Base.Schemas.Enumerators;
using HIPS.CommonSchemas;

namespace HIPS.Common.PcehrDataStore.DataAccess
{
    public abstract class DataAccessBase
    {
        #region Global variables

        private const Int32 mSqlConcurrencyError = 50001;
        private const Int32 mSqlUniqueConstraintError = 2627;

        /// <summary>
        /// List of Write operations
        /// </summary>
        protected enum UpdateType
        {
            /// <summary>
            /// Update operation
            /// </summary>
            Update,

            /// <summary>
            /// Insert operation
            /// </summary>
            Insert,

            /// <summary>
            /// Delete operation
            /// </summary>
            Delete
        }

        #endregion Global variables

        #region Properties

        /// <summary>
        /// Gets or sets the database connection.
        /// </summary>
        /// <value>
        /// The database connection.
        /// </value>
        protected SqlConnection DatabaseConnection { get; set; }

        /// <summary>
        /// Gets or sets the database command.
        /// </summary>
        protected SqlCommand Command { get; set; }

        /// <summary>
        /// Gets or sets user details so that UserModified columns are populated from the information given by the caller of a web service.
        /// </summary>
        protected UserDetails User { get; set; }

        #endregion Properties

        #region Methods

        public DataAccessBase()
        {
        }

        public DataAccessBase(UserDetails user)
        {
            User = user;
        }

        /// <summary>
        /// Populates the business object.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="reader">The reader.</param>
        /// <param name="businessObject">The business object.</param>
        /// <returns></returns>
        protected static bool PopulateBusinessObject<T>(SqlDataReader reader, T businessObject)
        {
            bool returnValue = false;

            if (reader.HasRows)
            {
                reader.Read();
                PopulateObject<T>(reader, businessObject);
                returnValue = true;
            }
            reader.Close();
            return returnValue;
        }

        /// <summary>
        /// Populates the business array list.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="reader">The reader.</param>
        /// <returns></returns>
        protected static ArrayList PopulateBusinessArrayList<T>(SqlDataReader reader) where T : new()
        {
            ArrayList results = new ArrayList();
            while (reader.Read())
            {
                T businessObject = new T();
                PopulateObject<T>(reader, businessObject);
                results.Add(businessObject);
            }
            reader.Close();
            return results;
        }

        /// <summary>
        /// Gets the populated business list.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="reader">The reader.</param>
        /// <returns></returns>
        protected static List<T> GetPopulatedBusinessList<T>(SqlDataReader reader) where T : new()
        {
            List<T> results = new List<T>();
            while (reader.Read())
            {
                T businessObject = new T();
                PopulateObject<T>(reader, businessObject);
                results.Add(businessObject);
            }
            reader.Close();
            return results;
        }

        /// <summary>
        /// Returns an Observable Collection of Generic objects from a data reader
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="reader">The reader.</param>
        /// <returns></returns>
        protected static ObservableCollection<T> GetPopulatedBusinessObservableCollection<T>(SqlDataReader reader) where T : new()
        {
            ObservableCollection<T> results = new ObservableCollection<T>();
            while (reader.Read())
            {
                T businessObject = new T();
                PopulateObject<T>(reader, businessObject);
                results.Add(businessObject);
            }
            reader.Close();
            return results;
        }

        protected static void PopulateObject<T>(SqlDataReader reader, T businessObject)
        {
            try
            {
                Dictionary<string, int> columnOrdinals = new Dictionary<string, int>();

                for (int count = 0; count < reader.FieldCount; count++)
                {
                    string columnname = reader.GetName(count);
                    if (!columnOrdinals.ContainsKey(columnname))
                    {
                        columnOrdinals.Add(columnname, count);
                    }
                    else
                    {
                        Debug.WriteLine(columnname);
                    }
                }
                PropertyInfo info = null;
                PropertyInfo[] properties = businessObject.GetType().GetProperties();

                foreach (KeyValuePair<string, int> column in columnOrdinals)
                {
                    // Find a Property with the same name as the Column
                    info = null;
                    for (int i = 0; i < properties.Length; i++)
                    {
                        if (string.Compare(properties[i].Name, column.Key, StringComparison.OrdinalIgnoreCase) == 0)
                        {
                            info = properties[i];
                            break;
                        }
                    }

                    // If one exists and it can be populated pass the data in
                    if (info != null && info.CanWrite)
                    {
                        if (reader[column.Value] == DBNull.Value)
                        {
                            info.SetValue(businessObject, null, null);
                        }
                        else
                        {
                            info.SetValue(businessObject, reader[column.Value], null);
                        }
                    }
                }
                try
                {
                    info = properties.FirstOrDefault(result => result.Name.ToUpper() == "ISDIRTY");
                    if (info != null)
                    {
                        info.SetValue(businessObject, false, null);
                    }
                }
                catch { }
            }
            catch (Exception ex)
            {
                throw new Exception("Failed to populate the Object", ex);
            }
        }

        /// <summary>
        /// Populates the parameters.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="command">The command.</param>
        /// <param name="businessObject">The business object.</param>
        /// <param name="typeOfUpdate">The identifier of update.</param>
        protected void PopulateParameters<T>(SqlCommand command, T businessObject, UpdateType typeOfUpdate)
        {
            foreach (PropertyInfo info in businessObject.GetType().GetProperties())
            {
                object[] attributes = info.GetCustomAttributes(typeof(DataBaseInfoAttributes), false);
                foreach (DataBaseInfoAttributes attribute in attributes)
                {
                    if (attribute is DataBaseInfoAttributes)
                    {
                        bool deleteParameter = false;
                        bool updateParameter = false;
                        bool insertParameter = false;

                        switch (attribute.ColumnType)
                        {
                            case DatabaseColumnType.ApplicationGeneratedKey:
                                deleteParameter = true;
                                updateParameter = true;
                                insertParameter = true;
                                break;

                            case DatabaseColumnType.AutoGeneratedKey:
                                deleteParameter = true;
                                updateParameter = true;
                                insertParameter = false;
                                break;

                            case DatabaseColumnType.Data:
                                deleteParameter = false;
                                updateParameter = true;
                                insertParameter = true;
                                break;

                            case DatabaseColumnType.UpdateData:
                                deleteParameter = false;
                                updateParameter = true;
                                insertParameter = false;
                                break;

                            case DatabaseColumnType.UpdateDeleteData:
                                deleteParameter = true;
                                updateParameter = true;
                                insertParameter = false;
                                break;
                        }

                        bool addParameter = false;

                        switch (typeOfUpdate)
                        {
                            // Deleting, so only want keys of any identifier
                            case UpdateType.Delete:
                                addParameter = deleteParameter;
                                break;

                            // Inserting a new object, don't want DB Auto Generated keys
                            case UpdateType.Insert:
                                addParameter = insertParameter;
                                break;

                            case UpdateType.Update:
                                addParameter = updateParameter;
                                break;
                        }

                        if (addParameter)
                        {
                            this.AddParameter<T>(command, info, info.GetValue(businessObject, null));
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Returns the string that is stored as the UserModified column in the database
        /// </summary>
        /// <returns></returns>
        protected string GetUserModified()
        {
            string value = string.Empty;
            if (User != null)
            {
                switch (User.Role)
                {
                    case UserRole.AuthorisedEmployee:
                        value = User.AuthorisedEmployeeUserId;
                        if (value == null)
                        {
                            value = "AEUID";
                        }
                        break;

                    case UserRole.InteractiveUser:
                        value = string.Format(@"{0}/{1}", User.Domain, User.Login);
                        break;

                    case UserRole.ProviderIndividual:
                        value = User.HpiI;
                        break;
                }
            }
            else if (System.Threading.Thread.CurrentPrincipal.Identity != null)
            {
                if (!string.IsNullOrEmpty(System.Threading.Thread.CurrentPrincipal.Identity.Name))
                {
                    value = System.Threading.Thread.CurrentPrincipal.Identity.Name;
                }
                else
                {
                    value = string.Format(@"{0}/{1}", Environment.UserDomainName, Environment.UserName);
                }
            }
            return value;
        }

        /// <summary>
        /// Adds a Parameter to the DB Command based on the business object properties
        /// </summary>
        /// <typeparam name="T">Business object identifier the method applies to</typeparam>
        /// <param name="command">The command.</param>
        /// <param name="info">The property.</param>
        /// <param name="value">The value held by the property.</param>
        private void AddParameter<T>(DbCommand command, PropertyInfo info, object value)
        {
            if (info.Name == "UserModified")
            {
                value = GetUserModified();
            }

            command.Parameters.Add(new SqlParameter("@" + info.Name, value));
        }

        /// <summary>
        /// Inserts the specified stored procedure.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="storedProcedure">The stored procedure.</param>
        /// <param name="businessObject">The business object.</param>
        /// <param name="command">The command.</param>
        /// <returns></returns>
        protected bool Insert<T>(T businessObject, SqlCommand command)
        {
            this.PopulateParameters<T>(command, businessObject, UpdateType.Insert);
            return this.ExecuteCommand<T>(command, businessObject);
        }

        /// <summary>
        /// Updates the specified stored procedure.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="storedProcedure">The stored procedure.</param>
        /// <param name="businessObject">The business object.</param>
        /// <param name="command">The command.</param>
        /// <returns></returns>
        protected bool Update<T>(T businessObject, SqlCommand command)
        {
            this.PopulateParameters<T>(command, businessObject, UpdateType.Update);
            return this.ExecuteCommand<T>(command, businessObject);
        }

        /// <summary>
        /// Deletes the specified stored procedure.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="storedProcedure">The stored procedure.</param>
        /// <param name="businessObject">The business object.</param>
        /// <param name="command">The command.</param>
        /// <returns></returns>
        protected bool Delete<T>(T businessObject, SqlCommand command)
        {
            this.PopulateParameters<T>(command, businessObject, UpdateType.Delete);
            return this.ExecuteNonQuery(command);
        }

        /// <summary>
        /// Executes the command.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="dbCommand">The db command.</param>
        /// <param name="businessObject">The business object.</param>
        /// <returns></returns>
        protected bool ExecuteCommand<T>(SqlCommand dbCommand, T businessObject)
        {
            bool instanceLoaded = false;

            try
            {
                OpenConnection(dbCommand);
                using (SqlDataReader reader = dbCommand.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        PopulateObject<T>(reader, businessObject);
                        instanceLoaded = true;
                    }
                }
            }
            catch (SqlException sqlException)
            {
                if (sqlException.Number == mSqlConcurrencyError)
                {
                    ConcurrencyCheckFailException exception = new ConcurrencyCheckFailException(ErrorMessage(sqlException.Number, this.GetType().BaseType.Name + "."), sqlException);
                    throw exception;
                }
                else
                {
                    throw;
                }
            }
            finally
            {
                if (dbCommand.Transaction == null)
                {
                    dbCommand.Connection.Close();
                }
            }

            return instanceLoaded;
        }

        private static void OpenConnection(SqlCommand dbCommand)
        {
            if (dbCommand.Connection.State == ConnectionState.Closed)
            {
                dbCommand.Connection.Open();
            }
        }

        /// <summary>
        /// Executes the non query.
        /// </summary>
        /// <param name="dbCommand">The db command.</param>
        /// <returns></returns>
        protected bool ExecuteNonQuery(SqlCommand dbCommand)
        {
            try
            {
                return System.Convert.ToBoolean(dbCommand.ExecuteNonQuery());
            }
            catch (SqlException sqlException)
            {
                if (sqlException.Number == mSqlConcurrencyError)
                {
                    ConcurrencyCheckFailException exception = new ConcurrencyCheckFailException(ErrorMessage(sqlException.Number, this.GetType().BaseType.Name + "."), sqlException);
                    throw exception;
                }
                else
                {
                    throw;
                }
            }
            finally
            {
                if (dbCommand.Transaction == null)
                {
                    dbCommand.Connection.Close();
                }
            }
        }

        /// <summary>
        /// Errors the message.
        /// </summary>
        /// <param name="ErrorNumber">The error number.</param>
        /// <param name="DefaultMessage">The default message.</param>
        /// <returns></returns>
        protected internal static string ErrorMessage(Int32 ErrorNumber, string DefaultMessage)
        {
            switch (ErrorNumber)
            {
                case 547:
                    return "This decord can't be deleted, it is referenced by another record.";
                case mSqlUniqueConstraintError:
                case 2601:
                    return "You can't add a duplicate record";
                case mSqlConcurrencyError:
                    return "This record has already been updated.";
                default:
                    return ErrorNumber.ToString(CultureInfo.CurrentCulture) + " : " + DefaultMessage;
            }
        }

        #endregion Methods
    }
}