﻿using System.Collections.Generic;
using System.Linq;
using HIPS.CommonBusinessLogic.Singleton;

namespace HIPS.CommonBusinessLogic
{
    /// <summary>
    /// This class holds the shared common functionality for Medicare.
    /// </summary>
    public static class Medicare
    {
        #region Fields

        private const int CARD_LENGTH = 10;
        private const int MIN_FIRST_DIGIT = 2;
        private const int MAX_FIRST_DIGIT = 6;
        private static List<string> medicareExclusions;

        #endregion Fields

        #region Properties

        /// <summary>
        /// The list of medicare numbers that are considered invalid and not used for searching on Medicare.
        /// This is populated from the database table hips.MedicareExclusion.
        /// </summary>
        private static List<string> MedicareExclusions
        {
            get
            {
                if (medicareExclusions == null)
                {
                    medicareExclusions = (from exclusion in ListSingleton.Instance.AllMedicareExclusions
                                          select exclusion.MedicareNumber).ToList();
                }
                return medicareExclusions;
            }
        }

        #endregion Properties

        /// <summary>
        /// Validates the specified Medicare card number is in a valid format.
        /// </summary>
        /// <param name="cardNumber">The card number.</param>
        /// <returns>
        /// If true is returned then the Medicare Number has passed the following format rules:
        ///     It has 10 digits
        ///     The first digit is 2,3,4,5 or 6.
        ///     Digit 9 is a valid check digit:
        ///         a) Calculate the sum of: ((digit 1) + (digit 2 * 3) + (digit 3 * 7) + (digit 4 * 9) + (digit 5) + (digit 6 * 3) + (digit 7 * 7) + (digit 8 * 9))
        ///         b) Remainder of Mod 10 of this calculated number is the check digit.
        ///
        /// If false is returned then this will be treated as an invalid Medicare Number and should be logged in the IhiLookupAlter table as such.
        ///
        /// If null is returned then the Medicare Number was noted as a known number used to ignore the Medicare Check:
        ///     These are specified as unknown or ineligible and thus should not be logged in the IhiLookupAlter as an invalid Medicare Card Number
        /// </returns>
        public static bool? Validate(string cardNumber)
        {
            if (string.IsNullOrEmpty(cardNumber) || MedicareExclusions.Contains(cardNumber))
            {
                return null;
            }
            string validCardNumber = cardNumber.Trim();
            if (validCardNumber.Length != CARD_LENGTH)
            {
                return false;
            }

            int[] digit = new int[CARD_LENGTH + 1];
            for (int i = 1; i <= CARD_LENGTH; i++)
            {
                digit[i] = validCardNumber[i - 1] - '0';

                if (digit[i] < 0 || digit[i] > 9)
                {
                    return false;
                }
            }

            if (digit[1] < MIN_FIRST_DIGIT || digit[1] > MAX_FIRST_DIGIT)
            {
                return false;
            }

            int checkSum = (digit[1] + (digit[2] * 3) + (digit[3] * 7) + (digit[4] * 9) + digit[5] + (digit[6] * 3) + (digit[7] * 7) + (digit[8] * 9));
            int remainder = checkSum % 10;
            bool result = (remainder == digit[9]);
            return result;
        }

        /// <summary>
        /// Determines whether the Medicare card number has changed sufficiently
        /// to warrant searching for the IHI again.
        ///
        /// The logic in this method considers that when only the issue number
        /// (10th digit) is changing, there's no need to validate the IHI.
        /// </summary>
        /// <param name="oldMedicareNumber">The first medicare card number</param>
        /// <param name="newMedicareNumber">The second medicare card number</param>
        /// <returns>True if the change warrants searching for the IHI again.</returns>
        public static bool ChangeNeedsIhiValidation(string oldMedicareNumber, string newMedicareNumber)
        {
            if (string.IsNullOrEmpty(oldMedicareNumber) && string.IsNullOrEmpty(newMedicareNumber))
            {
                // Both are empty - no point in revalidating.
                return false;
            }
            else if (string.IsNullOrEmpty(oldMedicareNumber) || string.IsNullOrEmpty(newMedicareNumber))
            {
                // One is empty and the other is not - we should revalidate.
                return true;
            }
            else if (oldMedicareNumber.Length < 9 || newMedicareNumber.Length < 9)
            {
                // At least one of them is under 9 characters long - we should revalidate if they are different at all.
                return oldMedicareNumber != newMedicareNumber;
            }
            else
            {
                // They are both at least 9 characters long - we should revalidate if they differ in the first 9 digits.
                return oldMedicareNumber.Substring(0, 9) != newMedicareNumber.Substring(0, 9);
            }
        }
    }
}