﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using HIPS.Configuration;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;

namespace HIPS.CommonBusinessLogic.Singleton
{
    public class ListSingleton
    {
        #region Private Members

        private static volatile ListSingleton instance;
        private static object syncRoot = new Object();

        private DateTime? lastUpdated;
        private List<Hospital> allHospitals;
        private List<Title> allTitles;
        private List<Suffix> allSuffixes;
        private List<Sex> allSexes;
        private List<Country> allCountries;
        private List<State> allStates;
        private List<EpisodeType> allEpisodeTypes;
        private List<CodeSystem> allCodeSystems;
        private List<HospitalCode> allHospitalCodes;
        private List<AddressType> allAddressTypes;
        private List<EpisodeLifecycle> allEpisodeLifecycles;
        private List<DocumentType> allDocumentTypes;
        private List<DocumentFormat> allDocumentFormats;
        private List<MedicareExclusion> allMedicareExclusions;
        private List<HealthProviderOrganisation> allHealthProvdierOrganisations;

        #endregion Private Members

        #region Private Constructor

        private ListSingleton()
        {
        }

        /// <summary>
        /// Gets the instance of the List Singleton.
        /// </summary>
        public static ListSingleton Instance
        {
            get
            {
                if (instance == null)
                {
                    lock (syncRoot)
                    {
                        if (instance == null)
                            instance = new ListSingleton();
                    }
                }
                instance.Refresh();
                return instance;
            }
        }

        #endregion Private Constructor

        #region Properties

        /// <summary>
        /// Gets all hospitals.
        /// </summary>
        public ReadOnlyCollection<Hospital> AllHospitals
        {
            get { return allHospitals.AsReadOnly(); }
        }

        /// <summary>
        /// Gets all titles.
        /// </summary>
        public ReadOnlyCollection<Title> AllTitles
        {
            get
            {
                return allTitles.AsReadOnly();
            }
        }

        /// <summary>
        /// Gets all suffixes.
        /// </summary>
        public ReadOnlyCollection<Suffix> AllSuffixes
        {
            get
            {
                return allSuffixes.AsReadOnly();
            }
        }

        /// <summary>
        /// Gets all sexes.
        /// </summary>
        public ReadOnlyCollection<Sex> AllSexes
        {
            get
            {
                return allSexes.AsReadOnly();
            }
        }

        /// <summary>
        /// Gets all countries.
        /// </summary>
        public ReadOnlyCollection<Country> AllCountries
        {
            get
            {
                return allCountries.AsReadOnly();
            }
        }

        /// <summary>
        /// Gets all states.
        /// </summary>
        public ReadOnlyCollection<State> AllStates
        {
            get
            {
                return allStates.AsReadOnly();
            }
        }

        /// <summary>
        /// Gets all episode types.
        /// </summary>
        public ReadOnlyCollection<EpisodeType> AllEpisodeTypes
        {
            get
            {
                return allEpisodeTypes.AsReadOnly();
            }
        }

        /// <summary>
        /// Gets all namespaces.
        /// </summary>
        public ReadOnlyCollection<CodeSystem> AllCodeSystems
        {
            get
            {
                return allCodeSystems.AsReadOnly();
            }
        }

        /// <summary>
        /// Gets all Hospital Codes
        /// </summary>
        public ReadOnlyCollection<HospitalCode> AllHospitalCodes
        {
            get
            {
                return allHospitalCodes.AsReadOnly();
            }
        }

        /// <summary>
        /// Gets all Episode LifeCycles
        /// </summary>
        public ReadOnlyCollection<EpisodeLifecycle> AllEpisodeLifecycles
        {
            get
            {
                return allEpisodeLifecycles.AsReadOnly();
            }
        }

        /// <summary>
        /// Gets all Document Types
        /// </summary>
        public ReadOnlyCollection<DocumentType> AllDocumentTypes
        {
            get
            {
                return allDocumentTypes.AsReadOnly();
            }
        }

        /// <summary>
        /// Gets all Document Formats
        /// </summary>
        public ReadOnlyCollection<DocumentFormat> AllDocumentFormats
        {
            get
            {
                return allDocumentFormats.AsReadOnly();
            }
        }

        /// <summary>
        /// Gets all address types.
        /// </summary>
        public ReadOnlyCollection<AddressType> AllAddressTypes
        {
            get
            {
                return allAddressTypes.AsReadOnly();
            }
        }

        /// <summary>
        /// Gets all address types.
        /// </summary>
        public ReadOnlyCollection<MedicareExclusion> AllMedicareExclusions
        {
            get
            {
                return allMedicareExclusions.AsReadOnly();
            }
        }

        /// <summary>
        /// Gets all Health Provider Organisations.
        /// </summary>
        public ReadOnlyCollection<HealthProviderOrganisation> AllHealthProviderOrganisations
        {
            get
            {
                return allHealthProvdierOrganisations.AsReadOnly();
            }
        }

        #endregion Properties

        #region Private Methods

        /// <summary>
        /// Refreshes this instance.
        /// </summary>
        public void Refresh(bool force = false)
        {
            if (force || (lastUpdated == null || lastUpdated.Value.AddSeconds(Settings.Instance.LookupRefreshSeconds) < DateTime.Now))
            {
                allHospitals = new HospitalDl().GetAll();
                allTitles = new TitleDl().GetAll();
                allSuffixes = new SuffixDl().GetAll();
                allSexes = new SexDl().GetAll();
                allCountries = new CountryDl().GetAll();
                allStates = new StateDl().GetAll();
                allEpisodeTypes = new EpisodeTypeDl().GetAll();
                allCodeSystems = new CodeSystemDl().GetAll();
                allHospitalCodes = new HospitalCodeDl().GetAll(null);
                allAddressTypes = new AddressTypeDl().GetAll();
                allEpisodeLifecycles = new EpisodeLifecycleDl().GetAll();
                allDocumentTypes = new DocumentTypeDl().GetAll();
                allDocumentFormats = new DocumentFormatDl().GetAll();
                allMedicareExclusions = new MedicareExclusionDl().GetAll();
                allHealthProvdierOrganisations = new HealthProviderOrganisationDl().GetAll();
                lastUpdated = DateTime.Now;
            }
        }

        #endregion Private Methods
    }
}