﻿using System;
using System.Runtime.Serialization;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.CommonSchemas.PatientIdentifier
{
    /// <summary>
    /// This Patient Identifier is used for the IHI
    /// </summary>
    [Serializable]
    [DataContract]
    public class ValidatedIhi : PatientIdentifierBase
    {
        #region Private Members

        private string ihi;
        private IhiStatus ihiStatus;
        private IhiRecordStatus ihiRecordStatus;
        private DateTime ihiLastValidated;
        private string familyName;
        private string givenName;
        private DateTime dateOfBirth;
        private SexEnumerator sex;

        #endregion Private Members

        #region Properties

        /// <summary>
        /// The value of the IHI.
        /// </summary>
        [DataMember]
        public string Ihi
        {
            get
            {
                return ihi;
            }
            set
            {
                ihi = value;
            }
        }

        /// <summary>
        /// The IHI status.
        /// </summary>
        [DataMember]
        public IhiStatus IhiStatus
        {
            get
            {
                return ihiStatus;
            }
            set
            {
                ihiStatus = value;
            }
        }

        /// <summary>
        /// The IHI record status
        /// Note: In this release HIPS only supports Verified IHIs.
        /// It is an error to provide an IHI with any record status apart from Verified.
        /// </summary>
        [DataMember]
        public IhiRecordStatus IhiRecordStatus
        {
            get
            {
                return ihiRecordStatus;
            }
            set
            {
                ihiRecordStatus = value;
            }
        }

        /// <summary>
        /// The date and time when the IHI was last validated with the HI Service.
        /// If this value is within the configured period then HIPS will not attempt
        /// to revalidate the IHI before use in a PCEHR service call.
        /// </summary>
        [DataMember]
        public DateTime IhiLastValidated
        {
            get
            {
                return ihiLastValidated;
            }
            set
            {
                ihiLastValidated = value;
            }
        }

        /// <summary>
        /// The family name that was used to obtain or validate the IHI with the HI Service.
        /// </summary>
        [DataMember]
        public string FamilyName
        {
            get
            {
                return familyName;
            }
            set
            {
                familyName = value;
            }
        }

        /// <summary>
        /// The given name that was used to obtain or validate the IHI with the HI Service.
        /// </summary>
        [DataMember]
        public string GivenName
        {
            get
            {
                return givenName;
            }
            set
            {
                givenName = value;
            }
        }

        /// <summary>
        /// The date of birth that was used to obtain or validate the IHI with the HI Service.
        /// </summary>
        [DataMember]
        public DateTime DateOfBirth
        {
            get
            {
                return dateOfBirth;
            }
            set
            {
                dateOfBirth = value;
            }
        }

        /// <summary>
        /// The sex that was used to obtain or validate the IHI with the HI Service.
        /// </summary>
        [DataMember]
        public SexEnumerator Sex
        {
            get
            {
                return sex;
            }
            set
            {
                sex = value;
            }
        }

        #endregion Properties

        #region Methods

        public override string ToString()
        {
            return string.Format("IHI {0} Last Validated {1:dd/MM/yyyy HH:mm}", ihi, ihiLastValidated);
        }

        /// <summary>
        /// Specific constructor for a validated IHI identifier.
        /// </summary>
        /// <param name="ihi">The IHI</param>
        /// <param name="ihiStatus">The IHI status</param>
        /// <param name="ihiRecordStatus">The IHI record status</param>
        /// <param name="ihiLastValidated">The date/time when the IHI was obtained or last validated</param>
        /// <param name="familyName">The family name that was used to obtain or validate the IHI with the HI Service</param>
        /// <param name="givenName">The given name that was used to obtain or validate the IHI with the HI Service</param>
        /// <param name="dateOfBirth">The date of birth that was used to obtain or validate the IHI with the HI Service</param>
        /// <param name="sex">The sex that was used to obtain or validate the IHI with the HI Service</param>
        /// <param name="hospitalCode">A code identifying the hospital in which this patient receives health care</param>
        /// <param name="hospitalCodeSystem">A code identifying the system under which the hospital code was allocated</param>
        public ValidatedIhi(string ihi, IhiStatus ihiStatus, IhiRecordStatus ihiRecordStatus, DateTime ihiLastValidated, string familyName, string givenName, DateTime dateOfBirth, SexEnumerator sex, string hospitalCode, string hospitalCodeSystem)
            : base(hospitalCode, hospitalCodeSystem)
        {
            this.ihi = ihi;
            this.ihiStatus = ihiStatus;
            this.ihiRecordStatus = ihiRecordStatus;
            this.ihiLastValidated = ihiLastValidated;
            this.familyName = familyName;
            this.givenName = givenName;
            this.dateOfBirth = dateOfBirth;
            this.sex = sex;
        }

        #endregion Methods
    }
}