﻿using System.Diagnostics;

namespace HIPS.Configuration.Tracing
{
    /// <summary>
    /// Represents a message to be traced.
    /// </summary>
    public class TraceMessage
    {
        #region Constructors

        /// <summary>
        /// Creates an instance of a TraceMessage.
        /// </summary>
        /// <remarks>
        /// Defaults an empty message, and a TraceEventType of Information.
        /// </remarks>
        public TraceMessage()
            : this(string.Empty, TraceEventType.Information)
        {
        }

        /// <summary>
        /// Creates an instance of a TraceMessage with the specified message and type.
        /// </summary>
        /// <param name="message"></param>
        /// <param name="type"></param>
        public TraceMessage(string message, TraceEventType type)
        {
            Message = message;
            Type = type;
            Index = -1;
            Level = -1;
            IsStartMessage = false;
            IsEndMessage = false;
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// The message to be traced.
        /// </summary>
        public string Message { get; set; }

        /// <summary>
        /// The type to trace.
        /// </summary>
        /// <remarks>
        /// Defaults to TraceEventType.Information
        /// </remarks>
        public TraceEventType Type { get; set; }

        /// <summary>
        /// Gets or sets the level of nesting that this message is in, in relation to the overall scope of the trace writer in which it is created.
        /// </summary>
        public int Level { get; set; }

        /// <summary>
        /// Gets or sets the sequential index of this message within the current level.
        /// </summary>
        public int Index { get; set; }

        /// <summary>
        /// Gets or sets a value that indicates if this message is a 'start trace' message.
        /// </summary>
        public bool IsStartMessage { get; set; }

        /// <summary>
        /// Gets or sets a value that indicates if this message is an 'end trace' message.
        /// </summary>
        public bool IsEndMessage { get; set; }

        #endregion Properties

        #region Overrides

        /// <summary>
        /// Returns a string representation of the class instance.
        /// </summary>
        /// <returns></returns>
        public override string ToString()
        {
            return Message;
        }

        #endregion Overrides
    }
}