﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.ConsentSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.ConsentBusinessLogic
{
    public class CheckPatientParticipation
    {
        /// <summary>
        /// Gets the participation status for a single hospital patient record. This involves
        /// checking both the HI Service and the PCEHR System, as well as local consent records.
        /// <list type="ordered">
        /// <item><description>
        /// If we do not have a valid IHI then the patient is not considered participating.
        /// We cannot upload a discharge summary for a patient without a valid IHI.
        /// </description></item>
        /// <item><description>
        /// If the patient has disclosed the existence of a PCEHR to a participating health provider
        /// organisation, they are taken to have consented to the upload of any discharge
        /// summary from any hospitals in that organisation.
        /// </description></item>
        /// <item><description>
        /// If the patient has an advertised PCEHR, then we assume they do want their discharge
        /// summary to be uploaded to PCEHR.</description></item>
        /// <item><description>
        /// With no advertised PCEHR and no explicit consents in the hospital, we assume the
        /// patient will not expect that hospital to upload the discharge summary.
        /// </description></item>
        /// </list>
        /// </summary>
        /// <param name="mrn">The local patient identifier at the hospital (HospitalPatient.Mrn)</param>
        /// <param name="hospitalCode">Code identifying the hospital (HospitalCode.Code)</param>
        /// <param name="hospitalCodeSystem">Code identifying the type of hospital code (CodeSystem.Code)</param>
        /// <param name="user">The user details for IHI validation and PCEHR advertised checking.</param>
        /// <returns>Response containing error indicator and the participation status for the specified patient</returns>
        public PatientParticipationResponse GetPatientParticipationStatus(PatientIdentifierBase patientIdentifier, UserDetails user)
        {
            PatientAccess patientAccess = new PatientAccess(user);
            PatientParticipationResponse list = new PatientParticipationResponse();
            Hospital hospital;
            HospitalPatient hospitalPatient;
            PatientMaster patientMaster;
            list.Response = patientAccess.GetHospital(patientIdentifier, out hospital);
            if (list.Response.Status != HipsResponseIndicator.OK)
            {
                return list;
            }

            // When checking participation status, it's OK if the patient has an invalid IHI or an unresolved IHI alert.
            list.Response = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
            if (list.Response.Status != HipsResponseIndicator.OK && list.Response.Status != HipsResponseIndicator.InvalidIhi && list.Response.Status != HipsResponseIndicator.UnresolvedIhiAlert)
            {
                return list;
            }
            list.PatientParticipationList = new PatientParticipationStatus[]
            {
                GetPatientParticipationStatus(hospital, hospitalPatient, patientIdentifier.HospitalCode, patientIdentifier.HospitalCodeSystem, user)
            };
            return list;
        }

        /// <summary>
        /// This service will obtain the participation status for all patients with changes to their participation status since the given date/time.
        /// It will only return the participation status for patients in hospitals that have a code in the given hospital code system
        /// (e.g. those which use a particular PAS or CIS).
        /// </summary>
        /// <param name="since">The date/time after which the episode must have been modified to be included</param>
        /// <param name="hospitalCodeSystem">Code that filters which hospitals are included</param>
        /// <param name="user">The user details for IHI validation and PCEHR advertised checking.</param>
        /// <returns>List of participation status for patients</returns>
        public PatientParticipationResponse GetRecentPatientParticipationStatus(DateTime since, string hospitalCodeSystem, UserDetails user)
        {
            PatientAccess dataAccessHelper = new PatientAccess(user);
            PatientParticipationResponse response = new PatientParticipationResponse();
            List<PatientParticipationStatus> statuses = new List<PatientParticipationStatus>();
            response.Response = new HipsResponse(HipsResponseIndicator.OK);

            List<HealthProviderOrganisationPatient> hpops = dataAccessHelper.HealthProviderOrganisationPatientDataAccess.GetAll(since);
            HashSet<int> patientMasterIds = new HashSet<int>();
            foreach (HealthProviderOrganisationPatient hpop in hpops)
            {
                List<HospitalPatient> hospitalPatients = dataAccessHelper.HospitalPatientDataAccess.GetAllActive(hpop.PatientMasterId.Value);
                foreach (HospitalPatient hospitalPatient in hospitalPatients)
                {
                    Hospital hospital = dataAccessHelper.Hospitals.Where(a => a.HospitalId == hospitalPatient.HospitalId).FirstOrDefault();
                    string hospitalCode = hospital.GetCode(hospitalCodeSystem);
                    if (hospitalCode != null)
                    {
                        // Include this hospital because it is relevant to the querying system
                        PatientParticipationStatus patientStatus = GetPatientParticipationStatus(hospital, hospitalPatient, hospitalCode, hospitalCodeSystem, user);
                        statuses.Add(patientStatus);
                    }
                }
            }
            string infoMessage = LogMessages.InfoPatientParticipation;
            string exceptionMessage = string.Format(ExceptionMessages.InfoPatientParticipation, statuses.Count, since);
            EventLogger.WriteLog(infoMessage, new Exception(exceptionMessage), user, LogMessage.HIPS_MESSAGE_113);
            response.PatientParticipationList = statuses.ToArray();
            return response;
        }

        /// <summary>
        /// Gets the participation status of a single hospital patient record.
        /// If we do not have a valid IHI then the patient cannot be participating.
        /// If the patient has explicitly consented to the upload of a discharge summary then they are considered participating.
        /// If the patient has an advertised PCEHR, then we assume they do want to participate.
        /// With no advertised PCEHR and no explicit consents, we assume they are not participating.
        /// </summary>
        /// <param name="hospitalPatient">The hospital patient record</param>
        /// <param name="hospitalCode">Code identifying the hospital (HospitalCode.Code)</param>
        /// <param name="hospitalCodeSystem">Code identifying the type of hospital code (CodeSystem.Code)</param>
        /// <param name="user">The user details of the authorised employee for IHI and PCEHR checks</param>
        /// <returns>The PCEHR participation status for the patient</returns>
        private PatientParticipationStatus GetPatientParticipationStatus(Hospital hospital, HospitalPatient hospitalPatient, string hospitalCode, string hospitalCodeSystem, UserDetails user)
        {
            Debug.Assert(hospital.HospitalId == hospitalPatient.HospitalId);
            PatientAccess dataAccessHelper = new PatientAccess(user);
            PatientMaster patientMaster;
            if (HipsResponseIndicator.OK != dataAccessHelper.PatientMasterDataAccess.Get(hospitalPatient.PatientMasterId, out patientMaster).Status)
            {
                return null;
            }

            // Create a patient participation status record
            PatientParticipationStatus patientStatus = new PatientParticipationStatus();
            patientStatus.Mrn = hospitalPatient.Mrn;
            patientStatus.StatePatientId = patientMaster.StatePatientId;
            patientStatus.ValidatedIhi = GetValidatedIhi(patientMaster, hospitalCode, hospitalCodeSystem);
            patientStatus.HospitalCode = hospitalCode;
            HealthProviderOrganisationPatient hpioPatient;
            dataAccessHelper.HealthProviderOrganisationPatientDataAccess.Get(hospital.HpiO, hospitalPatient.PatientMasterId, out hpioPatient);

            HipsResponse localIhiStatus = new HipsResponse(HipsResponseIndicator.OK);
            dataAccessHelper.ValidateLocalIhiInformation(patientMaster, localIhiStatus);

            if (localIhiStatus.Status != HipsResponseIndicator.OK)
            {
                patientStatus.ParticipationStatus = ParticipationStatus.NoValidIhi;
            }
            else if (hpioPatient.PcehrDisclosed)
            {
                patientStatus.ParticipationStatus = ParticipationStatus.RequestedUpload;
            }
            else
            {
                if (hpioPatient.PcehrAdvertised.HasValue && hpioPatient.PcehrAdvertised.Value)
                {
                    patientStatus.ParticipationStatus = ParticipationStatus.PcehrAdvertised;
                }
                else
                {
                    patientStatus.ParticipationStatus = ParticipationStatus.PcehrNotAdvertised;
                }
            }
            return patientStatus;
        }

        /// <summary>
        /// If the patient master has a validated IHI, returns an object that encapsulates the validated IHI information, otherwise null.
        /// </summary>
        /// <param name="patientMaster">The patient master</param>
        /// <returns>an object that encapsulates the validated IHI information</returns>
        private ValidatedIhi GetValidatedIhi(PatientMaster patientMaster, string hospitalCode, string hospitalCodeSystem)
        {
            if (patientMaster.IhiLastValidated.HasValue && !string.IsNullOrEmpty(patientMaster.Ihi))
            {
                ValidatedIhi ihi = new ValidatedIhi(
                    patientMaster.Ihi,
                    (IhiStatus)patientMaster.IhiStatusId,
                    (IhiRecordStatus)patientMaster.IhiRecordStatusId,
                    patientMaster.IhiLastValidated.Value,
                    patientMaster.RegisteredFamilyName,
                    patientMaster.RegisteredGivenName,
                    patientMaster.DateOfBirth,
                    (SexEnumerator)patientMaster.RegisteredSexId,
                    hospitalCode,
                    hospitalCodeSystem);
                return ihi;
            }
            else
            {
                // IHI has not been validated
                return null;
            }
        }
    }
}