﻿using System;
using HIPS.CommonBusinessLogic;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.ConsentSchemas;
using HIPS.PcehrDataStore.Schemas;

namespace HIPS.ConsentBusinessLogic
{
    public class CheckUploadConsent
    {
        /// <summary>
        /// Finds out whether HIPS has recorded a withdrawal of consent to upload
        /// clinical documents for a specified episode.
        /// </summary>
        /// <param name="patientIdentifier">The patient identifier issued by a hospital (HospitalPatient.Mrn), EMPI (PatientMaster.StatePatientId), HIPS (PatientMaster.PatientMasterId) or HI Service (PatientMasterIhi.Ihi)</param>
        /// <param name="hospitalCode">Code identifying the hospital that issued the MRN and in which the episode occurred (HospitalCode.Code)</param>
        /// <param name="hospitalCodeSystem">Code identifying the type of hospital code given (CodeSystem.Code)</param>
        /// <param name="admissionDate">The date and time when the episode started (Episode.AdmissionDate)</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <returns>Response containing the consent status for this episode and an error indicator</returns>
        public EpisodeConsentResponse CheckConsent(PatientIdentifierBase patientIdentifier, DateTime admissionDate, UserDetails user)
        {
            PatientAccess patientAccess = new PatientAccess(user);
            EpisodeConsentResponse response = new EpisodeConsentResponse();
            response.Response = new HipsResponse(HipsResponseIndicator.OK);
            response.ConsentStatus = EpisodeConsentStatus.Unknown;

            Hospital hospital;
            HospitalPatient hospitalPatient;
            PatientMaster patientMaster;
            HipsResponse status = patientAccess.GetHospital(patientIdentifier, out hospital);
            if (status.Status != HipsResponseIndicator.OK)
            {
                response.Response = status;
                return response;
            }

            // When checking consent, it's OK if the patient has an invalid IHI or an unresolved IHI alert.
            status = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
            if (status.Status != HipsResponseIndicator.OK && status.Status != HipsResponseIndicator.InvalidIhi && status.Status != HipsResponseIndicator.UnresolvedIhiAlert)
            {
                response.Response = status;
                return response;
            }

            //A Specific Episode that has not been cancelled must be found in this case
            Episode episode = patientAccess.GetEpisodeWithoutCancelled(patientIdentifier, admissionDate, hospitalPatient);
            if (episode == null)
            {
                response.Response = new HipsResponse(HipsResponseIndicator.InvalidEpisode);
            }
            else if (episode.ConsentWithdrawn)
            {
                response.ConsentStatus = EpisodeConsentStatus.WithdrewConsent;
            }
            else
            {
                response.ConsentStatus = EpisodeConsentStatus.NoConsentRecorded;
            }
            return response;
        }
    }
}