﻿using System;
using HIPS.Base.Schemas;
using HIPS.Base.Schemas.Enumerators;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Schemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.Helpers;

namespace Test.PcehrDataStore.DataAccess
{
    /// <summary>
    ///This is a test class for PatientMasterDlTest and is intended
    ///to contain all PatientMasterDlTest Unit Tests
    ///</summary>
    [TestClass()]
    public class PatientMasterDlTests
    {
        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region Additional test attributes

        //
        //You can use the following additional attributes as you write your tests:
        //
        //Use ClassInitialize to run code before running the first test in the class
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //Use ClassCleanup to run code after all tests in a class have run
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //Use TestInitialize to run code before running each test
        //[TestInitialize()]
        //public void MyTestInitialize()
        //{
        //}
        //
        //Use TestCleanup to run code after each test has run
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //

        #endregion Additional test attributes

        #region Test Methods

        /// <summary>
        ///A test for Insert
        ///</summary>
        [TestMethod()]
        public void InsertTest()
        {
            try
            {
                PatientTestHelpersBase testHelper = new PatientTestHelpersBase();
                SystemErrorLog log = testHelper.GetLastErrorMessage();
                int patientMasterId = testHelper.SetTestData();
                testHelper.AnySystemErrorsOccur(log);
                PatientMasterDl patientMasterDl = new PatientMasterDl(testHelper.GetTestUser());
                PatientMaster insertedPatient;
                HipsResponse read = patientMasterDl.Get(patientMasterId, out insertedPatient);
                if (read.Status != HipsResponseIndicator.OK)
                {
                    Assert.Fail("The test patient was not saved");
                }
                else
                {
                    if (insertedPatient.Names.Count != 3)
                    {
                        Assert.Fail("The test patient names were not saved");
                    }
                    if (insertedPatient.CurrentName.GivenNames != "FirstName3")
                    {
                        Assert.Fail("The current name logic has failed");
                    }
                    if (insertedPatient.Addresses.Count != 5)
                    {
                        Assert.Fail("The test patient names were not saved");
                    }
                    if (insertedPatient.Contacts.Count != 5)
                    {
                        Assert.Fail("The test patient contacts were not saved");
                    }
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(string.Format("Encountered an error {0}", ex.Message));
            }
        }

        [TestMethod]
        public void CurrentNameTest()
        {
            try
            {
                PatientTestHelpersBase testHelper = new PatientTestHelpersBase();
                PatientMasterDl patientMasterDl = new PatientMasterDl(testHelper.GetTestUser());
                int patientMasterId = testHelper.SetTestData();
                PatientMaster testPatient;
                HipsResponse read = patientMasterDl.Get(patientMasterId, out testPatient);
                if (read.Status != HipsResponseIndicator.OK)
                {
                    Assert.Fail("The test patient was not saved");
                }

                int count = testPatient.Names.FindAll(result => result.NameTypeId == (int)NameTypes.Current).Count;
                if (count != 1)
                {
                    Assert.Fail("Current name is missing from the name collection");
                }

                testPatient.SetNewCurrentName(null, "New Current", "Test", null);
                if (testPatient.Names.Count != 4)
                {
                    Assert.Fail("Current name has not been added");
                }

                testPatient.SetNewCurrentName(null, "New Current", "Test", null);
                if (testPatient.Names.Count != 4)
                {
                    Assert.Fail("Current name has duplicated");
                }
                if (testPatient.Names.FindAll(result => result.NameTypeId == (int)NameTypes.Current).Count != 1)
                {
                    Assert.Fail("Setting current name to existing current name causes problems");
                }

                testPatient.SetNewCurrentName(null, "New Current 2", "Test", null);
                count = testPatient.Names.FindAll(result => result.NameTypeId == (int)NameTypes.Current).Count;
                if (count != 1)
                {
                    Assert.Fail("Current name has not been added correctly");
                }
                if (testPatient.CurrentName.GivenNames != "New Current 2")
                {
                    Assert.Fail("Current name has not been set to New Current 2");
                }

                testPatient.SetNewCurrentName(null, "New Current", "Test", null);
                count = testPatient.Names.FindAll(result => result.NameTypeId == (int)NameTypes.Current).Count;
                if (count != 1)
                {
                    Assert.Fail("Current name has not been updated");
                }
                if (testPatient.CurrentName.GivenNames != "New Current")
                {
                    Assert.Fail("Previous current name has not been set to current name");
                }

                count = testPatient.Names.Count;
                testPatient.SetNewCurrentName(null, "new Current 2", "Test", null);

                if (count != testPatient.Names.Count)
                {
                    Assert.Fail("Setting a new current name when the name already exists does not work");
                }

                string currentFirstNames = testPatient.CurrentName.GivenNames;
                string currentLastName = testPatient.CurrentName.FamilyName;
            }
            catch (Exception ex)
            {
                Assert.Fail(string.Format("Encountered an error {0}", ex.Message));
            }
        }

        [TestMethod()]
        public void UpdatePatientMasterTest()
        {
            PatientTestHelpersBase testHelper = new PatientTestHelpersBase();
            PatientMasterDl patientMasterDl = new PatientMasterDl(testHelper.GetTestUser());
            int patientMasterId = testHelper.SetTestData();
            PatientMaster testPatient;
            HipsResponse read = patientMasterDl.Get(patientMasterId, out testPatient);
            if (read.Status != HipsResponseIndicator.OK)
            {
                Assert.Fail("The test patient was not saved");
            }
            int addressCount = testPatient.Addresses.Count;
            int contactCount = testPatient.Contacts.Count;
            int nameCount = testPatient.Names.Count;

            testPatient.DvaNumber = "new DVA Number";
            SystemErrorLog log = testHelper.GetLastErrorMessage();
            bool updated = patientMasterDl.Update(testPatient, null);
            testHelper.AnySystemErrorsOccur(log);
            if (!updated || testPatient.DvaNumber != "new DVA Number")
            {
                Assert.Fail("Failed to update the Patient Master");
            }
            if (addressCount != testPatient.Addresses.Count)
            {
                Assert.Fail("Failed to update the Patient Master Addresses");
            }
            if (contactCount != testPatient.Contacts.Count)
            {
                Assert.Fail("Failed to update the Patient Master Contacts");
            }
            if (nameCount != testPatient.Names.Count)
            {
                Assert.Fail("Failed to update the Patient Master Names");
            }
        }

        [TestMethod]
        public void UpdatePatientMasterAddressTest()
        {
            PatientTestHelpersBase testHelper = new PatientTestHelpersBase();
            PatientMasterDl patientMasterDl = new PatientMasterDl(testHelper.GetTestUser());
            int patientMasterId = testHelper.SetTestData();
            PatientMaster testPatient;
            HipsResponse read = patientMasterDl.Get(patientMasterId, out testPatient);
            if (read.Status != HipsResponseIndicator.OK)
            {
                Assert.Fail("The test patient was not saved");
            }

            testPatient.Addresses.Add(new Address("New 1-Line1", "1-Line2", "1-PlaceName", 1, "1-ISC", "1-PC", 1, 1));
            int addressCount = testPatient.Addresses.Count;
            bool updated = false;
            UpdatePatientMaster(patientMasterDl, patientMasterId, ref testPatient, ref read, ref updated, 2);
            if (addressCount != testPatient.Addresses.Count)
            {
                Assert.Fail("Failed to add the new address");
            }

            testPatient.Addresses[0].AddressLine1 = "MODIFIED";
            UpdatePatientMaster(patientMasterDl, patientMasterId, ref testPatient, ref read, ref updated, 3);
            if (testPatient.Addresses[0].AddressLine1 != "MODIFIED")
            {
                Assert.Fail("Failed to read the updated the first address");
            }
            if (addressCount != testPatient.Addresses.Count)
            {
                Assert.Fail("Updating an address caused an address missmatch");
            }

            testPatient.Addresses.RemoveAt(0);
            UpdatePatientMaster(patientMasterDl, patientMasterId, ref testPatient, ref read, ref updated, 4);
            if (addressCount - 1 != testPatient.Addresses.Count)
            {
                Assert.Fail("Failed to delete an address");
            }
            if (testPatient.Addresses.FindAll(result => result.AddressLine1 == "MODIFIED").Count > 0)
            {
                Assert.Fail("Failed to delete the correct address");
            }
        }

        [TestMethod]
        public void UpdatePatientMasterContactTest()
        {
            PatientTestHelpersBase testHelper = new PatientTestHelpersBase();
            PatientMasterDl patientMasterDl = new PatientMasterDl(testHelper.GetTestUser());
            int patientMasterId = testHelper.SetTestData();
            PatientMaster testPatient;
            HipsResponse read = patientMasterDl.Get(patientMasterId, out testPatient);
            if (read.Status != HipsResponseIndicator.OK)
            {
                Assert.Fail("The test patient was not saved");
            }

            testPatient.Contacts.Add(new Contact("Detail1", (int)ContactMethods.HomePhone, "Test Home Phone"));
            int contactCount = testPatient.Contacts.Count;
            bool updated = false;
            UpdatePatientMaster(patientMasterDl, patientMasterId, ref testPatient, ref read, ref updated, 2);
            if (contactCount != testPatient.Contacts.Count)
            {
                Assert.Fail("Failed to add the new contact");
            }

            testPatient.Contacts[0].Detail = "MODIFIED";
            UpdatePatientMaster(patientMasterDl, patientMasterId, ref testPatient, ref read, ref updated, 3);
            if (testPatient.Contacts[0].Detail != "MODIFIED")
            {
                Assert.Fail("Failed to read the updated the first contact");
            }
            if (contactCount != testPatient.Contacts.Count)
            {
                Assert.Fail("Updating an address caused an contact missmatch");
            }

            testPatient.Contacts.RemoveAt(0);
            UpdatePatientMaster(patientMasterDl, patientMasterId, ref testPatient, ref read, ref updated, 4);
            if (contactCount - 1 != testPatient.Contacts.Count)
            {
                Assert.Fail("Failed to delete a contact");
            }
            if (testPatient.Addresses.FindAll(result => result.AddressLine1 == "MODIFIED").Count > 0)
            {
                Assert.Fail("Failed to delete the correct contact");
            }
        }

        #endregion Test Methods

        #region Helper Methods

        /// <summary>
        /// Updates the patient master.
        /// </summary>
        /// <param name="patientMasterDl">The patient master dl.</param>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <param name="testPatient">The test patient.</param>
        /// <param name="read">if set to <c>true</c> [read].</param>
        /// <param name="updated">if set to <c>true</c> [updated].</param>
        /// <param name="count">The count.</param>
        private static void UpdatePatientMaster(PatientMasterDl patientMasterDl, int patientMasterId, ref PatientMaster testPatient, ref HipsResponse read, ref bool updated, int count)
        {
            updated = patientMasterDl.Update(testPatient, null);
            if (!updated)
            {
                Assert.Fail(string.Format("Failed to update the Patient Master - {0}", count));
            }
            read = patientMasterDl.Get(patientMasterId, out testPatient);
            if (read.Status != HipsResponseIndicator.OK)
            {
                Assert.Fail(string.Format("Failed to read the updated Patient Master - {0}", count));
            }
        }

        #endregion Helper Methods
    }
}