﻿using System;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.HibIntegration.HL7;
using HIPS.HibIntegration.HL7.DataStructure;
using HIPS.HibIntegration.HL7.Message;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.HibIntegration.Loader
{
    public class MessageLoader
    {
        private const string Facility = "SAH";
        private const string Application = "HIPS";
        private const string WrongMessageTypeError = "Unhandled HL7MessageLog Type: {0}";

        /// <summary>
        /// Processes the specified received message.
        /// </summary>
        /// <param name="receivedMessage">The received message.</param>
        /// <param name="fullMessage">The full message.</param>
        /// <param name="user">The user.</param>
        /// <param name="testMessageLog">The test message log - Only used to get the results back to the Unit Tests.</param>
        /// <returns>An HL7 acknowledgement message</returns>
        public HL7Message Process(HL7Message receivedMessage, string fullMessage, UserDetails user, out HL7MessageLog testMessageLog)
        {
            HL7Message ack;
            HL7MessageLog messageLog = PopulateHL7MessageLog(receivedMessage, fullMessage, user);
            testMessageLog = messageLog;
            try
            {
                if (receivedMessage is HL7GenericPasMessage)
                {
                    new PasLoader(user).Process(receivedMessage as HL7GenericPasMessage, messageLog);
                }
                else
                {
                    string errorMessage = string.Format(WrongMessageTypeError, receivedMessage.TypeName);
                    throw new HL7MessageErrorException(errorMessage);
                }
                ack = HL7AcknowledgementFactory.Acknowledge(receivedMessage, Facility, Application, null);
            }
            catch (IhiInfoException ex)
            {
                // The HL7 message was processed, but a known data quality issue was found while attempting to search for the IHI.

                // Write INFO message because Application Support do not need to know.
                string description = string.Format(ConstantsResource.InfoMessageIhiLookupFail, messageLog.MessageControlId, messageLog.SendingApplication, messageLog.SendingFacility);
                EventLogger.WriteLog(description, ex, user, LogMessage.HIPS_MESSAGE_107);

                // Tell the message broker that the message was successful, no need to retry.
                ack = HL7AcknowledgementFactory.Acknowledge(receivedMessage, Facility, Application, null);
            }
            catch (IhiErrorException ex)
            {
                // The HL7 message was processed but a serious failure occurred when attempting to search for the IHI.

                // Write ERROR message because Application Support need to know.
                string description = string.Format(ConstantsResource.ErrorMessageIhiLookupFail, messageLog.MessageControlId, messageLog.SendingApplication, messageLog.SendingFacility);
                EventLogger.WriteLog(description, ex, user, LogMessage.HIPS_MESSAGE_108);

                // Tell the message broker that the message was successful, no need to retry.
                ack = HL7AcknowledgementFactory.Acknowledge(receivedMessage, Facility, Application, null);
            }
            catch (HL7MessageInfoException ex)
            {
                // This message was not processed because it contains a known data quality issue.

                // Write INFO message because Application Support do not need to know.
                string description = string.Format(ConstantsResource.InfoMessageNotProcessed, messageLog.MessageControlId, messageLog.SendingApplication, messageLog.SendingFacility);
                EventLogger.WriteLog(description, ex, user, LogMessage.HIPS_MESSAGE_109);

                // Tell the message broker that the message was successful, no need to retry.
                ack = HL7AcknowledgementFactory.Acknowledge(receivedMessage, Facility, Application, null);
            }
            catch (HL7MessageErrorException ex)
            {
                // HIPS failed while processing this HL7 message. The message should be retried.

                // Write ERROR message because Application Support need to know.
                string description = string.Format(ConstantsResource.ErrorMessageHipsFailed, messageLog.MessageControlId, messageLog.SendingApplication, messageLog.SendingFacility);
                EventLogger.WriteLog(description, ex, user, LogMessage.HIPS_MESSAGE_110);

                // Tell the message broker to retry this message.
                ack = HL7AcknowledgementFactory.Acknowledge(receivedMessage, Facility, Application, new CE() { text = ex.Message });
            }
            catch (Exception nex)
            {
                // An unexpected exception occurred while HIPS was processing this HL7 message. The message should be retried.

                // Write ERROR message because Application Support need to know.
                string description = string.Format(ConstantsResource.ErrorMessageHipsFailed, messageLog.MessageControlId, messageLog.SendingApplication, messageLog.SendingFacility);
                EventLogger.WriteLog(description, nex, user, LogMessage.HIPS_MESSAGE_111);

                // Tell the message broker to retry this message.
                ack = HL7AcknowledgementFactory.Acknowledge(receivedMessage, Facility, Application, new CE() { text = nex.Message });
            }

            // Return the acknowledgement
            return ack;
        }

        /// <summary>
        /// Queries the HL7 Message Log for a record of this message being
        /// processed before. If not found, a new record will be populated.
        /// Does not insert or update the record. The insert or update must
        /// be done in a transaction created by the PasLoader or
        /// PathologyLoader.
        /// </summary>
        /// <param name="message">The parsed message</param>
        /// <param name="fullMessage">The raw encoded message</param>
        /// <returns>The existing or new HL7 Message log entry</returns>
        private static HL7MessageLog PopulateHL7MessageLog(HL7Message message, string fullMessage, UserDetails user)
        {
            string messageControlID = message.MessageHeader.MessageControlID;
            string sendingApplication = message.MessageHeader.SendingApplication.namespaceID;
            string sendingFacility = message.MessageHeader.SendingFacility.namespaceID;
            DateTime? messageTime;
            try
            {
                messageTime = message.MessageHeader.DateTimeOfMessage.TimestampValue;
            }
            catch (HL7MessageErrorException)
            {
                messageTime = null;
            }

            HL7MessageLogDl dataAccess = new HL7MessageLogDl(user);
            HL7MessageLog messageLogEntry;
            dataAccess.Get(sendingApplication, sendingFacility, messageControlID, messageTime, out messageLogEntry);
            messageLogEntry.FullMessage = fullMessage;
            messageLogEntry.SendingApplication = sendingApplication;
            messageLogEntry.SendingFacility = sendingFacility;
            messageLogEntry.MessageControlId = messageControlID;
            messageLogEntry.ShouldRetry = false;  // This can be set by SSIS to trigger re-processing of a message once the patient is registered
            messageLogEntry.DateTimeOfMessage = messageTime;
            messageLogEntry.QueueStatusId = (int)QueueStatus.Pending;
            messageLogEntry.FailureReason = null;
            return messageLogEntry;
        }
    }
}